﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/pinpoint-sms-voice-v2/PinpointSMSVoiceV2Request.h>
#include <aws/pinpoint-sms-voice-v2/PinpointSMSVoiceV2_EXPORTS.h>
#include <aws/pinpoint-sms-voice-v2/model/CloudWatchLogsDestination.h>
#include <aws/pinpoint-sms-voice-v2/model/EventType.h>
#include <aws/pinpoint-sms-voice-v2/model/KinesisFirehoseDestination.h>
#include <aws/pinpoint-sms-voice-v2/model/SnsDestination.h>

#include <utility>

namespace Aws {
namespace PinpointSMSVoiceV2 {
namespace Model {

/**
 */
class CreateEventDestinationRequest : public PinpointSMSVoiceV2Request {
 public:
  AWS_PINPOINTSMSVOICEV2_API CreateEventDestinationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateEventDestination"; }

  AWS_PINPOINTSMSVOICEV2_API Aws::String SerializePayload() const override;

  AWS_PINPOINTSMSVOICEV2_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Either the name of the configuration set or the configuration set ARN to
   * apply event logging to. The ConfigurateSetName and ConfigurationSetArn can be
   * found using the <a>DescribeConfigurationSets</a> action.</p>
   */
  inline const Aws::String& GetConfigurationSetName() const { return m_configurationSetName; }
  inline bool ConfigurationSetNameHasBeenSet() const { return m_configurationSetNameHasBeenSet; }
  template <typename ConfigurationSetNameT = Aws::String>
  void SetConfigurationSetName(ConfigurationSetNameT&& value) {
    m_configurationSetNameHasBeenSet = true;
    m_configurationSetName = std::forward<ConfigurationSetNameT>(value);
  }
  template <typename ConfigurationSetNameT = Aws::String>
  CreateEventDestinationRequest& WithConfigurationSetName(ConfigurationSetNameT&& value) {
    SetConfigurationSetName(std::forward<ConfigurationSetNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name that identifies the event destination.</p>
   */
  inline const Aws::String& GetEventDestinationName() const { return m_eventDestinationName; }
  inline bool EventDestinationNameHasBeenSet() const { return m_eventDestinationNameHasBeenSet; }
  template <typename EventDestinationNameT = Aws::String>
  void SetEventDestinationName(EventDestinationNameT&& value) {
    m_eventDestinationNameHasBeenSet = true;
    m_eventDestinationName = std::forward<EventDestinationNameT>(value);
  }
  template <typename EventDestinationNameT = Aws::String>
  CreateEventDestinationRequest& WithEventDestinationName(EventDestinationNameT&& value) {
    SetEventDestinationName(std::forward<EventDestinationNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An array of event types that determine which events to log. If "ALL" is used,
   * then End User MessagingSMS logs every event type.</p>  <p>The
   * <code>TEXT_SENT</code> event type is not supported.</p>
   */
  inline const Aws::Vector<EventType>& GetMatchingEventTypes() const { return m_matchingEventTypes; }
  inline bool MatchingEventTypesHasBeenSet() const { return m_matchingEventTypesHasBeenSet; }
  template <typename MatchingEventTypesT = Aws::Vector<EventType>>
  void SetMatchingEventTypes(MatchingEventTypesT&& value) {
    m_matchingEventTypesHasBeenSet = true;
    m_matchingEventTypes = std::forward<MatchingEventTypesT>(value);
  }
  template <typename MatchingEventTypesT = Aws::Vector<EventType>>
  CreateEventDestinationRequest& WithMatchingEventTypes(MatchingEventTypesT&& value) {
    SetMatchingEventTypes(std::forward<MatchingEventTypesT>(value));
    return *this;
  }
  inline CreateEventDestinationRequest& AddMatchingEventTypes(EventType value) {
    m_matchingEventTypesHasBeenSet = true;
    m_matchingEventTypes.push_back(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An object that contains information about an event destination for logging to
   * Amazon CloudWatch Logs.</p>
   */
  inline const CloudWatchLogsDestination& GetCloudWatchLogsDestination() const { return m_cloudWatchLogsDestination; }
  inline bool CloudWatchLogsDestinationHasBeenSet() const { return m_cloudWatchLogsDestinationHasBeenSet; }
  template <typename CloudWatchLogsDestinationT = CloudWatchLogsDestination>
  void SetCloudWatchLogsDestination(CloudWatchLogsDestinationT&& value) {
    m_cloudWatchLogsDestinationHasBeenSet = true;
    m_cloudWatchLogsDestination = std::forward<CloudWatchLogsDestinationT>(value);
  }
  template <typename CloudWatchLogsDestinationT = CloudWatchLogsDestination>
  CreateEventDestinationRequest& WithCloudWatchLogsDestination(CloudWatchLogsDestinationT&& value) {
    SetCloudWatchLogsDestination(std::forward<CloudWatchLogsDestinationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An object that contains information about an event destination for logging to
   * Amazon Data Firehose.</p>
   */
  inline const KinesisFirehoseDestination& GetKinesisFirehoseDestination() const { return m_kinesisFirehoseDestination; }
  inline bool KinesisFirehoseDestinationHasBeenSet() const { return m_kinesisFirehoseDestinationHasBeenSet; }
  template <typename KinesisFirehoseDestinationT = KinesisFirehoseDestination>
  void SetKinesisFirehoseDestination(KinesisFirehoseDestinationT&& value) {
    m_kinesisFirehoseDestinationHasBeenSet = true;
    m_kinesisFirehoseDestination = std::forward<KinesisFirehoseDestinationT>(value);
  }
  template <typename KinesisFirehoseDestinationT = KinesisFirehoseDestination>
  CreateEventDestinationRequest& WithKinesisFirehoseDestination(KinesisFirehoseDestinationT&& value) {
    SetKinesisFirehoseDestination(std::forward<KinesisFirehoseDestinationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An object that contains information about an event destination for logging to
   * Amazon SNS.</p>
   */
  inline const SnsDestination& GetSnsDestination() const { return m_snsDestination; }
  inline bool SnsDestinationHasBeenSet() const { return m_snsDestinationHasBeenSet; }
  template <typename SnsDestinationT = SnsDestination>
  void SetSnsDestination(SnsDestinationT&& value) {
    m_snsDestinationHasBeenSet = true;
    m_snsDestination = std::forward<SnsDestinationT>(value);
  }
  template <typename SnsDestinationT = SnsDestination>
  CreateEventDestinationRequest& WithSnsDestination(SnsDestinationT&& value) {
    SetSnsDestination(std::forward<SnsDestinationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Unique, case-sensitive identifier that you provide to ensure the idempotency
   * of the request. If you don't specify a client token, a randomly generated token
   * is used for the request to ensure idempotency.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateEventDestinationRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_configurationSetName;
  bool m_configurationSetNameHasBeenSet = false;

  Aws::String m_eventDestinationName;
  bool m_eventDestinationNameHasBeenSet = false;

  Aws::Vector<EventType> m_matchingEventTypes;
  bool m_matchingEventTypesHasBeenSet = false;

  CloudWatchLogsDestination m_cloudWatchLogsDestination;
  bool m_cloudWatchLogsDestinationHasBeenSet = false;

  KinesisFirehoseDestination m_kinesisFirehoseDestination;
  bool m_kinesisFirehoseDestinationHasBeenSet = false;

  SnsDestination m_snsDestination;
  bool m_snsDestinationHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace PinpointSMSVoiceV2
}  // namespace Aws
