﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iotdeviceadvisor/IoTDeviceAdvisorRequest.h>
#include <aws/iotdeviceadvisor/IoTDeviceAdvisor_EXPORTS.h>
#include <aws/iotdeviceadvisor/model/SuiteDefinitionConfiguration.h>

#include <utility>

namespace Aws {
namespace IoTDeviceAdvisor {
namespace Model {

/**
 */
class CreateSuiteDefinitionRequest : public IoTDeviceAdvisorRequest {
 public:
  AWS_IOTDEVICEADVISOR_API CreateSuiteDefinitionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateSuiteDefinition"; }

  AWS_IOTDEVICEADVISOR_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Creates a Device Advisor test suite with suite definition configuration.</p>
   */
  inline const SuiteDefinitionConfiguration& GetSuiteDefinitionConfiguration() const { return m_suiteDefinitionConfiguration; }
  inline bool SuiteDefinitionConfigurationHasBeenSet() const { return m_suiteDefinitionConfigurationHasBeenSet; }
  template <typename SuiteDefinitionConfigurationT = SuiteDefinitionConfiguration>
  void SetSuiteDefinitionConfiguration(SuiteDefinitionConfigurationT&& value) {
    m_suiteDefinitionConfigurationHasBeenSet = true;
    m_suiteDefinitionConfiguration = std::forward<SuiteDefinitionConfigurationT>(value);
  }
  template <typename SuiteDefinitionConfigurationT = SuiteDefinitionConfiguration>
  CreateSuiteDefinitionRequest& WithSuiteDefinitionConfiguration(SuiteDefinitionConfigurationT&& value) {
    SetSuiteDefinitionConfiguration(std::forward<SuiteDefinitionConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags to be attached to the suite definition.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateSuiteDefinitionRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateSuiteDefinitionRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The client token for the test suite definition creation. This token is used
   * for tracking test suite definition creation using retries and obtaining its
   * status. This parameter is optional.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateSuiteDefinitionRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  SuiteDefinitionConfiguration m_suiteDefinitionConfiguration;
  bool m_suiteDefinitionConfigurationHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace IoTDeviceAdvisor
}  // namespace Aws
