"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.uiToBackendReportDefinition = exports.getBackendReportState = exports.getBackendReportSource = void 0;

var _backendModel = require("../../../model/backendModel");

var _constants = require("../constants");

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
const uiToBackendReportDefinition = reportDefinition => {
  const {
    report_params: {
      report_name: reportName,
      description,
      report_source: reportSource,
      core_params: {
        base_url: baseUrl,
        time_duration: timeDuration,
        report_format: reportFormat,
        header,
        footer,
        limit,
        origin
      }
    },
    trigger,
    delivery
  } = reportDefinition;
  let backendReportDefinition = {
    name: reportName,
    isEnabled: getBackendIsEnabled(trigger),
    source: {
      description: description,
      type: getBackendReportSource(reportSource),
      id: getBackendReportSourceId(baseUrl),
      origin: origin
    },
    format: {
      duration: timeDuration,
      fileFormat: getBackendReportFormat(reportFormat),
      ...(limit && {
        limit: limit
      }),
      ...(header && {
        header: header
      }),
      ...(footer && {
        footer: footer
      })
    },
    trigger: getBackendTrigger(trigger),
    ...(getBackendDelivery(delivery) && {
      delivery: getBackendDelivery(delivery)
    })
  };
  return backendReportDefinition;
};

exports.uiToBackendReportDefinition = uiToBackendReportDefinition;

const getBackendIsEnabled = trigger => {
  let enabled = true;

  if (trigger.trigger_params) {
    enabled = trigger.trigger_params.enabled;
  }

  return enabled;
};

const getBackendDelivery = delivery => {
  const {
    configIds: configIds,
    title: title,
    textDescription: textDescription,
    htmlDescription: htmlDescription
  } = delivery;
  let res = {
    configIds: configIds,
    title: title,
    textDescription: textDescription,
    htmlDescription: htmlDescription
  };
  return res;
};

const getBackendTrigger = trigger => {
  const {
    trigger_params: scheduleParams
  } = trigger;
  const {
    schedule
  } = { ...scheduleParams
  };
  let res = {
    triggerType: scheduleParams ? getBackendTriggerType(scheduleParams) : _backendModel.BACKEND_TRIGGER_TYPE.onDemand,
    schedule: schedule
  };
  return res;
};

const getBackendTriggerType = scheduleParams => {
  const {
    schedule_type: scheduleType
  } = scheduleParams;
  let res;

  switch (scheduleType) {
    case _constants.SCHEDULE_TYPE.cron:
      res = _backendModel.BACKEND_TRIGGER_TYPE.cronSchedule;
      break;

    case _constants.SCHEDULE_TYPE.recurring:
      res = _backendModel.BACKEND_TRIGGER_TYPE.intervalSchedule;
      break;
  }

  return res;
};

const getBackendReportFormat = reportFormat => {
  return _backendModel.REPORT_FORMAT_DICT[reportFormat];
};

const getBackendReportState = reportState => {
  return _backendModel.REPORT_STATE_DICT[reportState];
};

exports.getBackendReportState = getBackendReportState;

const getBackendReportSource = reportSource => {
  return _backendModel.REPORT_SOURCE_DICT[reportSource];
}; //TODO: tmp solution, we are extracting the id from the baseUrl, e.g. /app/dashboards#/view/<id>
// since currently dashboard/visualization id are not required in the UI model, will add in the future


exports.getBackendReportSource = getBackendReportSource;

const getBackendReportSourceId = baseUrl => {
  const id = baseUrl.split('/').pop() || '';
  return id;
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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