

################################################################################
##<#GAPDoc Label="SCNrCyclic3Mflds">
## <ManSection>
## <Func Name="SCNrCyclic3Mflds" Arg="i"/>
## <Returns> integer upon success, <K>fail</K> otherwise.</Returns>
## <Description>	
## Returns the number of combinatorial 3-manifolds with transitive cyclic 
## symmetry with <Arg>i</Arg> vertices.
## 
## See <Cite Key="Spreer11CyclicCombMflds"/> for more about the classification 
## of combinatorial 3-manifolds with transitive cyclic symmetry up to 
## <M>22</M> vertices.
## <Example><![CDATA[
## gap> SCNrCyclic3Mflds(22);
## ]]></Example>
## </Description>
## </ManSection>
##<#/GAPDoc>
################################################################################
InstallGlobalFunction(SCNrCyclic3Mflds,
function(i)
	local allowedInts;
	
	allowedInts:=[0, 0, 0, 0, 1, 1, 1, 2, 2, 8, 6, 20, 15, 50, 34, 107, 89, 319, 
    279, 1008, 1038, 3090];
	if i < 5 or i > 22 then
		Info(InfoSimpcomp,1,"SCSeriesCyclic3Mflds: argument must be a positive ",
      "integer 5 <= i <= 22.");
		return fail;
	fi;
	
	return allowedInts[i];
	
end);
	

################################################################################
##<#GAPDoc Label="SCCyclic3MfldTopTypes">
## <ManSection>
## <Func Name="SCCyclic3MfldTopTypes" Arg="i"/>
## <Returns> a list of strings upon success, <K>fail</K> otherwise.</Returns>
## <Description>	
## Returns a list of all topological types that occur in the classification 
## combinatorial 3-manifolds with transitive cyclic symmetry with <Arg>i</Arg> 
## vertices.
## 
## See <Cite Key="Spreer11CyclicCombMflds"/> for more about the classification 
## of combinatorial 3-manifolds with transitive cyclic symmetry up to 
## <M>22</M> vertices.
## <Example><![CDATA[
## gap> SCCyclic3MfldTopTypes(19);
## ]]></Example>
## </Description>
## </ManSection>
##<#/GAPDoc>
################################################################################
InstallGlobalFunction(SCCyclic3MfldTopTypes,
function(i)
	local tt;

	tt:=[
	[],
	[],
	[],
	[],
	["S^3"],
	["S^3"],
	["S^3"],
	["S^3"],
	["S^2~S^1","S^3"],
	["S^2xS^1","S^2~S^1","S^3"],
	["S^2~S^1","S^3"],
	["(S^2xS^1)#2","S^2xS^1","S^2~S^1","S^3"],
	["S^2~S^1","S^3"],
	["L(3,1)","S^2xS^1","S^2~S^1","S^3"],
	["RP^3","S^2~S^1","S^3","S^3/Q8","T^3"],
	["(S^2xS^1)#5","B2","L(3,1)","S^2xS^1","S^2~S^1","S^3","S^3/P24","T^3"],
	["B2","RP^2xS^1","RP^3","S^2~S^1","S^3","Sigma^3","T^3"],
	["(S^2xS^1)#7","(S^2~S^1)#7","B2","B4","K^2xS^1","L(5,1)","L(8,3)","RP^3",
    "SFS[RP^2:(2,1)(2,1)(2,1)]","SFS[T^2:(1,1)]",
	"S^2xS^1","S^2~S^1","S^3","S^3/Q8","T^3"],
	["B2","RP^2xS^1","SFS[RP^2:(2,1)(3,1)]","S^2~S^1","S^3","Sigma(2,3,7)","T^3"],
	["(S^2xS^1)#4","(S^2xS^1)#6","(S^2~S^1)#6","(S^2~S^1)#9","B2","B4","G2",
    "K^2xS^1","L(3,1)","L(8,3)","RP^2xS^1",
    "SFS[D:(3,1)(3,1)]U/mSFS[D:(3,1)(3,1)],m=[-4,5|-3,4]","SFS[K^2/n2:(1,5)]",
    "SFS[RP^2:(2,1)(3,1)]","SFS[RP^2:(3,1)(3,2)]","SFS[S^2:(3,1)(3,1)(4,-3)]",
    "S^2xS^1","S^2~S^1","S^3","S^3/Q28","T^3"],
	["(RP^2)#3xS^1","(S^2xS^1)#12","B2","G3","K^2xS^1","L(3,1)","RP^2xS^1",
    "RP^3","SFS[K^2:(2,1)(2,1)(2,1)]","SFS[K^2:(2,1)]",
    "SFS[RP^2:(2,1)(2,1)(2,1)]","SFS[RP^2:(3,1)(3,1)(3,1)]",
    "SFS[RP^2:(3,1)(3,1)(3,2)]","SFS[S^2:(2,1)(2,1)(2,1)(3,-5)]",
    "SFS[S^2:(3,2)(3,2)(3,-1)]","SFS[S^2:(4,1)(4,1)(4,-3)]",
    "SFS[S^2:(5,1)(5,1)(5,-4)]","SFS[T^2:(1,7)]","S^2~S^1","S^3","S^3/Q8",
    "T^3"],
	["(S^2xS^1)#12","(S^2~S^1)#12","B2","B4","HS","K^2xS^1","L(15,4)",
    "L(3,1)","L(5,1)","L(7,1)","L(8,3)","RP^2xS^1","RP^3",
    "SFS[(RP^2)#3:(1,1)]","SFS[A:(2,1)(2,1)]/[1,-11|1,-10]",
    "SFS[D:(2,1)(2,1)]U/mSFS[D:(2,1)(3,1)],m=[-5,11|-4,9]",
    "SFS[D:(2,1)(3,1)]U/mSFS[D:(3,1)(3,1)],m=[-8,11|-5,7]",
    "SFS[D_:(3,1)(3,1)]","SFS[K^2:(2,1)(2,1)]","SFS[K^2:(2,1)]",
    "SFS[K^2:(3,1)]","SFS[RP^2:(2,1)(3,1)]","SFS[RP^2:(2,1)(5,1)]",
    "SFS[RP^2:(3,1)(3,2)]","SFS[RP^2:(3,1)(4,3)]","SFS[RP^2:(5,2)(5,3)]",
    "SFS[S^2:(2,1)(2,1)(3,1)(3,-2)]","SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]",
    "SFS[S^2:(3,1)(3,1)(5,-3)]","SFS[S^2:(3,1)(3,1)(9,-7)]",
    "SFS[S^2:(3,2)(4,1)(4,-3)]","SFS[S^2:(4,1)(5,2)(5,-3)]","S^2xS^1",
    "S^2~S^1","S^3","S^3/P24","S^3/Q16","S^3/Q32","Sigma^3","T^3"]
	];

	if i > 22 then
		Info(InfoSimpcomp,1,"SCCyclic3MfldsTopTypes: argument must be a positive ",
      "integer < 23.");
		return fail;
	else
		return tt[i];
	fi;	
end);

SCIntFunc.SCCyclic3MfldEx:=function(i,j,type)
	local all,dc,list,c,k,l,allowedInts,topTypes;

	# type = 0: normal
	# type = 1: first of given type
	# type = 2: list of indices of all of given type
	if type = 0 then
		if not IsPosInt(j) or not IsPosInt(i) then
			Info(InfoSimpcomp,1,"SCCyclic3MfldsEx: arguments must be positive ",
        "integers.");
			return fail;
		fi;
		
		if i < 5 or i > 22 then
			Info(InfoSimpcomp,1,"SCCyclic3MfldsEx: first argument must be a ",
        "positive integer 5 <= i <= 22.");
			return fail;
		fi;
		
		allowedInts:=[0, 0, 0, 0, 1, 1, 1, 2, 2, 8, 6, 20, 15, 50, 34, 107, 89, 
      319, 279, 1008, 1038, 3090];
		if j > allowedInts[i] then
			Info(InfoSimpcomp,1,"SCCyclic3MfldsEx: if first argument i = ",i,
        ", then second argument must be a positive integer j <= ",
        allowedInts[i],".");
			return fail;
		fi;
	fi;

	
	all:=[];
	all[1]:=[];
	all[2]:=[];
	all[3]:=[];
	all[4]:=[];
	all[5]:=[[1,1,1,2]];
	all[6]:=[[1,1,1,3],[1,2,1,2]];
	all[7]:=[[1,1,1,4],[1,1,2,3],[1,1,3,2],[1,2,1,3],
	[1,2,2,2]];
	all[8]:=[[1,1,1,5],[1,1,2,4],[1,1,3,3],[1,1,4,2],
	[1,2,1,4],[1,2,2,3],[1,2,3,2],[1,3,1,3],
	[1,3,2,2],[2,2,2,2]];
	all[9]:=[[1,1,1,6],[1,1,2,5],[1,1,3,4],[1,1,4,3],
	[1,1,5,2],[1,2,1,5],[1,2,2,4],[1,2,4,2],
	[1,3,1,4],[1,3,2,3],[1,4,2,2],[2,2,2,3]];
	all[10]:=[[1,1,1,7],[1,1,2,6],[1,1,3,5],[1,1,4,4],
	[1,1,5,3],[1,1,6,2],[1,2,1,6],[1,2,2,5],
	[1,2,3,4],[1,2,4,3],[1,2,5,2],[1,3,1,5],
	[1,3,2,4],[1,3,3,3],[1,3,4,2],[1,4,1,4],
	[1,4,2,3],[1,4,3,2],[1,5,2,2],[2,2,2,4],
	[2,2,3,3],[2,3,2,3]];
	all[11]:=[[1,1,1,8],[1,1,2,7],[1,1,3,6],[1,1,4,5],
	[1,1,5,4],[1,1,6,3],[1,1,7,2],[1,2,1,7],
	[1,2,2,6],[1,2,3,5],[1,2,4,4],[1,2,5,3],
	[1,2,6,2],[1,3,1,6],[1,3,2,5],[1,3,3,4],
	[1,3,4,3],[1,3,5,2],[1,4,1,5],[1,4,2,4],
	[1,4,3,3],[1,4,4,2],[1,5,2,3],[1,5,3,2],
	[1,6,2,2],[2,2,2,5],[2,2,3,4],[2,2,4,3],
	[2,3,2,4],[2,3,3,3]];
	all[12]:=[[1,1,1,9],[1,1,2,8],[1,1,3,7],[1,1,4,6],
	[1,1,5,5],[1,1,6,4],[1,1,7,3],[1,1,8,2],
	[1,2,1,8],[1,2,2,7],[1,2,3,6],[1,2,4,5],
	[1,2,5,4],[1,2,6,3],[1,2,7,2],[1,3,1,7],
	[1,3,2,6],[1,3,3,5],[1,3,5,3],[1,3,6,2],
	[1,4,1,6],[1,4,2,5],[1,4,3,4],[1,4,5,2],
	[1,5,1,5],[1,5,2,4],[1,5,3,3],[1,5,4,2],
	[1,6,2,3],[1,6,3,2],[1,7,2,2],[2,2,2,6],
	[2,2,3,5],[2,2,5,3],[2,3,2,5],[2,3,3,4],
	[2,3,4,3],[2,4,2,4],[2,4,3,3],[3,3,3,3]];
	all[13]:=[[1,1,1,10],[1,1,2,9],[1,1,3,8],[1,1,4,7],
	[1,1,5,6],[1,1,6,5],[1,1,7,4],[1,1,8,3],
	[1,1,9,2],[1,2,1,9],[1,2,2,8],[1,2,3,7],
	[1,2,4,6],[1,2,5,5],[1,2,6,4],[1,2,7,3],
	[1,2,8,2],[1,3,1,8],[1,3,2,7],[1,3,3,6],
	[1,3,4,5],[1,3,5,4],[1,3,6,3],[1,3,7,2],
	[1,4,1,7],[1,4,2,6],[1,4,3,5],[1,4,4,4],
	[1,4,5,3],[1,4,6,2],[1,5,1,6],[1,5,2,5],
	[1,5,3,4],[1,5,4,3],[1,5,5,2],[1,6,2,4],
	[1,6,3,3],[1,6,4,2],[1,7,2,3],[1,7,3,2],
	[1,8,2,2],[2,2,2,7],[2,2,3,6],[2,2,4,5],
	[2,2,5,4],[2,2,6,3],[2,3,2,6],[2,3,3,5],
	[2,3,4,4],[2,3,5,3],[2,4,2,5],[2,4,3,4],
	[2,4,4,3],[2,5,3,3],[3,3,3,4]];
	all[14]:=[[1,1,1,11],[1,1,2,10],[1,1,3,9],[1,1,4,8],
	[1,1,5,7],[1,1,6,6],[1,1,7,5],[1,1,8,4],
	[1,1,9,3],[1,1,10,2],[1,2,1,10],[1,2,2,9],
	[1,2,3,8],[1,2,4,7],[1,2,5,6],[1,2,6,5],
	[1,2,7,4],[1,2,8,3],[1,2,9,2],[1,3,1,9],
	[1,3,2,8],[1,3,3,7],[1,3,4,6],[1,3,5,5],
	[1,3,6,4],[1,3,7,3],[1,3,8,2],[1,4,1,8],
	[1,4,2,7],[1,4,3,6],[1,4,4,5],[1,4,5,4],
	[1,4,6,3],[1,4,7,2],[1,5,1,7],[1,5,2,6],
	[1,5,3,5],[1,5,4,4],[1,5,5,3],[1,5,6,2],
	[1,6,1,6],[1,6,2,5],[1,6,3,4],[1,6,4,3],
	[1,6,5,2],[1,7,2,4],[1,7,3,3],[1,7,4,2],
	[1,8,2,3],[1,8,3,2],[1,9,2,2],[2,2,2,8],
	[2,2,3,7],[2,2,4,6],[2,2,5,5],[2,2,6,4],
	[2,2,7,3],[2,3,2,7],[2,3,3,6],[2,3,4,5],
	[2,3,5,4],[2,3,6,3],[2,4,2,6],[2,4,3,5],
	[2,4,4,4],[2,4,5,3],[2,5,2,5],[2,5,3,4],
	[2,5,4,3],[2,6,3,3],[3,3,3,5],[3,3,4,4],
	[3,4,3,4]];
	all[15]:=[[1,1,1,12],[1,1,2,11],[1,1,3,10],[1,1,4,9],
	[1,1,5,8],[1,1,6,7],[1,1,7,6],[1,1,8,5],
	[1,1,9,4],[1,1,10,3],[1,1,11,2],[1,2,1,11],
	[1,2,2,10],[1,2,3,9],[1,2,4,8],[1,2,5,7],
	[1,2,6,6],[1,2,7,5],[1,2,8,4],[1,2,9,3],
	[1,2,10,2],[1,3,1,10],[1,3,2,9],[1,3,3,8],
	[1,3,4,7],[1,3,5,6],[1,3,6,5],[1,3,7,4],
	[1,3,8,3],[1,3,9,2],[1,4,1,9],[1,4,2,8],
	[1,4,3,7],[1,4,4,6],[1,4,6,4],[1,4,7,3],
	[1,4,8,2],[1,5,1,8],[1,5,2,7],[1,5,3,6],
	[1,5,4,5],[1,5,6,3],[1,5,7,2],[1,6,1,7],
	[1,6,2,6],[1,6,3,5],[1,6,4,4],[1,6,5,3],
	[1,6,6,2],[1,7,2,5],[1,7,3,4],[1,7,4,3],
	[1,7,5,2],[1,8,2,4],[1,8,3,3],[1,8,4,2],
	[1,9,2,3],[1,9,3,2],[1,10,2,2],[2,2,2,9],
	[2,2,3,8],[2,2,4,7],[2,2,5,6],[2,2,6,5],
	[2,2,7,4],[2,2,8,3],[2,3,2,8],[2,3,3,7],
	[2,3,4,6],[2,3,6,4],[2,3,7,3],[2,4,2,7],
	[2,4,3,6],[2,4,4,5],[2,4,5,4],[2,4,6,3],
	[2,5,2,6],[2,5,3,5],[2,5,4,4],[2,6,3,4],
	[2,6,4,3],[2,7,3,3],[3,3,3,6],[3,3,4,5],
	[3,3,5,4],[3,4,3,5],[3,4,4,4]];
	all[16]:=[[1,1,1,13],[1,1,2,12],[1,1,3,11],[1,1,4,10],
	[1,1,5,9],[1,1,6,8],[1,1,7,7],[1,1,8,6],
	[1,1,9,5],[1,1,10,4],[1,1,11,3],[1,1,12,2],
	[1,2,1,12],[1,2,2,11],[1,2,3,10],[1,2,4,9],
	[1,2,5,8],[1,2,6,7],[1,2,7,6],[1,2,8,5],
	[1,2,9,4],[1,2,10,3],[1,2,11,2],[1,3,1,11],
	[1,3,2,10],[1,3,3,9],[1,3,4,8],[1,3,5,7],
	[1,3,6,6],[1,3,7,5],[1,3,8,4],[1,3,9,3],
	[1,3,10,2],[1,4,1,10],[1,4,2,9],[1,4,3,8],
	[1,4,4,7],[1,4,5,6],[1,4,6,5],[1,4,7,4],
	[1,4,8,3],[1,4,9,2],[1,5,1,9],[1,5,2,8],
	[1,5,3,7],[1,5,4,6],[1,5,5,5],[1,5,6,4],
	[1,5,7,3],[1,5,8,2],[1,6,1,8],[1,6,2,7],
	[1,6,3,6],[1,6,4,5],[1,6,5,4],[1,6,6,3],
	[1,6,7,2],[1,7,1,7],[1,7,2,6],[1,7,3,5],
	[1,7,4,4],[1,7,5,3],[1,7,6,2],[1,8,2,5],
	[1,8,3,4],[1,8,4,3],[1,8,5,2],[1,9,2,4],
	[1,9,3,3],[1,9,4,2],[1,10,2,3],[1,10,3,2],
	[1,11,2,2],[2,2,2,10],[2,2,3,9],[2,2,4,8],
	[2,2,5,7],[2,2,6,6],[2,2,7,5],[2,2,8,4],
	[2,2,9,3],[2,3,2,9],[2,3,3,8],[2,3,4,7],
	[2,3,5,6],[2,3,6,5],[2,3,7,4],[2,3,8,3],
	[2,4,2,8],[2,4,3,7],[2,4,4,6],[2,4,5,5],
	[2,4,6,4],[2,4,7,3],[2,5,2,7],[2,5,3,6],
	[2,5,4,5],[2,5,5,4],[2,5,6,3],[2,6,2,6],
	[2,6,3,5],[2,6,4,4],[2,6,5,3],[2,7,3,4],
	[2,7,4,3],[2,8,3,3],[3,3,3,7],[3,3,4,6],
	[3,3,5,5],[3,3,6,4],[3,4,3,6],[3,4,4,5],
	[3,4,5,4],[3,5,3,5],[3,5,4,4],[4,4,4,4]];
	all[17]:=[[1,1,1,14],[1,1,2,13],[1,1,3,12],[1,1,4,11],
	[1,1,5,10],[1,1,6,9],[1,1,7,8],[1,1,8,7],
	[1,1,9,6],[1,1,10,5],[1,1,11,4],[1,1,12,3],
	[1,1,13,2],[1,2,1,13],[1,2,2,12],[1,2,3,11],
	[1,2,4,10],[1,2,5,9],[1,2,6,8],[1,2,7,7],
	[1,2,8,6],[1,2,9,5],[1,2,10,4],[1,2,11,3],
	[1,2,12,2],[1,3,1,12],[1,3,2,11],[1,3,3,10],
	[1,3,4,9],[1,3,5,8],[1,3,6,7],[1,3,7,6],
	[1,3,8,5],[1,3,9,4],[1,3,10,3],[1,3,11,2],
	[1,4,1,11],[1,4,2,10],[1,4,3,9],[1,4,4,8],
	[1,4,5,7],[1,4,6,6],[1,4,7,5],[1,4,8,4],
	[1,4,9,3],[1,4,10,2],[1,5,1,10],[1,5,2,9],
	[1,5,3,8],[1,5,4,7],[1,5,5,6],[1,5,6,5],
	[1,5,7,4],[1,5,8,3],[1,5,9,2],[1,6,1,9],
	[1,6,2,8],[1,6,3,7],[1,6,4,6],[1,6,5,5],
	[1,6,6,4],[1,6,7,3],[1,6,8,2],[1,7,1,8],
	[1,7,2,7],[1,7,3,6],[1,7,4,5],[1,7,5,4],
	[1,7,6,3],[1,7,7,2],[1,8,2,6],[1,8,3,5],
	[1,8,4,4],[1,8,5,3],[1,8,6,2],[1,9,2,5],
	[1,9,3,4],[1,9,4,3],[1,9,5,2],[1,10,2,4],
	[1,10,3,3],[1,10,4,2],[1,11,2,3],[1,11,3,2],
	[1,12,2,2],[2,2,2,11],[2,2,3,10],[2,2,4,9],
	[2,2,5,8],[2,2,6,7],[2,2,7,6],[2,2,8,5],
	[2,2,9,4],[2,2,10,3],[2,3,2,10],[2,3,3,9],
	[2,3,4,8],[2,3,5,7],[2,3,6,6],[2,3,7,5],
	[2,3,8,4],[2,3,9,3],[2,4,2,9],[2,4,3,8],
	[2,4,4,7],[2,4,5,6],[2,4,6,5],[2,4,7,4],
	[2,4,8,3],[2,5,2,8],[2,5,3,7],[2,5,4,6],
	[2,5,5,5],[2,5,6,4],[2,5,7,3],[2,6,2,7],
	[2,6,3,6],[2,6,4,5],[2,6,5,4],[2,6,6,3],
	[2,7,3,5],[2,7,4,4],[2,7,5,3],[2,8,3,4],
	[2,8,4,3],[2,9,3,3],[3,3,3,8],[3,3,4,7],
	[3,3,5,6],[3,3,6,5],[3,3,7,4],[3,4,3,7],
	[3,4,4,6],[3,4,5,5],[3,4,6,4],[3,5,3,6],
	[3,5,4,5],[3,5,5,4],[3,6,4,4],[4,4,4,5]];
	all[18]:=[[1,1,1,15],[1,1,2,14],[1,1,3,13],[1,1,4,12],
	[1,1,5,11],[1,1,6,10],[1,1,7,9],[1,1,8,8],
	[1,1,9,7],[1,1,10,6],[1,1,11,5],[1,1,12,4],
	[1,1,13,3],[1,1,14,2],[1,2,1,14],[1,2,2,13],
	[1,2,3,12],[1,2,4,11],[1,2,5,10],[1,2,6,9],
	[1,2,7,8],[1,2,8,7],[1,2,9,6],[1,2,10,5],
	[1,2,11,4],[1,2,12,3],[1,2,13,2],[1,3,1,13],
	[1,3,2,12],[1,3,3,11],[1,3,4,10],[1,3,5,9],
	[1,3,6,8],[1,3,7,7],[1,3,8,6],[1,3,9,5],
	[1,3,10,4],[1,3,11,3],[1,3,12,2],[1,4,1,12],
	[1,4,2,11],[1,4,3,10],[1,4,4,9],[1,4,5,8],
	[1,4,6,7],[1,4,7,6],[1,4,8,5],[1,4,9,4],
	[1,4,10,3],[1,4,11,2],[1,5,1,11],[1,5,2,10],
	[1,5,3,9],[1,5,4,8],[1,5,5,7],[1,5,7,5],
	[1,5,8,4],[1,5,9,3],[1,5,10,2],[1,6,1,10],
	[1,6,2,9],[1,6,3,8],[1,6,4,7],[1,6,5,6],
	[1,6,7,4],[1,6,8,3],[1,6,9,2],[1,7,1,9],
	[1,7,2,8],[1,7,3,7],[1,7,4,6],[1,7,5,5],
	[1,7,6,4],[1,7,7,3],[1,7,8,2],[1,8,1,8],
	[1,8,2,7],[1,8,3,6],[1,8,4,5],[1,8,5,4],
	[1,8,6,3],[1,8,7,2],[1,9,2,6],[1,9,3,5],
	[1,9,4,4],[1,9,5,3],[1,9,6,2],[1,10,2,5],
	[1,10,3,4],[1,10,4,3],[1,10,5,2],[1,11,2,4],
	[1,11,3,3],[1,11,4,2],[1,12,2,3],[1,12,3,2],
	[1,13,2,2],[2,2,2,12],[2,2,3,11],[2,2,4,10],
	[2,2,5,9],[2,2,6,8],[2,2,7,7],[2,2,8,6],
	[2,2,9,5],[2,2,10,4],[2,2,11,3],[2,3,2,11],
	[2,3,3,10],[2,3,4,9],[2,3,5,8],[2,3,6,7],
	[2,3,7,6],[2,3,8,5],[2,3,9,4],[2,3,10,3],
	[2,4,2,10],[2,4,3,9],[2,4,4,8],[2,4,5,7],
	[2,4,7,5],[2,4,8,4],[2,4,9,3],[2,5,2,9],
	[2,5,3,8],[2,5,4,7],[2,5,5,6],[2,5,6,5],
	[2,5,7,4],[2,5,8,3],[2,6,2,8],[2,6,3,7],
	[2,6,4,6],[2,6,5,5],[2,6,7,3],[2,7,2,7],
	[2,7,3,6],[2,7,4,5],[2,7,5,4],[2,7,6,3],
	[2,8,3,5],[2,8,4,4],[2,8,5,3],[2,9,3,4],
	[2,9,4,3],[2,10,3,3],[3,3,3,9],[3,3,4,8],
	[3,3,5,7],[3,3,7,5],[3,3,8,4],[3,4,3,8],
	[3,4,4,7],[3,4,5,6],[3,4,6,5],[3,4,7,4],
	[3,5,3,7],[3,5,4,6],[3,5,5,5],[3,5,6,4],
	[3,6,3,6],[3,6,4,5],[3,6,5,4],[3,7,4,4],
	[4,4,4,6],[4,4,5,5],[4,5,4,5]];
	all[19]:=[[1,1,1,16],[1,1,2,15],[1,1,3,14],[1,1,4,13],
	[1,1,5,12],[1,1,6,11],[1,1,7,10],[1,1,8,9],
	[1,1,9,8],[1,1,10,7],[1,1,11,6],[1,1,12,5],
	[1,1,13,4],[1,1,14,3],[1,1,15,2],[1,2,1,15],
	[1,2,2,14],[1,2,3,13],[1,2,4,12],[1,2,5,11],
	[1,2,6,10],[1,2,7,9],[1,2,8,8],[1,2,9,7],
	[1,2,10,6],[1,2,11,5],[1,2,12,4],[1,2,13,3],
	[1,2,14,2],[1,3,1,14],[1,3,2,13],[1,3,3,12],
	[1,3,4,11],[1,3,5,10],[1,3,6,9],[1,3,7,8],
	[1,3,8,7],[1,3,9,6],[1,3,10,5],[1,3,11,4],
	[1,3,12,3],[1,3,13,2],[1,4,1,13],[1,4,2,12],
	[1,4,3,11],[1,4,4,10],[1,4,5,9],[1,4,6,8],
	[1,4,7,7],[1,4,8,6],[1,4,9,5],[1,4,10,4],
	[1,4,11,3],[1,4,12,2],[1,5,1,12],[1,5,2,11],
	[1,5,3,10],[1,5,4,9],[1,5,5,8],[1,5,6,7],
	[1,5,7,6],[1,5,8,5],[1,5,9,4],[1,5,10,3],
	[1,5,11,2],[1,6,1,11],[1,6,2,10],[1,6,3,9],
	[1,6,4,8],[1,6,5,7],[1,6,6,6],[1,6,7,5],
	[1,6,8,4],[1,6,9,3],[1,6,10,2],[1,7,1,10],
	[1,7,2,9],[1,7,3,8],[1,7,4,7],[1,7,5,6],
	[1,7,6,5],[1,7,7,4],[1,7,8,3],[1,7,9,2],
	[1,8,1,9],[1,8,2,8],[1,8,3,7],[1,8,4,6],
	[1,8,5,5],[1,8,6,4],[1,8,7,3],[1,8,8,2],
	[1,9,2,7],[1,9,3,6],[1,9,4,5],[1,9,5,4],
	[1,9,6,3],[1,9,7,2],[1,10,2,6],[1,10,3,5],
	[1,10,4,4],[1,10,5,3],[1,10,6,2],[1,11,2,5],
	[1,11,3,4],[1,11,4,3],[1,11,5,2],[1,12,2,4],
	[1,12,3,3],[1,12,4,2],[1,13,2,3],[1,13,3,2],
	[1,14,2,2],[2,2,2,13],[2,2,3,12],[2,2,4,11],
	[2,2,5,10],[2,2,6,9],[2,2,7,8],[2,2,8,7],
	[2,2,9,6],[2,2,10,5],[2,2,11,4],[2,2,12,3],
	[2,3,2,12],[2,3,3,11],[2,3,4,10],[2,3,5,9],
	[2,3,6,8],[2,3,7,7],[2,3,8,6],[2,3,9,5],
	[2,3,10,4],[2,3,11,3],[2,4,2,11],[2,4,3,10],
	[2,4,4,9],[2,4,5,8],[2,4,6,7],[2,4,7,6],
	[2,4,8,5],[2,4,9,4],[2,4,10,3],[2,5,2,10],
	[2,5,3,9],[2,5,4,8],[2,5,5,7],[2,5,6,6],
	[2,5,7,5],[2,5,8,4],[2,5,9,3],[2,6,2,9],
	[2,6,3,8],[2,6,4,7],[2,6,5,6],[2,6,6,5],
	[2,6,7,4],[2,6,8,3],[2,7,2,8],[2,7,3,7],
	[2,7,4,6],[2,7,5,5],[2,7,6,4],[2,7,7,3],
	[2,8,3,6],[2,8,4,5],[2,8,5,4],[2,8,6,3],
	[2,9,3,5],[2,9,4,4],[2,9,5,3],[2,10,3,4],
	[2,10,4,3],[2,11,3,3],[3,3,3,10],[3,3,4,9],
	[3,3,5,8],[3,3,6,7],[3,3,7,6],[3,3,8,5],
	[3,3,9,4],[3,4,3,9],[3,4,4,8],[3,4,5,7],
	[3,4,6,6],[3,4,7,5],[3,4,8,4],[3,5,3,8],
	[3,5,4,7],[3,5,5,6],[3,5,6,5],[3,5,7,4],
	[3,6,3,7],[3,6,4,6],[3,6,5,5],[3,6,6,4],
	[3,7,4,5],[3,7,5,4],[3,8,4,4],[4,4,4,7],
	[4,4,5,6],[4,4,6,5],[4,5,4,6],[4,5,5,5]];
	all[20]:=[[1,1,1,17],[1,1,2,16],[1,1,3,15],[1,1,4,14],
	[1,1,5,13],[1,1,6,12],[1,1,7,11],[1,1,8,10],
	[1,1,9,9],[1,1,10,8],[1,1,11,7],[1,1,12,6],
	[1,1,13,5],[1,1,14,4],[1,1,15,3],[1,1,16,2],
	[1,2,1,16],[1,2,2,15],[1,2,3,14],[1,2,4,13],
	[1,2,5,12],[1,2,6,11],[1,2,7,10],[1,2,8,9],
	[1,2,9,8],[1,2,10,7],[1,2,11,6],[1,2,12,5],
	[1,2,13,4],[1,2,14,3],[1,2,15,2],[1,3,1,15],
	[1,3,2,14],[1,3,3,13],[1,3,4,12],[1,3,5,11],
	[1,3,6,10],[1,3,7,9],[1,3,8,8],[1,3,9,7],
	[1,3,10,6],[1,3,11,5],[1,3,12,4],[1,3,13,3],
	[1,3,14,2],[1,4,1,14],[1,4,2,13],[1,4,3,12],
	[1,4,4,11],[1,4,5,10],[1,4,6,9],[1,4,7,8],
	[1,4,8,7],[1,4,9,6],[1,4,10,5],[1,4,11,4],
	[1,4,12,3],[1,4,13,2],[1,5,1,13],[1,5,2,12],
	[1,5,3,11],[1,5,4,10],[1,5,5,9],[1,5,6,8],
	[1,5,7,7],[1,5,8,6],[1,5,9,5],[1,5,10,4],
	[1,5,11,3],[1,5,12,2],[1,6,1,12],[1,6,2,11],
	[1,6,3,10],[1,6,4,9],[1,6,5,8],[1,6,6,7],
	[1,6,7,6],[1,6,8,5],[1,6,9,4],[1,6,10,3],
	[1,6,11,2],[1,7,1,11],[1,7,2,10],[1,7,3,9],
	[1,7,4,8],[1,7,5,7],[1,7,6,6],[1,7,7,5],
	[1,7,8,4],[1,7,9,3],[1,7,10,2],[1,8,1,10],
	[1,8,2,9],[1,8,3,8],[1,8,4,7],[1,8,5,6],
	[1,8,6,5],[1,8,7,4],[1,8,8,3],[1,8,9,2],
	[1,9,1,9],[1,9,2,8],[1,9,3,7],[1,9,4,6],
	[1,9,5,5],[1,9,6,4],[1,9,7,3],[1,9,8,2],
	[1,10,2,7],[1,10,3,6],[1,10,4,5],[1,10,5,4],
	[1,10,6,3],[1,10,7,2],[1,11,2,6],[1,11,3,5],
	[1,11,4,4],[1,11,5,3],[1,11,6,2],[1,12,2,5],
	[1,12,3,4],[1,12,4,3],[1,12,5,2],[1,13,2,4],
	[1,13,3,3],[1,13,4,2],[1,14,2,3],[1,14,3,2],
	[1,15,2,2],[2,2,2,14],[2,2,3,13],[2,2,4,12],
	[2,2,5,11],[2,2,6,10],[2,2,7,9],[2,2,8,8],
	[2,2,9,7],[2,2,10,6],[2,2,11,5],[2,2,12,4],
	[2,2,13,3],[2,3,2,13],[2,3,3,12],[2,3,4,11],
	[2,3,5,10],[2,3,6,9],[2,3,7,8],[2,3,8,7],
	[2,3,9,6],[2,3,10,5],[2,3,11,4],[2,3,12,3],
	[2,4,2,12],[2,4,3,11],[2,4,4,10],[2,4,5,9],
	[2,4,6,8],[2,4,7,7],[2,4,8,6],[2,4,9,5],
	[2,4,10,4],[2,4,11,3],[2,5,2,11],[2,5,3,10],
	[2,5,4,9],[2,5,5,8],[2,5,6,7],[2,5,7,6],
	[2,5,8,5],[2,5,9,4],[2,5,10,3],[2,6,2,10],
	[2,6,3,9],[2,6,4,8],[2,6,5,7],[2,6,6,6],
	[2,6,7,5],[2,6,8,4],[2,6,9,3],[2,7,2,9],
	[2,7,3,8],[2,7,4,7],[2,7,5,6],[2,7,6,5],
	[2,7,7,4],[2,7,8,3],[2,8,2,8],[2,8,3,7],
	[2,8,4,6],[2,8,5,5],[2,8,6,4],[2,8,7,3],
	[2,9,3,6],[2,9,4,5],[2,9,5,4],[2,9,6,3],
	[2,10,3,5],[2,10,4,4],[2,10,5,3],[2,11,3,4],
	[2,11,4,3],[2,12,3,3],[3,3,3,11],[3,3,4,10],
	[3,3,5,9],[3,3,6,8],[3,3,7,7],[3,3,8,6],
	[3,3,9,5],[3,3,10,4],[3,4,3,10],[3,4,4,9],
	[3,4,5,8],[3,4,6,7],[3,4,7,6],[3,4,8,5],
	[3,4,9,4],[3,5,3,9],[3,5,4,8],[3,5,5,7],
	[3,5,6,6],[3,5,7,5],[3,5,8,4],[3,6,3,8],
	[3,6,4,7],[3,6,5,6],[3,6,6,5],[3,6,7,4],
	[3,7,3,7],[3,7,4,6],[3,7,5,5],[3,7,6,4],
	[3,8,4,5],[3,8,5,4],[3,9,4,4],[4,4,4,8],
	[4,4,5,7],[4,4,6,6],[4,4,7,5],[4,5,4,7],
	[4,5,5,6],[4,5,6,5],[4,6,4,6],[4,6,5,5],
	[5,5,5,5]];
	all[21]:=[[1,1,1,18],[1,1,2,17],[1,1,3,16],[1,1,4,15],
	[1,1,5,14],[1,1,6,13],[1,1,7,12],[1,1,8,11],
	[1,1,9,10],[1,1,10,9],[1,1,11,8],[1,1,12,7],
	[1,1,13,6],[1,1,14,5],[1,1,15,4],[1,1,16,3],
	[1,1,17,2],[1,2,1,17],[1,2,2,16],[1,2,3,15],
	[1,2,4,14],[1,2,5,13],[1,2,6,12],[1,2,7,11],
	[1,2,8,10],[1,2,9,9],[1,2,10,8],[1,2,11,7],
	[1,2,12,6],[1,2,13,5],[1,2,14,4],[1,2,15,3],
	[1,2,16,2],[1,3,1,16],[1,3,2,15],[1,3,3,14],
	[1,3,4,13],[1,3,5,12],[1,3,6,11],[1,3,7,10],
	[1,3,8,9],[1,3,9,8],[1,3,10,7],[1,3,11,6],
	[1,3,12,5],[1,3,13,4],[1,3,14,3],[1,3,15,2],
	[1,4,1,15],[1,4,2,14],[1,4,3,13],[1,4,4,12],
	[1,4,5,11],[1,4,6,10],[1,4,7,9],[1,4,8,8],
	[1,4,9,7],[1,4,10,6],[1,4,11,5],[1,4,12,4],
	[1,4,13,3],[1,4,14,2],[1,5,1,14],[1,5,2,13],
	[1,5,3,12],[1,5,4,11],[1,5,5,10],[1,5,6,9],
	[1,5,7,8],[1,5,8,7],[1,5,9,6],[1,5,10,5],
	[1,5,11,4],[1,5,12,3],[1,5,13,2],[1,6,1,13],
	[1,6,2,12],[1,6,3,11],[1,6,4,10],[1,6,5,9],
	[1,6,6,8],[1,6,8,6],[1,6,9,5],[1,6,10,4],
	[1,6,11,3],[1,6,12,2],[1,7,1,12],[1,7,2,11],
	[1,7,3,10],[1,7,4,9],[1,7,5,8],[1,7,6,7],
	[1,7,8,5],[1,7,9,4],[1,7,10,3],[1,7,11,2],
	[1,8,1,11],[1,8,2,10],[1,8,3,9],[1,8,4,8],
	[1,8,5,7],[1,8,6,6],[1,8,7,5],[1,8,8,4],
	[1,8,9,3],[1,8,10,2],[1,9,1,10],[1,9,2,9],
	[1,9,3,8],[1,9,4,7],[1,9,5,6],[1,9,6,5],
	[1,9,7,4],[1,9,8,3],[1,9,9,2],[1,10,2,8],
	[1,10,3,7],[1,10,4,6],[1,10,5,5],[1,10,6,4],
	[1,10,7,3],[1,10,8,2],[1,11,2,7],[1,11,3,6],
	[1,11,4,5],[1,11,5,4],[1,11,6,3],[1,11,7,2],
	[1,12,2,6],[1,12,3,5],[1,12,4,4],[1,12,5,3],
	[1,12,6,2],[1,13,2,5],[1,13,3,4],[1,13,4,3],
	[1,13,5,2],[1,14,2,4],[1,14,3,3],[1,14,4,2],
	[1,15,2,3],[1,15,3,2],[1,16,2,2],[2,2,2,15],
	[2,2,3,14],[2,2,4,13],[2,2,5,12],[2,2,6,11],
	[2,2,7,10],[2,2,8,9],[2,2,9,8],[2,2,10,7],
	[2,2,11,6],[2,2,12,5],[2,2,13,4],[2,2,14,3],
	[2,3,2,14],[2,3,3,13],[2,3,4,12],[2,3,5,11],
	[2,3,6,10],[2,3,7,9],[2,3,8,8],[2,3,9,7],
	[2,3,10,6],[2,3,11,5],[2,3,12,4],[2,3,13,3],
	[2,4,2,13],[2,4,3,12],[2,4,4,11],[2,4,5,10],
	[2,4,6,9],[2,4,7,8],[2,4,8,7],[2,4,9,6],
	[2,4,10,5],[2,4,11,4],[2,4,12,3],[2,5,2,12],
	[2,5,3,11],[2,5,4,10],[2,5,5,9],[2,5,6,8],
	[2,5,8,6],[2,5,9,5],[2,5,10,4],[2,5,11,3],
	[2,6,2,11],[2,6,3,10],[2,6,4,9],[2,6,5,8],
	[2,6,6,7],[2,6,7,6],[2,6,8,5],[2,6,9,4],
	[2,6,10,3],[2,7,2,10],[2,7,3,9],[2,7,4,8],
	[2,7,5,7],[2,7,6,6],[2,7,8,4],[2,7,9,3],
	[2,8,2,9],[2,8,3,8],[2,8,4,7],[2,8,5,6],
	[2,8,6,5],[2,8,7,4],[2,8,8,3],[2,9,3,7],
	[2,9,4,6],[2,9,5,5],[2,9,6,4],[2,9,7,3],
	[2,10,3,6],[2,10,4,5],[2,10,5,4],[2,10,6,3],
	[2,11,3,5],[2,11,4,4],[2,11,5,3],[2,12,3,4],
	[2,12,4,3],[2,13,3,3],[3,3,3,12],[3,3,4,11],
	[3,3,5,10],[3,3,6,9],[3,3,7,8],[3,3,8,7],
	[3,3,9,6],[3,3,10,5],[3,3,11,4],[3,4,3,11],
	[3,4,4,10],[3,4,5,9],[3,4,6,8],[3,4,8,6],
	[3,4,9,5],[3,4,10,4],[3,5,3,10],[3,5,4,9],
	[3,5,5,8],[3,5,6,7],[3,5,7,6],[3,5,8,5],
	[3,5,9,4],[3,6,3,9],[3,6,4,8],[3,6,5,7],
	[3,6,6,6],[3,6,7,5],[3,6,8,4],[3,7,3,8],
	[3,7,4,7],[3,7,5,6],[3,7,6,5],[3,8,4,6],
	[3,8,5,5],[3,8,6,4],[3,9,4,5],[3,9,5,4],
	[3,10,4,4],[4,4,4,9],[4,4,5,8],[4,4,6,7],
	[4,4,7,6],[4,4,8,5],[4,5,4,8],[4,5,5,7],
	[4,5,6,6],[4,5,7,5],[4,6,4,7],[4,6,5,6],
	[4,6,6,5],[4,7,5,5],[5,5,5,6]];
	all[22]:=[[1,1,1,19],[1,1,2,18],[1,1,3,17],[1,1,4,16],
	[1,1,5,15],[1,1,6,14],[1,1,7,13],[1,1,8,12],
	[1,1,9,11],[1,1,10,10],[1,1,11,9],[1,1,12,8],
	[1,1,13,7],[1,1,14,6],[1,1,15,5],[1,1,16,4],
	[1,1,17,3],[1,1,18,2],[1,2,1,18],[1,2,2,17],
	[1,2,3,16],[1,2,4,15],[1,2,5,14],[1,2,6,13],
	[1,2,7,12],[1,2,8,11],[1,2,9,10],[1,2,10,9],
	[1,2,11,8],[1,2,12,7],[1,2,13,6],[1,2,14,5],
	[1,2,15,4],[1,2,16,3],[1,2,17,2],[1,3,1,17],
	[1,3,2,16],[1,3,3,15],[1,3,4,14],[1,3,5,13],
	[1,3,6,12],[1,3,7,11],[1,3,8,10],[1,3,9,9],
	[1,3,10,8],[1,3,11,7],[1,3,12,6],[1,3,13,5],
	[1,3,14,4],[1,3,15,3],[1,3,16,2],[1,4,1,16],
	[1,4,2,15],[1,4,3,14],[1,4,4,13],[1,4,5,12],
	[1,4,6,11],[1,4,7,10],[1,4,8,9],[1,4,9,8],
	[1,4,10,7],[1,4,11,6],[1,4,12,5],[1,4,13,4],
	[1,4,14,3],[1,4,15,2],[1,5,1,15],[1,5,2,14],
	[1,5,3,13],[1,5,4,12],[1,5,5,11],[1,5,6,10],
	[1,5,7,9],[1,5,8,8],[1,5,9,7],[1,5,10,6],
	[1,5,11,5],[1,5,12,4],[1,5,13,3],[1,5,14,2],
	[1,6,1,14],[1,6,2,13],[1,6,3,12],[1,6,4,11],
	[1,6,5,10],[1,6,6,9],[1,6,7,8],[1,6,8,7],
	[1,6,9,6],[1,6,10,5],[1,6,11,4],[1,6,12,3],
	[1,6,13,2],[1,7,1,13],[1,7,2,12],[1,7,3,11],
	[1,7,4,10],[1,7,5,9],[1,7,6,8],[1,7,7,7],
	[1,7,8,6],[1,7,9,5],[1,7,10,4],[1,7,11,3],
	[1,7,12,2],[1,8,1,12],[1,8,2,11],[1,8,3,10],
	[1,8,4,9],[1,8,5,8],[1,8,6,7],[1,8,7,6],
	[1,8,8,5],[1,8,9,4],[1,8,10,3],[1,8,11,2],
	[1,9,1,11],[1,9,2,10],[1,9,3,9],[1,9,4,8],
	[1,9,5,7],[1,9,6,6],[1,9,7,5],[1,9,8,4],
	[1,9,9,3],[1,9,10,2],[1,10,1,10],[1,10,2,9],
	[1,10,3,8],[1,10,4,7],[1,10,5,6],[1,10,6,5],
	[1,10,7,4],[1,10,8,3],[1,10,9,2],[1,11,2,8],
	[1,11,3,7],[1,11,4,6],[1,11,5,5],[1,11,6,4],
	[1,11,7,3],[1,11,8,2],[1,12,2,7],[1,12,3,6],
	[1,12,4,5],[1,12,5,4],[1,12,6,3],[1,12,7,2],
	[1,13,2,6],[1,13,3,5],[1,13,4,4],[1,13,5,3],
	[1,13,6,2],[1,14,2,5],[1,14,3,4],[1,14,4,3],
	[1,14,5,2],[1,15,2,4],[1,15,3,3],[1,15,4,2],
	[1,16,2,3],[1,16,3,2],[1,17,2,2],[2,2,2,16],
	[2,2,3,15],[2,2,4,14],[2,2,5,13],[2,2,6,12],
	[2,2,7,11],[2,2,8,10],[2,2,9,9],[2,2,10,8],
	[2,2,11,7],[2,2,12,6],[2,2,13,5],[2,2,14,4],
	[2,2,15,3],[2,3,2,15],[2,3,3,14],[2,3,4,13],
	[2,3,5,12],[2,3,6,11],[2,3,7,10],[2,3,8,9],
	[2,3,9,8],[2,3,10,7],[2,3,11,6],[2,3,12,5],
	[2,3,13,4],[2,3,14,3],[2,4,2,14],[2,4,3,13],
	[2,4,4,12],[2,4,5,11],[2,4,6,10],[2,4,7,9],
	[2,4,8,8],[2,4,9,7],[2,4,10,6],[2,4,11,5],
	[2,4,12,4],[2,4,13,3],[2,5,2,13],[2,5,3,12],
	[2,5,4,11],[2,5,5,10],[2,5,6,9],[2,5,7,8],
	[2,5,8,7],[2,5,9,6],[2,5,10,5],[2,5,11,4],
	[2,5,12,3],[2,6,2,12],[2,6,3,11],[2,6,4,10],
	[2,6,5,9],[2,6,6,8],[2,6,7,7],[2,6,8,6],
	[2,6,9,5],[2,6,10,4],[2,6,11,3],[2,7,2,11],
	[2,7,3,10],[2,7,4,9],[2,7,5,8],[2,7,6,7],
	[2,7,7,6],[2,7,8,5],[2,7,9,4],[2,7,10,3],
	[2,8,2,10],[2,8,3,9],[2,8,4,8],[2,8,5,7],
	[2,8,6,6],[2,8,7,5],[2,8,8,4],[2,8,9,3],
	[2,9,2,9],[2,9,3,8],[2,9,4,7],[2,9,5,6],
	[2,9,6,5],[2,9,7,4],[2,9,8,3],[2,10,3,7],
	[2,10,4,6],[2,10,5,5],[2,10,6,4],[2,10,7,3],
	[2,11,3,6],[2,11,4,5],[2,11,5,4],[2,11,6,3],
	[2,12,3,5],[2,12,4,4],[2,12,5,3],[2,13,3,4],
	[2,13,4,3],[2,14,3,3],[3,3,3,13],[3,3,4,12],
	[3,3,5,11],[3,3,6,10],[3,3,7,9],[3,3,8,8],
	[3,3,9,7],[3,3,10,6],[3,3,11,5],[3,3,12,4],
	[3,4,3,12],[3,4,4,11],[3,4,5,10],[3,4,6,9],
	[3,4,7,8],[3,4,8,7],[3,4,9,6],[3,4,10,5],
	[3,4,11,4],[3,5,3,11],[3,5,4,10],[3,5,5,9],
	[3,5,6,8],[3,5,7,7],[3,5,8,6],[3,5,9,5],
	[3,5,10,4],[3,6,3,10],[3,6,4,9],[3,6,5,8],
	[3,6,6,7],[3,6,7,6],[3,6,8,5],[3,6,9,4],
	[3,7,3,9],[3,7,4,8],[3,7,5,7],[3,7,6,6],
	[3,7,7,5],[3,7,8,4],[3,8,3,8],[3,8,4,7],
	[3,8,5,6],[3,8,6,5],[3,8,7,4],[3,9,4,6],
	[3,9,5,5],[3,9,6,4],[3,10,4,5],[3,10,5,4],
	[3,11,4,4],[4,4,4,10],[4,4,5,9],[4,4,6,8],
	[4,4,7,7],[4,4,8,6],[4,4,9,5],[4,5,4,9],
	[4,5,5,8],[4,5,6,7],[4,5,7,6],[4,5,8,5],
	[4,6,4,8],[4,6,5,7],[4,6,6,6],[4,6,7,5],
	[4,7,4,7],[4,7,5,6],[4,7,6,5],[4,8,5,5],
	[5,5,5,7],[5,5,6,6],[5,6,5,6]];
	
	
	dc:=[];
	dc[5]:=[[[1],"S^3"]];
	
	dc[6]:=[[[1,2],"S^3"]];
	
	dc[7]:=[[[1,4],"S^3"]];
	
	dc[8]:=[[[1,5,8],"S^3"],[[1,7],"S^3"]];
	
	dc[9]:=[[[1,6,9],"S^3"],[[2,5,6],"S^2~S^1"]];
	
	dc[10]:=[[[1,7,12,16],"S^3"],[[1,7,14],"S^3"],
	[[1,11,22],"S^3"],[[2,3,8,22],"S^3"],
	[[2,6,7],"S^2xS^1"],[[3,4,13,22],"S^2xS^1"],
	[[3,5,12,22],"S^2~S^1"],[[8,11,16,19],"S^2~S^1"]];
	
	dc[11]:=[[[1,8,14,19],"S^3"],[[1,8,17],"S^3"],
	[[1,11,16,18],"S^3"],[[1,11,20,22],"S^2~S^1"],
	[[2,6,9,14],"S^2~S^1"],[[2,7,8],"S^2~S^1"]];
	
	dc[12]:=[[[1,9,16,21,25],"S^3"],[[1,10,24,34],"S^3"],
	[[1,12,22,24,38],"S^2xS^1"],[[1,13,23,24],"S^2~S^1"],
	[[1,15,35],"S^3"],[[2,7,10,16],"S^2xS^1"],
	[[2,8,9],"S^2xS^1"],[[3,6,16,29,38],"S^2xS^1"],
	[[4,5,22,38],"S^2xS^1"],[[4,6,21,38],"S^2xS^1"],
	[[9,11,20,25,29],"S^2~S^1"],
	[[9,12,20,25,27],"S^2~S^1"],[[9,14,20,38],"S^2xS^1"],
	[[10,15,21,25,31],"S^2~S^1"],
	[[11,12,25,32,36],"(S^2xS^1)#2"],
	[[11,15,17,20,38],"S^2xS^1"],[[11,15,25,30],"S^2~S^1"],
	[[12,14,18,19,38],"S^2xS^1"],
	[[12,14,25,27,32],"(S^2xS^1)#2"],
	[[18,19,25,27],"S^2~S^1"]];
	
	dc[13]:=[[[1,10,18,25,31],"S^3"],[[1,10,18,28],"S^3"],
	[[1,12,35,48],"S^3"],[[1,13,26,30,51],"S^2~S^1"],
	[[1,13,28,30,36],"S^3"],[[1,13,38,52],"S^2~S^1"],
	[[1,14,25,30,33],"S^2~S^1"],[[1,14,32,35],"S^2~S^1"],
	[[1,17,49,52,53],"S^2~S^1"],[[1,17,50],"S^3"],
	[[2,3,14,27,43],"S^3"],[[2,7,11,19,25],"S^2~S^1"],
	[[2,8,11,18],"S^2~S^1"],[[2,9,10],"S^2~S^1"],
	[[3,4,19,47,51],"S^2~S^1"]];
	
	dc[14]:=[[[1,11,20,28,35,41],"S^3"],[[1,11,20,32],"S^3"],
	[[1,11,24,31,33],"S^3"],[[1,11,24,37,39],"S^2~S^1"],
	[[1,11,26,73],"S^3"],[[1,12,34,57,67],"S^3"],
	[[1,14,22,27,73],"S^3"],[[1,14,29,34,52,67],"S^3"],
	[[1,14,29,34,63,67],"S^2xS^1"],
	[[1,14,29,34,65,67],"L(3,1)"],
	[[1,14,32,34,41,46],"S^3"],[[1,14,45,52,67,68],"S^3"],
	[[1,14,45,63,67,68],"S^2xS^1"],
	[[1,14,45,65,67,68],"L(3,1)"],
	[[1,15,30,34,71,73],"S^3"],[[1,15,36,40,67],"S^2xS^1"],
	[[1,16,37,40],"S^2~S^1"],[[1,19,58,67],"S^3"],
	[[2,3,12,58,67],"S^3"],[[2,5,12,21,29,67],"S^3"],
	[[2,8,12,21,28],"S^2xS^1"],[[2,9,12,20],"S^2xS^1"],
	[[2,10,11],"S^2xS^1"],[[3,4,13,14,22,73],"S^3"],
	[[3,4,21,52,58,67],"S^3"],
	[[3,4,21,58,63,67],"S^2xS^1"],
	[[3,4,21,58,65,67],"L(3,1)"],
	[[3,7,20,46,49,67],"S^2~S^1"],
	[[3,8,20,49,52],"S^2xS^1"],[[3,8,20,49,63],"S^2xS^1"],
	[[3,8,20,49,65],"S^2xS^1"],
	[[3,8,21,28,58,67],"S^2~S^1"],
	[[3,9,20,58,67],"S^2~S^1"],[[4,5,12,14,57,67],"S^3"],
	[[4,5,22,27,34,73],"S^3"],[[4,5,29,52,67],"S^3"],
	[[4,5,29,63,67],"S^2xS^1"],[[4,5,29,65,67],"L(3,1)"],
	[[4,5,33,44,64,67],"L(3,1)"],
	[[4,7,28,46,67],"S^2~S^1"],[[4,8,28,52],"S^2xS^1"],
	[[4,8,28,63],"S^2xS^1"],[[4,8,28,65],"S^2xS^1"],
	[[5,6,36,67],"S^2xS^1"],[[5,7,35,67],"S^2~S^1"],
	[[11,14,20,34,41,46],"S^2~S^1"],
	[[11,15,20,34,41,43],"S^2~S^1"],
	[[11,16,26,37,40,73],"S^2~S^1"],
	[[12,19,34,53,73],"S^2~S^1"],
	[[23,26,41,44],"S^2~S^1"]];
	
	dc[15]:=[[[1,12,22,31,38,44],"S^3"],
	[[1,12,26,38,42,46],"S^2~S^1"],
	[[1,12,27,41,42],"S^2~S^1"],[[1,12,29,87],"S^3"],
	[[1,14,43,68,86,87],"RP^3"],[[1,15,24,30,87],"S^3"],
	[[1,15,28,30,47,84],"S^3"],
	[[1,15,32,37,72,77],"S^2~S^1"],
	[[1,15,49,69,70,75],"S^2~S^1"],
	[[1,15,49,72,80],"S^2~S^1"],[[1,15,49,73,87],"S^3/Q8"],
	[[1,16,33,37,86],"S^2~S^1"],
	[[1,16,39,43,77],"S^2~S^1"],[[1,16,53,78],"S^2~S^1"],
	[[1,17,26,27,41,43],"S^2~S^1"],
	[[1,17,34,37,79,80],"S^3/Q8"],
	[[1,17,38,43,46],"S^2~S^1"],
	[[1,17,40,43,86,87],"S^3/Q8"],[[1,17,45,49],"S^2~S^1"],
	[[1,19,24,30,47,74],"S^3"],
	[[1,21,67,74,75,79],"S^2~S^1"],
	[[1,21,69,72,76,80],"S^2~S^1"],
	[[1,21,70,75,76],"S^2~S^1"],[[1,21,71,86,87],"RP^3"],
	[[2,8,13,23,32,38],"S^2~S^1"],
	[[2,8,18,26,63,78],"S^2~S^1"],
	[[2,9,13,23,31],"S^2~S^1"],[[2,10,13,22],"S^2~S^1"],
	[[2,11,12],"S^2~S^1"],[[3,4,23,67,72,77],"S^2~S^1"],
	[[4,5,15,16,33,86],"S^2~S^1"],
	[[4,5,32,72,77],"S^2~S^1"],
	[[4,5,37,49,72,80],"S^2~S^1"],
	[[15,19,32,37,54,56],"T^3"]];
	
	dc[16]:=[[[1,13,24,34,43,51,58],"S^3"],
	[[1,13,24,34,47],"S^3"],[[1,13,28,45,49,107,114],"S^3"],
	[[1,13,28,47,49,58,60],"S^3"],
	[[1,13,29,43,49,54],"S^2~S^1"],
	[[1,13,29,53,56],"S^2~S^1"],
	[[1,14,35,57,81,104],"S^3"],[[1,14,42,81,95],"S^3"],
	[[1,15,50,83,114],"S^3"],[[1,16,35,42,74,95],"S^3"],
	[[1,16,35,42,89,95,100],"S^2xS^1"],
	[[1,16,35,42,93,95],"L(3,1)"],[[1,16,57,74,104],"S^3"],
	[[1,16,57,79,81,94],"S^3"],
	[[1,16,57,89,100,104],"S^2xS^1"],
	[[1,16,57,90,111],"S^2~S^1"],
	[[1,16,57,93,104],"L(3,1)"],
	[[1,16,57,94,105,113],"S^2~S^1"],
	[[1,17,44,50,95,100],"S^2xS^1"],
	[[1,17,47,50,58,64],"S^3"],[[1,17,63,96,107,114],"S^3"],
	[[1,18,52,57,100],"S^2xS^1"],
	[[1,19,53,57],"S^2~S^1"],[[1,23,74,84,94,104],"S^3"],
	[[1,23,75,77,105],"S^3"],[[1,23,82,95],"S^3"],
	[[1,23,84,89,94,100,104],"S^2xS^1"],
	[[1,23,84,90,94,111],"S^2~S^1"],
	[[1,23,84,93,94,104],"L(3,1)"],
	[[1,23,84,105,113],"S^2~S^1"],[[1,23,88,114],"S^3"],
	[[2,3,14,88,114],"S^3"],[[2,3,17,36,75,114],"S^3"],
	[[2,6,17,26,75,83,114],"S^3"],
	[[2,9,14,25,35,43],"S^2xS^1"],
	[[2,10,14,25,34],"S^2xS^1"],[[2,11,14,24],"S^2xS^1"],
	[[2,12,13],"S^2xS^1"],[[3,4,25,74,88,114],"S^3"],
	[[3,4,25,84,94,104],"B2"],[[3,4,25,87,90,105],"B2"],
	[[3,4,25,88,93,114],"S^3/P24"],
	[[3,4,28,44,58,85,92],"S^3"],
	[[3,4,28,45,74,87,114],"S^3"],
	[[3,4,28,45,87,93,114],"S^3/P24"],
	[[3,4,28,49,58,60,71],"S^2~S^1"],
	[[3,4,29,46,88,94,114],"S^3/P24"],
	[[3,4,30,45,58,62,71],"S^2~S^1"],
	[[3,4,33,50,83,114],"S^3"],
	[[3,5,15,17,26,36,114],"S^3"],
	[[3,6,28,35,58,87,101],"S^3"],
	[[3,8,24,64,68,71,100],"S^2xS^1"],
	[[3,9,25,35,43,88,114],"S^2~S^1"],
	[[3,10,24,54,56,87],"S^2xS^1"],
	[[3,10,24,71,74],"S^2xS^1"],
	[[3,10,24,71,89,100],"S^2xS^1"],
	[[3,10,24,71,93],"S^2xS^1"],
	[[3,10,25,34,88,114],"S^2~S^1"],
	[[3,10,28,34,45,87,114],"S^2~S^1"],
	[[3,11,24,88,114],"S^2~S^1"],
	[[4,5,35,74,82,88,114],"S^3"],
	[[4,5,35,82,88,93,114],"S^3/P24"],
	[[4,5,42,57,74,104],"S^3"],
	[[4,5,42,57,89,100,104],"S^2xS^1"],
	[[4,5,42,57,90,111],"S^2~S^1"],
	[[4,5,42,57,93,104],"L(3,1)"],
	[[4,8,34,64,68,100],"S^2xS^1"],
	[[4,8,42,50,65,68,100],"S^2xS^1"],
	[[4,9,14,16,43,81],"S^2xS^1"],
	[[4,9,34,68,82,88,114],"S^2~S^1"],
	[[4,9,35,43,74],"S^2xS^1"],
	[[4,9,35,43,89,100],"S^2xS^1"],
	[[4,9,35,43,93],"S^2xS^1"],[[4,10,34,74],"S^2xS^1"],
	[[4,10,34,89,100],"S^2xS^1"],
	[[4,10,34,90,104,111],"T^3"],[[4,10,34,93],"S^2xS^1"],
	[[5,6,15,17,83,114],"S^3"],[[5,8,43,64,100],"S^2xS^1"],
	[[5,9,43,82,88,114],"S^2~S^1"],
	[[6,7,52,100],"S^2xS^1"],[[6,8,51,100],"S^2xS^1"],
	[[13,16,24,42,51,58,68],"S^2~S^1"],
	[[13,17,24,34,50,58,64],"S^2~S^1"],
	[[13,17,24,42,58,65],"S^2~S^1"],
	[[13,17,32,36,42,114],"S^2~S^1"],
	[[13,17,32,63,96,111,114],"S^2~S^1"],
	[[13,23,29,53,56,88,114],"S^2~S^1"],
	[[14,17,42,44,50,81,100],"S^2xS^1"],
	[[14,17,42,63,81,100,101],"S^2xS^1"],
	[[14,21,24,32,81],"S^2xS^1"],
	[[16,21,24,32,35,74],"S^2xS^1"],
	[[16,21,24,32,35,89,100],"S^2xS^1"],
	[[16,21,24,32,35,93],"S^2xS^1"],
	[[17,18,27,28,96,100,112],"(S^2xS^1)#5"],
	[[17,18,27,31,65,96,100],"(S^2xS^1)#5"],
	[[17,18,28,31,61,96,100],"(S^2xS^1)#5"],
	[[17,18,36,37,96,100,115],"(S^2xS^1)#5"],
	[[17,18,36,41,66,96,100],"(S^2xS^1)#5"],
	[[17,18,37,41,62,96,100],"(S^2xS^1)#5"],
	[[17,18,61,65,96,100,112],"(S^2xS^1)#5"],
	[[17,18,62,66,96,100,115],"(S^2xS^1)#5"],
	[[17,23,44,50,82,100],"S^2xS^1"],
	[[18,21,44,49,61,66,100],"(S^2xS^1)#5"],
	[[24,27,41,58,65],"S^2~S^1"],
	[[24,28,41,58,61],"S^2~S^1"],
	[[24,31,41,114],"S^2~S^1"]];
	
	dc[17]:=[[[1,14,26,37,47,56,64],"S^3"],
	[[1,14,26,37,52],"S^3"],[[1,14,26,42,51,53],"S^3"],
	[[1,14,26,42,59,61],"S^2~S^1"],
	[[1,14,26,44,140],"S^3"],[[1,14,28,62,131],"S^3"],
	[[1,14,30,49,54,136],"S^2~S^1"],
	[[1,14,30,52,54,72],"S^3"],
	[[1,14,30,74,137],"S^2~S^1"],
	[[1,14,31,41,42,51,54],"S^3"],
	[[1,14,31,47,54,56,67],"S^2~S^1"],
	[[1,14,31,52,54,64,67],"S^3"],
	[[1,14,31,56,62,66],"S^2~S^1"],
	[[1,14,31,58,62,127],"S^3"],
	[[1,14,32,59,62],"S^2~S^1"],[[1,14,35,132],"S^3"],
	[[1,14,35,134,137,138],"S^2~S^1"],
	[[1,15,41,79,94,134],"S^3"],
	[[1,15,46,89,123,134],"S^3"],[[1,15,46,94,113],"S^3"],
	[[1,16,50,79,96,118,140],"RP^3"],
	[[1,16,55,96,137,140],"RP^3"],
	[[1,17,28,36,134,137,138],"S^2~S^1"],
	[[1,17,38,41,79,86,134],"S^3"],
	[[1,17,38,46,86,113],"S^3"],
	[[1,17,38,46,103,110,116],"S^2~S^1"],
	[[1,17,63,86,110,113,124],"S^3"],
	[[1,17,63,97,99,103,119],"S^2~S^1"],
	[[1,17,63,97,99,106,140],"Sigma^3"],
	[[1,17,63,103,116,124],"S^2~S^1"],
	[[1,18,30,33,67,75,112],"S^2~S^1"],
	[[1,18,48,55,89,91,118],"S^2~S^1"],
	[[1,18,48,55,90,92,112],"S^2~S^1"],
	[[1,18,48,55,110,116],"S^2~S^1"],
	[[1,18,70,110,121],"S^2~S^1"],
	[[1,18,70,111,137,140],"Sigma^3"],
	[[1,19,49,55,136],"S^2~S^1"],[[1,19,52,55,72],"S^3"],
	[[1,19,54,55,74,137],"S^2~S^1"],
	[[1,19,57,63,116],"S^2~S^1"],[[1,19,75,117],"S^2~S^1"],
	[[1,20,28,31,63,131],"S^3"],
	[[1,20,28,36,58,104],"S^3"],
	[[1,20,31,32,59,63],"S^2~S^1"],
	[[1,20,56,63,66],"S^2~S^1"],
	[[1,20,58,63,136,137,140],"Sigma^3"],
	[[1,20,65,70],"S^2~S^1"],
	[[1,22,43,46,89,91,134],"S^3"],
	[[1,25,95,103,106,114,118],"S^2~S^1"],
	[[1,25,95,105,108,110,122],"S^2~S^1"],
	[[1,25,95,107,108,114],"S^2~S^1"],
	[[1,25,97,103,109,116,124],"S^2~S^1"],
	[[1,25,97,119,120,122,133],"Sigma^3"],
	[[1,25,98,105,109,138,139],"RP^3"],
	[[1,25,99,103,109,119],"S^2~S^1"],
	[[1,25,99,108,109,112,114],"S^2~S^1"],
	[[1,25,99,117,120],"S^2~S^1"],
	[[1,25,102,133,135,136,139],"RP^2xS^1"],
	[[1,25,102,137,140],"RP^3"],
	[[2,3,15,99,117,120],"S^2~S^1"],
	[[2,3,21,42,90,99,117],"S^2~S^1"],
	[[2,9,15,27,38,48,56],"S^2~S^1"],
	[[2,9,15,31,42,117,120],"S^2~S^1"],
	[[2,9,21,31,90,117],"S^2~S^1"],
	[[2,9,21,33,76,90,112],"S^2~S^1"],
	[[2,10,15,27,38,47],"S^2~S^1"],
	[[2,10,20,32,76,89,118],"S^2~S^1"],
	[[2,10,22,32,91,110,111],"S^2~S^1"],
	[[2,11,15,27,37],"S^2~S^1"],[[2,12,15,26],"S^2~S^1"],
	[[2,13,14],"S^2~S^1"],[[3,4,16,20,28,58,104],"S^3"],
	[[3,4,27,95,103,110,116],"S^2~S^1"],
	[[3,4,30,54,72,83],"S^2~S^1"],
	[[3,4,33,49,74,83],"S^2~S^1"],
	[[3,4,35,49,74,97,104],"S^2~S^1"],
	[[3,6,26,61,69,99,117],"B2"],
	[[3,12,26,99,117,120],"B2"],
	[[4,5,28,36,39,70,111],"S^3"],
	[[4,5,38,103,110,116],"S^2~S^1"],
	[[4,5,46,63,103,116,124],"S^2~S^1"],
	[[4,6,39,55,63,104,136],"S^2~S^1"],
	[[4,9,15,17,48,56,94],"S^2~S^1"],
	[[4,10,15,17,47,94],"S^2~S^1"],
	[[5,6,48,110,116],"S^2~S^1"],
	[[5,6,55,70,110,121],"S^2~S^1"],
	[[14,18,33,50,55,69,76],"B2"],
	[[17,21,38,46,82,104,110],"T^3"],
	[[17,21,63,82,104,124],"T^3"]];
	
	dc[18]:=[[[1,15,28,40,51,60,68,76],"S^3"],
	[[1,15,28,45,60,65,71],"S^2~S^1"],
	[[1,15,28,45,63,65,165],"S^3"],
	[[1,15,28,46,64,65],"S^2~S^1"],
	[[1,15,28,48,167],"S^3"],
	[[1,15,30,66,151,157,159],"RP^3"],
	[[1,15,31,74,153],"S^3"],[[1,15,32,43,49,167],"S^3"],
	[[1,15,32,53,58,159,161],"S^3/Q8"],
	[[1,15,32,81,158,165,167],"S^3"],
	[[1,15,32,81,159,161,162],"S^3/Q8"],
	[[1,15,33,54,58,165,167],"S^3"],
	[[1,15,33,62,66,157,159,161],"S^3/Q8"],
	[[1,15,34,45,46,64,66],"S^2~S^1"],
	[[1,15,34,60,66,71],"S^2~S^1"],
	[[1,15,34,63,66,165],"S^3"],
	[[1,15,34,70,74],"S^2~S^1"],
	[[1,15,38,148,150,159,160],"RP^3"],
	[[1,15,38,152,157,159],"RP^3"],
	[[1,15,38,154,158,160,165,167],"S^3"],
	[[1,15,38,154,159,160,161,162],"S^3/Q8"],
	[[1,15,38,156],"S^3"],[[1,16,41,67,107,136,144],"S^3"],
	[[1,16,50,107,124,136],"S^3"],
	[[1,17,55,91,109,134],"S^3"],[[1,17,59,109,159],"S^3"],
	[[1,17,59,114,134,149],"S^3"],
	[[1,18,30,39,152,157,159],"RP^3"],
	[[1,18,34,39,63,148],"S^3"],
	[[1,18,41,50,98,124,136],"S^3"],
	[[1,18,41,50,117,124,131,136],"S^2xS^1"],
	[[1,18,41,50,122,124,136,165],"L(8,3)"],
	[[1,18,63,91,121,148,150],"S^3"],
	[[1,18,63,91,121,154,161,162],"S^3/Q8"],
	[[1,18,65,89,91,121,159],"T^3"],
	[[1,18,67,98,136,144],"S^3"],
	[[1,18,67,105,107,123,136],"S^3"],
	[[1,18,67,117,131,136,144],"S^2xS^1"],
	[[1,18,67,118,152,157,159,161],"S^3/Q8"],
	[[1,18,67,118,156,161],"S^3/Q8"],
	[[1,18,67,120,136,155,162,165],"L(8,3)"],
	[[1,18,67,121,138,154,161],"S^3/Q8"],
	[[1,18,67,122,136,144,165],"L(8,3)"],
	[[1,19,31,32,43,50,167],"S^3"],
	[[1,19,35,39,73,92,149],"S^3"],
	[[1,19,42,50,148,150,160],"S^2~S^1"],
	[[1,19,42,50,149,151,155],"S^2~S^1"],
	[[1,19,42,50,152,157],"S^2~S^1"],
	[[1,19,42,50,154,160,161,162],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,19,42,50,155,158,161,163],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,19,42,50,156,159],"T^3"],
	[[1,19,44,50,72,84,154,161],"S^3/Q8"],
	[[1,19,46,50,71,155,159],"T^3"],
	[[1,19,46,50,74,93,149],"S^3"],
	[[1,19,49,50,74,153],"S^3"],
	[[1,19,52,59,98,117,124,136],"S^3"],
	[[1,19,52,59,124,131,136],"S^2xS^1"],
	[[1,19,52,59,124,133,136,165],"L(5,1)"],
	[[1,19,72,84,87,127,161],"S^3/Q8"],
	[[1,19,75,124,136,141],"S^2xS^1"],
	[[1,19,75,125,157],"S^2~S^1"],
	[[1,20,43,49,58,59,167],"S^3"],
	[[1,20,45,50,79,88,154,161],"S^3/Q8"],
	[[1,20,57,59,80,158,161],"S^3/Q8"],
	[[1,20,61,67,98,117,136],"S^3"],
	[[1,20,61,67,131,136],"S^2xS^1"],
	[[1,20,61,67,133,136,165],"L(5,1)"],
	[[1,20,82,98,117,136,137],"S^3"],
	[[1,20,82,98,123,135,136,139],"S^3"],
	[[1,20,82,131,136,137],"S^2xS^1"],
	[[1,20,82,132,157,159,161],"S^3/Q8"],
	[[1,20,82,133,136,137,165],"L(5,1)"],
	[[1,20,82,135,145,154,156,161],"S^3/Q8"],
	[[1,21,54,58,66,67,165,167],"S^3"],
	[[1,21,54,59,132,136,158,165],"L(5,1)"],
	[[1,21,62,67,157,159,161],"S^3/Q8"],
	[[1,21,66,67,86,162,165,167],"S^3"],
	[[1,21,69,75,136],"S^2xS^1"],[[1,22,70,75],"S^2~S^1"],
	[[1,23,46,50,129,144,161],"S^3/Q8"],
	[[1,23,54,59,83,136,158,165],"L(5,1)"],
	[[1,24,47,50,127,148,150],"S^3"],
	[[1,24,47,50,127,154,161,162],"S^3/Q8"],
	[[1,25,30,39,65,121],"S^3"],
	[[1,25,37,39,121,150,164],"S^3"],
	[[1,27,98,110,123,136,144],"S^3"],
	[[1,27,98,112,117,135,136,137],"S^3"],
	[[1,27,98,112,123,136,139],"S^3"],
	[[1,27,99,101,136,145],"S^3"],[[1,27,108,124,136],"S^3"],
	[[1,27,110,117,123,131,136,144],"S^2xS^1"],
	[[1,27,110,122,123,136,144,165],"L(8,3)"],
	[[1,27,112,117,123,131,136,139],"S^2xS^1"],
	[[1,27,112,122,123,136,139,165],"L(8,3)"],
	[[1,27,112,131,135,136,137],"S^2xS^1"],
	[[1,27,112,133,135,136,137,165],"L(5,1)"],
	[[2,3,16,108,124,136],"S^3"],
	[[2,3,16,112,131,135,136,137],"S^2xS^1"],
	[[2,3,16,113,131,132,136,140],"S^2xS^1"],
	[[2,3,19,47,72,99,160],"S^3"],
	[[2,3,24,42,72,99,127],"S^3"],
	[[2,5,16,29,41,124,136],"S^3"],
	[[2,5,24,35,51,104,125],"S^2xS^1"],
	[[2,7,16,29,41,52,61,136],"S^3"],
	[[2,7,24,36,52,53,105,136],"S^2~S^1"],
	[[2,10,16,29,41,52,60],"S^2xS^1"],
	[[2,11,16,29,41,51],"S^2xS^1"],
	[[2,11,24,35,104,124,125,136],"S^2xS^1"],
	[[2,11,24,35,104,141,157],"S^2~S^1"],
	[[2,12,16,29,40],"S^2xS^1"],[[2,13,16,28],"S^2xS^1"],
	[[2,14,15],"S^2xS^1"],[[3,4,17,18,34,63,148],"S^3"],
	[[3,4,29,98,108,124,136],"S^3"],
	[[3,4,29,108,117,124,131,136],"S^2xS^1"],
	[[3,4,29,108,122,124,136,165],"L(8,3)"],
	[[3,4,29,110,123,136,144],"B2"],
	[[3,4,29,112,117,135,136,137],"B2"],
	[[3,4,29,112,123,136,139],"B2"],
	[[3,4,29,113,117,132,136,140],"B2"],
	[[3,4,29,115,118,136,145],"B2"],
	[[3,4,29,115,120,136,140],"B2"],
	[[3,4,32,53,116,118,136,145],"B2"],
	[[3,4,32,53,116,120,136,140],"B2"],
	[[3,4,32,56,76,84,112,120],"S^2~S^1"],
	[[3,4,32,58,76,84,95],"S^2~S^1"],
	[[3,4,33,51,66,76,78,95],"S^2~S^1"],
	[[3,4,33,54,110,122,136,144],"B2"],
	[[3,4,33,54,112,122,136,139],"B2"],
	[[3,4,33,56,76,79,112,120],"S^2~S^1"],
	[[3,4,33,58,76,79,95],"S^2~S^1"],
	[[3,4,35,51,62,76,81,95],"S^2~S^1"],
	[[3,4,36,53,76,86,95],"S^2~S^1"],
	[[3,4,36,54,76,81,95],"S^2~S^1"],
	[[3,4,38,53,76,86,110,118],"S^2~S^1"],
	[[3,4,38,54,76,81,110,118],"S^2~S^1"],
	[[3,4,39,55,91,109,134],"S^3"],
	[[3,6,28,66,73,113,133],"K^2xS^1"],
	[[3,7,29,42,53,115,136,145],"B2"],
	[[3,7,30,42,61,116,136,146],"B2"],
	[[3,7,32,42,112,136,160,162],"B2"],
	[[3,7,32,42,116,136,145],"B2"],
	[[3,7,33,42,76,110,155,163],"S^2~S^1"],
	[[3,7,36,42,76,86,95,118],"S^2~S^1"],
	[[3,7,38,42,76,83,93,109],"S^2~S^1"],
	[[3,7,38,42,76,86,110],"S^2~S^1"],
	[[3,9,28,83,88,92,95,136],"S^2~S^1"],
	[[3,9,29,44,55,115,136,140],"B2"],
	[[3,10,28,88,92,95,131],"S^2xS^1"],
	[[3,10,30,42,60,116,146],"T^3"],
	[[3,10,35,45,112,136,137,165],"L(5,1)"],
	[[3,10,35,48,70,73,113,167],"S^2~S^1"],
	[[3,11,28,92,95,124,136],"S^2~S^1"],
	[[3,12,28,95,98],"S^2xS^1"],
	[[3,12,28,95,117,131],"S^2xS^1"],
	[[3,12,28,95,122,165],"S^2xS^1"],
	[[3,12,29,40,108,124,136],"S^2~S^1"],
	[[3,13,28,108,124,136],"S^2~S^1"],
	[[3,13,28,112,131,135,136,137],"B4"],
	[[4,5,16,18,107,124,136],"S^3"],
	[[4,5,18,19,31,32,43,167],"S^3"],
	[[4,5,18,19,42,152,157],"S^2~S^1"],
	[[4,5,28,33,66,76,78,92],"S^2~S^1"],
	[[4,5,28,35,62,76,81,92],"S^2~S^1"],
	[[4,5,41,98,124,136],"S^3"],
	[[4,5,41,98,125,141,157],"S^2~S^1"],
	[[4,5,41,117,124,131,136],"S^2xS^1"],
	[[4,5,41,122,124,136,165],"L(8,3)"],
	[[4,5,50,67,98,136,144],"S^3"],
	[[4,5,50,67,117,131,136,144],"S^2xS^1"],
	[[4,5,50,67,122,136,144,165],"L(8,3)"],
	[[4,7,16,18,52,61,107,136],"S^3"],
	[[4,7,29,31,43,123,136,144],"B2"],
	[[4,7,41,52,61,98,136],"S^3"],
	[[4,9,29,33,45,123,136,139],"B2"],
	[[4,9,40,83,88,92,136],"S^2~S^1"],
	[[4,9,45,53,76,120,154,155],"S^2~S^1"],
	[[4,9,45,54,118,136,158,160],"B2"],
	[[4,9,45,54,122,136,139],"B2"],
	[[4,9,45,56,76,79,120],"S^2~S^1"],
	[[4,9,45,58,76,79,95,112],"S^2~S^1"],
	[[4,9,50,59,83,89,92,136],"S^2~S^1"],
	[[4,10,16,18,52,60,107],"S^2xS^1"],
	[[4,10,22,24,40,92,125],"S^2xS^1"],
	[[4,10,40,88,92,131],"S^2xS^1"],
	[[4,10,41,52,60,98],"S^2xS^1"],
	[[4,10,42,52,91,94,98],"S^2xS^1"],
	[[4,10,50,59,89,92,131],"S^2xS^1"],
	[[4,11,16,18,51,107],"S^2xS^1"],
	[[4,11,23,24,42,52,118,136],"S^2~S^1"],
	[[4,11,23,25,40,118,136],"S^2~S^1"],
	[[4,11,40,92,124,136],"S^2~S^1"],
	[[4,11,40,92,125,141,157],"B2"],
	[[4,11,41,51,98],"S^2xS^1"],
	[[4,11,41,51,117,131],"S^2xS^1"],
	[[4,11,41,51,122,165],"S^2xS^1"],
	[[4,12,40,98],"S^2xS^1"],[[4,12,40,117,131],"S^2xS^1"],
	[[4,12,40,122,148,151,152],"T^3"],
	[[4,12,40,122,165],"S^2xS^1"],
	[[5,6,19,20,43,49,58,167],"S^3"],
	[[5,6,19,20,57,80,158,161],"S^3/Q8"],
	[[5,6,31,32,43,50,59,167],"S^3"],
	[[5,6,42,50,59,152,157],"S^2~S^1"],
	[[5,6,52,98,117,124,136],"S^3"],
	[[5,6,52,124,131,136],"S^2xS^1"],
	[[5,6,52,124,133,136,165],"L(5,1)"],
	[[5,6,59,75,124,136,141],"S^2xS^1"],
	[[5,6,59,75,125,157],"S^2~S^1"],
	[[5,9,40,50,67,84,89,136],"S^2~S^1"],
	[[5,9,51,83,88,136],"S^2~S^1"],
	[[5,9,59,67,84,88,136],"S^2~S^1"],
	[[5,10,22,24,51,125],"S^2xS^1"],
	[[5,10,51,88,98,117],"S^2xS^1"],
	[[5,10,51,88,131],"S^2xS^1"],
	[[5,10,51,88,133,165],"S^2xS^1"],
	[[5,10,52,60,124,136],"S^2~S^1"],
	[[5,11,51,124,136],"S^2~S^1"],
	[[5,11,51,125,141,157],"B2"],
	[[6,7,16,20,41,107,136,144],"S^3"],
	[[6,7,18,20,98,136,144],"S^3"],
	[[6,7,18,20,105,107,123,136],"S^3"],
	[[6,7,18,20,121,138,154,161],"S^3/Q8"],
	[[6,7,30,32,109,116,136,145],"B2"],
	[[6,7,30,38,76,83,93],"S^2~S^1"],
	[[6,7,30,38,76,86,109,110],"S^2~S^1"],
	[[6,7,43,49,58,59,67,167],"S^3"],
	[[6,7,43,49,66,132,136,158],"L(5,1)"],
	[[6,7,61,98,117,136],"S^3"],
	[[6,7,61,109,116,136,146],"B2"],
	[[6,7,61,131,136],"S^2xS^1"],
	[[6,7,61,133,136,165],"L(5,1)"],
	[[6,7,66,81,132,136,165],"L(5,1)"],
	[[6,9,60,83,136],"S^2~S^1"],
	[[6,10,60,98,117],"S^2xS^1"],
	[[6,10,60,109,116,146],"T^3"],[[6,10,60,131],"S^2xS^1"],
	[[6,10,60,133,165],"S^2xS^1"],
	[[7,8,69,110,112,147,154,167],"(S^2xS^1)#7"],
	[[7,8,69,110,115,144,147,167],"(S^2xS^1)#7"],
	[[7,8,69,112,115,139,147,167],"(S^2xS^1)#7"],
	[[7,8,69,118,120,147,163,167],"(S^2xS^1)#7"],
	[[7,8,69,118,123,145,147,167],"(S^2xS^1)#7"],
	[[7,8,69,120,123,140,147,167],"(S^2xS^1)#7"],
	[[7,8,69,136],"S^2xS^1"],
	[[7,8,69,139,144,147,154,167],"(S^2xS^1)#7"],
	[[7,8,69,140,145,147,163,167],"(S^2xS^1)#7"],
	[[7,9,68,110,112,147,154,167],"(S^2~S^1)#7"],
	[[7,9,68,110,115,144,147,167],"(S^2~S^1)#7"],
	[[7,9,68,112,115,139,147,167],"(S^2~S^1)#7"],
	[[7,9,68,118,120,147,163,167],"(S^2~S^1)#7"],
	[[7,9,68,136],"S^2~S^1"],
	[[15,17,39,51,60,68,76,95],"S^2~S^1"],
	[[15,18,28,50,60,68,76,92],"S^2~S^1"],
	[[15,18,39,60,68,76,93],"S^2~S^1"],
	[[15,18,39,66,148,165],"S^2xS^1"],
	[[15,19,28,40,59,68,76,88],"S^2~S^1"],
	[[15,19,28,50,68,76,89],"S^2~S^1"],
	[[15,19,39,68,76,89,92,93],"S^2~S^1"],
	[[15,20,28,40,51,67,76,83],"S^2~S^1"],
	[[15,20,28,40,59,76,84],"S^2~S^1"],
	[[15,20,28,50,76,84,88,89],"S^2~S^1"],
	[[15,21,28,40,51,67,76,78],"S^2~S^1"],
	[[15,22,28,48,70,75,167],"S^2~S^1"],
	[[15,22,39,88,95,109,125],"S^2xS^1"],
	[[15,24,39,95,109,131],"S^2xS^1"],
	[[15,26,39,98],"S^2xS^1"],
	[[15,26,39,117,131],"S^2xS^1"],
	[[15,26,39,122,165],"S^2xS^1"],
	[[16,18,50,67,101,123,167],"S^2~S^1"],
	[[16,19,50,52,59,107,131],"S^2xS^1"],
	[[16,19,50,75,104,146,148],"S^2xS^1"],
	[[16,19,50,75,107,141],"S^2xS^1"],
	[[16,22,45,107,125,128,155],"S^2xS^1"],
	[[16,22,50,91,107,125],"S^2xS^1"],
	[[16,24,29,37,40,106],"S^2xS^1"],
	[[16,24,50,88,91,107,131],"S^2xS^1"],
	[[16,25,28,38,107],"S^2xS^1"],
	[[16,25,29,37,42,52,106],"S^2xS^1"],
	[[16,25,42,73,107,155,165],"S^2xS^1"],
	[[16,25,45,89,107,160,165],"S^2xS^1"],
	[[16,25,48,99,108,167],"S^2~S^1"],
	[[16,26,30,38,41,107],"S^2xS^1"],
	[[16,26,41,60,68,76,93,107],"SFS[T^2:(1,1)]"],
	[[16,27,40,51,60,68,76,97],"S^2~S^1"],
	[[16,27,48,97,167],"S^2~S^1"],
	[[16,27,50,101,110,167],"S^2~S^1"],
	[[17,24,40,48,109,131,167],"SFS[T^2:(1,1)]"],
	[[17,24,40,48,113,149,157,167],"S^2~S^1"],
	[[17,25,42,48,52,109,131,167],"SFS[T^2:(1,1)]"],
	[[17,26,51,60,68,76,95,98],"SFS[T^2:(1,1)]"],
	[[17,27,29,35,94,151,165],"S^2xS^1"],
	[[17,27,29,39,98],"S^2xS^1"],
	[[17,27,29,39,117,131],"S^2xS^1"],
	[[17,27,29,39,122,165],"S^2xS^1"],
	[[17,27,40,48,96,167],"S^2~S^1"],
	[[17,27,51,60,68,76,96],"S^2~S^1"],
	[[17,27,52,75,104,107,148],"S^2xS^1"],
	[[17,27,59,98,117,146],"S^2xS^1"],
	[[17,27,59,131,146],"S^2xS^1"],
	[[17,27,59,133,146,165],"S^2xS^1"],
	[[17,27,59,135,151,160,165],"S^2xS^1"],
	[[18,20,43,50,67,100,101,167],"S^2~S^1"],
	[[18,24,41,50,88,91,117],"K^2xS^1"],
	[[18,25,28,38,41,98],"S^2xS^1"],
	[[18,25,28,38,41,117,131],"S^2xS^1"],
	[[18,25,28,38,41,122,165],"S^2xS^1"],
	[[18,25,41,50,92,94],"T^3"],
	[[18,25,42,50,60,91,92],"T^3"],
	[[19,23,73,92,120,158,165,167],"S^3"],
	[[20,21,43,48,85,132,136,147],"(S^2~S^1)#7"],
	[[20,23,43,48,83,85,136,147],"(S^2~S^1)#7"],
	[[21,22,69,70,136],"S^2~S^1"],
	[[21,23,43,48,78,85,136,147],"(S^2~S^1)#7"],
	[[21,23,60,62,70,136],"S^2~S^1"],
	[[28,33,40,51,66,76,78],"S^2~S^1"],
	[[28,33,40,58,76,79],"S^2~S^1"],
	[[33,36,76,79,101,105,124,147],"(S^2~S^1)#7"],
	[[43,48,76,85],"S^2~S^1"]];
	
	dc[19]:=[[[1,16,30,43,55,66,76,85],"S^3"],
	[[1,16,30,43,55,71],"S^3"],
	[[1,16,30,48,66,73,76,88],"S^2~S^1"],
	[[1,16,30,48,71,73,85,88],"S^3"],
	[[1,16,30,49,66,73,80],"S^2~S^1"],
	[[1,16,30,49,79,82],"S^2~S^1"],
	[[1,16,30,52,204],"S^3"],[[1,16,34,46,53,204],"S^3"],
	[[1,16,34,51,53,89,201],"S^3"],
	[[1,16,34,55,64,71,85,100],"S^3"],
	[[1,16,34,57,64,175,188],"S^3"],
	[[1,16,34,91,189,200,204],"S^3"],
	[[1,16,35,55,64,71,95],"S^3"],
	[[1,16,35,58,64,203],"S^2~S^1"],
	[[1,16,35,68,74,193],"S^2~S^1"],
	[[1,16,35,71,74,94],"S^3"],
	[[1,16,35,97,194],"S^2~S^1"],
	[[1,16,36,48,49,66,74,80],"S^2~S^1"],
	[[1,16,36,59,64,195,197,200],"S^3"],
	[[1,16,36,66,74,76,88],"S^2~S^1"],
	[[1,16,36,69,74,200,203,204],"S^3"],
	[[1,16,36,71,74,85,88],"S^3"],
	[[1,16,36,76,83,87],"S^2~S^1"],
	[[1,16,37,57,64,87,175,197],"S^3"],
	[[1,16,37,79,83],"S^2~S^1"],
	[[1,16,39,46,53,89,190],"S^3"],
	[[1,16,41,182,190,191,193,195],"RP^2xS^1"],
	[[1,16,41,185,194,196],"S^2~S^1"],
	[[1,17,45,84,120,173,197],"S^3"],
	[[1,17,48,103,118,155,168],"S^2~S^1"],
	[[1,17,48,103,124,155,156,185],"S^2~S^1"],
	[[1,17,49,70,75,119,171,195],"S^3"],
	[[1,17,49,107,119,162,171],"S^3"],
	[[1,17,49,107,124,186],"S^3"],
	[[1,17,50,99,103,118,161,168],"S^2~S^1"],
	[[1,17,54,117,164,184],"S^3"],
	[[1,17,54,118,148,155,168,185],"S^2~S^1"],
	[[1,17,54,124,148,155,156],"S^2~S^1"],
	[[1,17,54,124,149],"S^3"],[[1,18,57,92,126,153],"S^3"],
	[[1,18,65,126,182,184,192,197],"S^2~S^1"],
	[[1,18,65,126,182,186,189,198],"S^2~S^1"],
	[[1,18,65,126,184,186,189,204],"S^3"],
	[[1,18,65,126,186,187,192],"S^2~S^1"],
	[[1,18,65,126,192,197,198,204],"S^3"],
	[[1,18,65,130,154,176,187,192],"S^2~S^1"],
	[[1,19,36,42,69,176,200],"S^3"],
	[[1,19,36,42,73,101,176,201],"S^3"],
	[[1,19,40,42,69,101,176,197],"S^3"],
	[[1,19,44,49,107,114,186],"S^3"],
	[[1,19,44,54,114,149],"S^3"],
	[[1,19,44,54,135,144,152,159],"S^2~S^1"],
	[[1,19,48,54,67,85,139,148],"S^3"],
	[[1,19,48,54,73,85,88,108],"S^2~S^1"],
	[[1,19,49,54,70,114,150],"S^3"],
	[[1,19,49,54,70,117,119,138],"S^3"],
	[[1,19,50,54,69,85,90,108],"S^2~S^1"],
	[[1,19,50,54,71,138,146,204],"S^3"],
	[[1,19,68,98,139,185],"S^3"],
	[[1,19,72,77,81,98,138,184],"S^3"],
	[[1,19,72,95,98,139,184],"S^3"],
	[[1,19,75,127,129,135,152,167],"S^2~S^1"],
	[[1,19,75,135,152,159,172],"S^2~S^1"],
	[[1,19,75,136,182,193],"S^2~S^1"],
	[[1,19,75,138,164,171,184,195],"S^3"],
	[[1,19,75,143,164,182,198],"S^2~S^1"],
	[[1,19,75,143,164,184,204],"S^3"],
	[[1,19,75,143,164,187,189,192],"S^2~S^1"],
	[[1,20,33,34,51,54,89,201],"S^3"],
	[[1,20,33,39,46,54,89,190],"S^3"],
	[[1,20,45,54,185,188,194,196],"SFS[RP^2:(2,1)(3,1)]"],
	[[1,20,56,65,117,119,152,166],"S^2~S^1"],
	[[1,20,56,65,118,120,144,161],"S^2~S^1"],
	[[1,20,56,65,119,121,144,154],"S^2~S^1"],
	[[1,20,56,65,120,122,146,152],"S^2~S^1"],
	[[1,20,56,65,144,152,159],"S^2~S^1"],
	[[1,20,58,65,77,146,154,200],"S^3"],
	[[1,20,63,65,82,118,120,147],"S^3"],
	[[1,20,77,98,150,170,189,202],"S^3"],
	[[1,20,83,91,92,146,189],"S^2~S^1"],
	[[1,20,84,115,120,130,192],"S^3"],
	[[1,20,84,117,119,166,169],"S^2~S^1"],
	[[1,20,84,120,122,146,169],"S^2~S^1"],
	[[1,20,84,128,130,147],"S^3"],
	[[1,20,84,144,159,169],"S^2~S^1"],
	[[1,21,34,35,58,65,203],"S^2~S^1"],
	[[1,21,34,35,98,154,189,203],"S^2~S^1"],
	[[1,21,35,38,58,65,88,189],"S^2~S^1"],
	[[1,21,35,38,88,98,154],"S^2~S^1"],
	[[1,21,46,52,71,75,99,101],"S^2~S^1"],
	[[1,21,46,53,64,65,204],"S^3"],
	[[1,21,67,75,118,120,161],"S^2~S^1"],
	[[1,21,67,75,119,121,154],"S^2~S^1"],
	[[1,21,67,75,152,159],"S^2~S^1"],
	[[1,21,73,75,90,119,121,155],"S^2~S^1"],
	[[1,21,92,126,129,180],"S^3"],
	[[1,21,92,129,131,155],"S^2~S^1"],
	[[1,21,92,152,165],"S^2~S^1"],[[1,21,92,153,175],"S^3"],
	[[1,21,92,153,194,203,204],"Sigma(2,3,7)"],
	[[1,22,58,61,80,84,163,167],"S^3"],
	[[1,22,58,62,81,84,163,167],"S^2~S^1"],
	[[1,22,58,64,69,75,102,197],"S^2~S^1"],
	[[1,22,58,64,74,75,203],"S^2~S^1"],
	[[1,22,58,65,157,163,192,198],"S^3"],
	[[1,22,60,61,80,84,93],"S^3"],
	[[1,22,60,62,81,84,93],"S^2~S^1"],
	[[1,22,66,71,84,93],"S^3"],
	[[1,22,66,75,79,94,200],"S^3"],
	[[1,22,68,75,193],"S^2~S^1"],
	[[1,22,70,72,81,84,93],"S^3"],
	[[1,22,74,75,97,194],"S^2~S^1"],
	[[1,22,77,84,159],"S^2~S^1"],
	[[1,22,79,84,93,200],"S^3"],[[1,22,98,160],"S^2~S^1"],
	[[1,23,36,37,79,84],"S^2~S^1"],
	[[1,23,36,42,141,166,180,197],"S^2~S^1"],
	[[1,23,69,75,161,165,203,204],"Sigma(2,3,7)"],
	[[1,23,76,84,87],"S^2~S^1"],
	[[1,23,78,84,193,194,203,204],"Sigma(2,3,7)"],
	[[1,23,86,92],"S^2~S^1"],
	[[1,24,49,54,117,119,184],"S^3"],
	[[1,24,70,75,164,171,195],"S^3"],
	[[1,24,73,75,105,127,129,189],"S^3"],
	[[1,27,32,42,69,101,141,201],"S^3"],
	[[1,27,32,42,73,137,166,201],"S^3"],
	[[1,29,125,135,138,150,152,166],"S^2~S^1"],
	[[1,29,125,135,139,144,157,161],"S^2~S^1"],
	[[1,29,125,135,140,150,156],"S^2~S^1"],
	[[1,29,125,137,140,148,166,201],"Sigma(2,3,7)"],
	[[1,29,125,137,142,144,159,170],"S^2~S^1"],
	[[1,29,125,139,142,144,163],"S^2~S^1"],
	[[1,29,125,139,142,150,154,156],"S^2~S^1"],
	[[1,29,125,148,155,156],"S^2~S^1"],
	[[1,29,125,149],"S^3"],
	[[1,29,126,154,173,176,187,197],"S^2~S^1"],
	[[1,29,126,158,180],"S^3"],
	[[1,29,127,135,143,152,159,172],"S^2~S^1"],
	[[1,29,127,136,143,182,193],"S^2~S^1"],
	[[1,29,127,164,182,198],"S^2~S^1"],
	[[1,29,127,164,187,189,192],"S^2~S^1"],
	[[1,29,129,135,143,146,148,157],"S^2~S^1"],
	[[1,29,129,135,143,152,167],"S^2~S^1"],
	[[1,29,129,152,158,165],"S^2~S^1"],
	[[1,29,130,147,151],"S^3"],
	[[1,29,130,160,164],"S^2~S^1"],
	[[1,29,131,155,158],"S^2~S^1"],
	[[1,29,134,182,184,192,197],"S^2~S^1"],
	[[1,29,134,186,187,192],"S^2~S^1"],
	[[1,29,134,191,201,202,203],"RP^2xS^1"],
	[[2,3,17,125,148,155,156],"S^2~S^1"],
	[[2,3,17,126,158,180],"S^3"],
	[[2,3,17,129,152,158,165],"S^2~S^1"],
	[[2,3,17,130,147,151],"S^3"],
	[[2,3,17,130,160,164],"S^2~S^1"],
	[[2,3,17,131,152,153,168],"S^2~S^1"],
	[[2,3,17,131,155,158],"S^2~S^1"],
	[[2,3,23,45,78,115,145],"S^3"],
	[[2,3,23,45,78,118,126,181],"S^3"],
	[[2,3,23,48,115,145,180,181],"S^3"],
	[[2,3,23,48,118,126,180],"S^3"],
	[[2,3,23,48,118,131,155],"S^2~S^1"],
	[[2,3,23,50,99,118,131,161],"S^2~S^1"],
	[[2,3,24,49,119,130,160],"S^2~S^1"],
	[[2,3,24,49,119,132,162],"S^3"],
	[[2,3,25,50,120,131,152,153],"S^2~S^1"],
	[[2,5,17,31,49,70,150],"S^3"],
	[[2,5,17,36,49,59,70,151],"S^3"],
	[[2,5,24,31,70,119,150,164],"S^3"],
	[[2,6,17,32,48,78,158,181],"S^3"],
	[[2,6,23,32,45,48,118,180],"S^3"],
	[[2,6,23,32,78,115,126,145],"S^3"],
	[[2,6,23,32,78,118,181],"S^3"],
	[[2,7,17,38,51,80,89,164],"S^3"],
	[[2,10,17,31,44,56,67,76],"S^2~S^1"],
	[[2,10,17,31,49,59,147,150],"S^3"],
	[[2,10,17,36,49,147,151],"S^3"],
	[[2,10,17,36,49,160,164],"S^2~S^1"],
	[[2,10,17,36,51,104,151,154],"S^3"],
	[[2,10,17,38,49,99,154,164],"S^2~S^1"],
	[[2,10,24,36,119,160],"S^2~S^1"],
	[[2,10,24,36,122,169,189],"S^2~S^1"],
	[[2,10,24,38,99,119,154],"S^2~S^1"],
	[[2,10,24,39,100,122,169,193],"S^2~S^1"],
	[[2,11,17,31,44,56,66],"S^2~S^1"],
	[[2,11,17,37,48,99,158,161],"S^2~S^1"],
	[[2,11,17,37,50,152,153,168],"S^2~S^1"],
	[[2,11,17,37,50,155,158],"S^2~S^1"],
	[[2,11,23,37,48,50,118,155],"S^2~S^1"],
	[[2,11,23,37,99,118,161],"S^2~S^1"],
	[[2,11,25,37,120,152,153],"S^2~S^1"],
	[[2,11,25,37,120,155,158,168],"S^2~S^1"],
	[[2,11,25,39,104,120,146,152],"S^2~S^1"],
	[[2,12,17,31,44,55],"S^2~S^1"],
	[[2,12,24,38,104,119,144,154],"S^2~S^1"],
	[[2,12,26,38,121,144,145,159],"S^2~S^1"],
	[[2,13,17,31,43],"S^2~S^1"],
	[[2,13,25,39,80,82,120,167],"S^3"],
	[[2,14,17,30],"S^2~S^1"],[[2,15,16],"S^2~S^1"],
	[[3,4,31,125,135,144,152,159],"S^2~S^1"],
	[[3,4,35,55,74,94,111],"S^2~S^1"],
	[[3,4,35,62,77,81,129,139],"S^2~S^1"],
	[[3,4,35,62,95,129,138],"S^2~S^1"],
	[[3,4,35,64,95,111],"S^2~S^1"],
	[[3,4,38,55,68,97,111],"S^2~S^1"],
	[[3,4,39,58,97,111],"S^2~S^1"],
	[[3,4,41,58,93,106,127,137],"S^2~S^1"],
	[[3,4,41,58,97,127,136],"S^2~S^1"],
	[[3,4,42,57,92,126,153],"S^3"],
	[[3,5,18,23,32,45,78,145],"S^3"],
	[[3,6,30,67,83,90,131,154],"B2"],
	[[3,6,30,73,83,129,152,165],"B2"],
	[[3,6,30,73,83,131,155],"B2"],
	[[3,6,36,48,74,83,131,155],"B2"],
	[[3,7,30,56,63,91,132,161],"S^2~S^1"],
	[[3,7,41,45,57,58,97,127],"S^2~S^1"],
	[[3,7,42,45,92,126,136,153],"S^3"],
	[[3,10,24,25,38,48,129,194],"S^2~S^1"],
	[[3,10,31,45,57,76,133,173],"B2"],
	[[3,10,32,45,67,76,134,174],"B2"],
	[[3,10,34,45,76,134,173],"B2"],
	[[3,11,32,45,66,134,174],"B2"],
	[[3,11,34,45,66,67,134,173],"B2"],
	[[3,11,35,46,87,94,131,202],"S^3"],
	[[3,12,30,80,82,89,91,132],"S^2~S^1"],
	[[3,12,30,80,82,111,141],"S^2~S^1"],
	[[3,12,35,50,95,129,138,141],"S^2~S^1"],
	[[3,12,36,48,88,89,125,151],"B2"],
	[[3,13,31,43,130,160,164],"B2"],
	[[3,13,37,43,57,87,133,197],"S^2~S^1"],
	[[3,14,30,125,148,155,156],"B2"],
	[[3,14,30,130,160,164],"B2"],
	[[4,5,19,24,49,117,119,184],"S^3"],
	[[4,5,30,35,74,94,108],"S^2~S^1"],
	[[4,5,30,38,68,97,108],"S^2~S^1"],
	[[4,5,30,40,68,97,137,145],"S^2~S^1"],
	[[4,5,35,36,48,73,94,108],"S^2~S^1"],
	[[4,5,44,135,144,152,159],"S^2~S^1"],
	[[4,5,45,75,84,136,145,193],"S^2~S^1"],
	[[4,5,49,70,117,119,138],"S^3"],
	[[4,5,50,68,90,91,97,108],"S^2~S^1"],
	[[4,5,54,75,135,152,159,172],"S^2~S^1"],
	[[4,5,54,75,136,182,193],"S^2~S^1"],
	[[4,6,19,21,45,57,182,193],"S^2~S^1"],
	[[4,10,17,19,56,67,76,124],"S^2~S^1"],
	[[4,10,45,57,76,139,194,196],"B2"],
	[[4,11,17,19,56,66,124],"S^2~S^1"],
	[[4,12,17,19,55,124],"S^2~S^1"],
	[[4,12,43,80,82,141],"S^2~S^1"],
	[[5,6,18,23,78,126,145],"S^3"],
	[[5,6,20,21,34,35,58,203],"S^2~S^1"],
	[[5,6,56,144,152,159],"S^2~S^1"],
	[[5,6,65,83,91,92,146,189],"S^2~S^1"],
	[[5,6,65,84,144,159,169],"S^2~S^1"],
	[[5,7,45,54,57,75,182,193],"S^2~S^1"],
	[[5,7,57,75,84,145,193],"S^2~S^1"],
	[[5,7,64,75,84,91,145,197],"S^2~S^1"],
	[[6,7,19,21,136,182,193],"S^2~S^1"],
	[[6,7,34,35,58,65,75,203],"S^2~S^1"],
	[[6,7,67,152,159],"S^2~S^1"],
	[[6,10,67,76,126,134,174],"B2"],
	[[6,11,66,126,134,174],"B2"],
	[[6,11,66,128,134,152,171],"B2"],
	[[7,10,76,127,133,172,194,203],"B2"],
	[[16,20,34,57,64,84,145,193],"B2"],
	[[16,20,34,84,91,145,197],"B2"],
	[[16,20,39,57,65,76,102,104],"T^3"],
	[[16,21,30,50,69,75,90,99],"B2"],
	[[16,21,30,54,90,150,154,172],"S^2~S^1"],
	[[16,21,36,57,65,102,189,193],"T^3"],
	[[16,21,38,69,75,91,99],"B2"],
	[[16,21,39,57,65,100,102],"T^3"],
	[[19,25,75,110,136,172],"T^3"],
	[[19,26,31,39,57,67,143,172],"B2"],
	[[19,27,45,54,66,107,108],"T^3"],
	[[19,27,66,84,107,108,145,182],"T^3"],
	[[20,24,45,54,107,169,182],"T^3"],
	[[20,24,84,107,145,169],"T^3"]];
	
	dc[20]:=[[[1,17,32,46,59,71,82,92,101],"S^3"],
	[[1,17,32,46,59,77],"S^3"],
	[[1,17,32,46,65,76,78],"S^3"],
	[[1,17,32,46,65,86,88],"S^2~S^1"],
	[[1,17,32,51,77,79,101,104],"S^3"],
	[[1,17,32,52,64,65,76,79],"S^3"],
	[[1,17,32,52,71,79,82,96],"S^2~S^1"],
	[[1,17,32,52,77,79,92,96,101],"S^3"],
	[[1,17,32,52,82,89,95],"S^2~S^1"],
	[[1,17,32,53,86,89],"S^2~S^1"],
	[[1,17,34,75,118,205,222,234],"S^2~S^1"],
	[[1,17,34,80,210,229],"S^3"],
	[[1,17,36,59,69,77,92,101,116],"S^3"],
	[[1,17,36,61,69,218,224],"S^2~S^1"],
	[[1,17,36,99,213,216,222],"S^2~S^1"],
	[[1,17,36,99,218,233],"S^2~S^1"],
	[[1,17,37,59,69,77,101,111],"S^3"],
	[[1,17,37,65,69,76,78,101,111],"S^3"],
	[[1,17,37,65,69,86,88,101,111],"S^2~S^1"],
	[[1,17,37,66,69,87,88,101,111],"S^3"],
	[[1,17,37,73,80,203,229],"S^3"],
	[[1,17,37,77,80,101,110],"S^3"],
	[[1,17,37,107,203,229,230],"S^3"],
	[[1,17,38,71,77,90,101,103],"S^3"],
	[[1,17,38,77,80,101,104],"S^3"],
	[[1,17,39,52,53,86,90],"S^2~S^1"],
	[[1,17,39,82,90,95],"S^2~S^1"],
	[[1,17,39,94,99],"S^2~S^1"],[[1,17,44,211,229],"S^3"],
	[[1,17,44,213,218,223,233],"S^2~S^1"],
	[[1,17,44,216,222,223],"S^2~S^1"],
	[[1,18,47,72,100,141,193,200],"S^3"],
	[[1,18,47,81,141,180,200],"S^3"],
	[[1,18,54,96,100,137,175,179],"S^2~S^1"],
	[[1,18,58,141,163,180],"S^3"],
	[[1,19,70,143,212,217,218,235],"RP^2xS^1"],
	[[1,19,70,143,222,237,239,240],"RP^2xS^1"],
	[[1,20,34,45,211,229],"S^3"],
	[[1,20,34,45,213,218,223,233],"S^2~S^1"],
	[[1,20,34,45,216,218,219,234],"S^2~S^1"],
	[[1,20,34,45,216,222,223],"S^2~S^1"],
	[[1,20,39,45,75,205,216,222],"S^2~S^1"],
	[[1,20,47,49,100,160,195,212],"B2"],
	[[1,20,47,58,130,163,180],"S^3"],
	[[1,20,47,58,153,163,172,180,187],"S^2xS^1"],
	[[1,20,47,58,153,163,176,180],"L(3,1)"],
	[[1,20,47,58,156,170,194],"B2"],
	[[1,20,47,58,160,165,195],"B2"],
	[[1,20,47,58,161,163,180,243],"L(8,3)"],
	[[1,20,51,58,79,101,104,124],"S^2~S^1"],
	[[1,20,52,58,72,92,101,158,168],"S^3"],
	[[1,20,52,58,75,156,169,194],"B2"],
	[[1,20,52,58,79,92,96,101,124],"S^2~S^1"],
	[[1,20,54,58,74,101,106,124],"S^2~S^1"],
	[[1,20,54,58,75,92,98,101,124],"S^2~S^1"],
	[[1,20,59,66,101,108,111,157,213],"S^2~S^1"],
	[[1,20,72,92,101,119,158,215],"S^3"],
	[[1,20,72,100,130,193,200],"S^3"],
	[[1,20,72,100,139,141,162,193],"S^3"],
	[[1,20,72,100,161,193,200,243],"L(8,3)"],
	[[1,20,73,101,114,158,215],"S^3"],
	[[1,20,75,119,130,149,151,213],"S^3"],
	[[1,20,75,119,149,151,161,213,243],"L(8,3)"],
	[[1,20,78,97,100,156,212,237],"SFS[K^2/n2:(1,5)]"],
	[[1,20,78,97,100,156,216,233],"L(3,1)"],
	[[1,20,81,130,144,146,195],"S^3"],
	[[1,20,81,130,180,200],"S^3"],
	[[1,20,81,144,146,153,172,187,195],"S^2xS^1"],
	[[1,20,81,144,146,153,176,195],"L(3,1)"],
	[[1,20,81,144,146,156,242],"S^2~S^1"],
	[[1,20,81,144,146,161,195,243],"L(8,3)"],
	[[1,20,81,146,149,156,183,213],"S^2~S^1"],
	[[1,20,81,153,172,180,187,200],"S^2xS^1"],
	[[1,20,81,153,176,180,200],"L(3,1)"],
	[[1,20,81,154,203,211,229],"S^3"],
	[[1,20,81,154,203,216,222,223],"S^2~S^1"],
	[[1,20,81,161,180,200,243],"L(8,3)"],
	[[1,21,48,58,211,221,226,227,229],"SFS[RP^2:(2,1)(3,1)]"],
	[[1,21,51,58,71,79,101,104,121],"S^2~S^1"],
	[[1,21,51,58,87,101,104,215,221],"S^3"],
	[[1,21,54,58,71,74,101,106,121],"S^2~S^1"],
	[[1,21,60,70,130,153,163,180],"S^3"],
	[[1,21,60,70,133,135,172,187,194],"S^2xS^1"],
	[[1,21,60,70,133,135,176,194],"L(3,1)"],
	[[1,21,60,70,137,139,165,172,187],"S^2xS^1"],
	[[1,21,60,70,137,139,165,176],"L(3,1)"],
	[[1,21,60,70,153,156,170,194],"B2"],
	[[1,21,60,70,153,160,165,195],"B2"],
	[[1,21,60,70,153,161,163,180,243],"L(8,3)"],
	[[1,21,60,70,163,172,180,187],"S^2xS^1"],
	[[1,21,60,70,163,176,180],"L(3,1)"],
	[[1,21,60,70,165,172,187,194,240],"B4"],
	[[1,21,60,70,165,176,194,240],"SFS[RP^2:(3,1)(3,2)]"],
	[[1,21,64,70,85,165,178,195],"L(3,1)"],
	[[1,21,91,130,156,170,187,194,197],"S^2xS^1"],
	[[1,21,91,130,160,165,187,195,197],"S^2xS^1"],
	[[1,21,91,133,135,187,194,197],"S^2xS^1"],
	[[1,21,91,137,139,165,187,197],"S^2xS^1"],
	[[1,21,91,156,160,165,187,197,242],"B4"],
	[[1,21,91,163,180,187,197],"S^2xS^1"],
	[[1,21,91,164,221,226,227,229],"SFS[RP^2:(2,1)(3,1)]"],
	[[1,21,91,165,187,194,197,240],"B4"],
	[[1,21,91,165,191,195,221,230,243],"L(8,3)"],
	[[1,21,91,170,187,194,195,197,242],"B4"],
	[[1,21,91,170,191,221,225,226,229],"SFS[RP^2:(2,1)(3,1)]"],
	[[1,22,36,37,59,70,77,101,111],"S^3"],
	[[1,22,52,58,92,101,170,175,200],"S^3"],
	[[1,22,59,70,77,92,101,116],"S^3"],
	[[1,22,61,70,218,224],"S^2~S^1"],
	[[1,22,69,70,99,213,216,222],"S^2~S^1"],
	[[1,22,69,70,99,218,233],"S^2~S^1"],
	[[1,22,72,81,130,153,180],"S^3"],
	[[1,22,72,81,153,161,180,243],"L(8,3)"],
	[[1,22,72,81,172,180,187],"S^2xS^1"],
	[[1,22,72,81,176,180],"L(3,1)"],
	[[1,22,74,81,98,101,106,177,184],"S^3"],
	[[1,22,75,81,92,101,177,184],"S^3"],
	[[1,22,78,81,97,175,180],"L(3,1)"],
	[[1,22,100,130,153,193],"S^3"],
	[[1,22,100,130,162,179,195],"S^3"],
	[[1,22,100,137,139,160,162,179],"S^3"],
	[[1,22,100,139,141,179,195,200],"S^3"],
	[[1,22,100,153,161,193,243],"L(8,3)"],
	[[1,22,100,156,162,175,201,234],"S^2~S^1"],
	[[1,22,100,156,162,179,242],"S^2~S^1"],
	[[1,22,100,160,162,179,194,240],"S^2~S^1"],
	[[1,22,100,161,162,179,195,243],"L(8,3)"],
	[[1,22,100,165,169,177,228,234],"B2"],
	[[1,22,100,168,169,177,193],"B2"],
	[[1,22,100,172,187,193],"S^2xS^1"],
	[[1,22,100,173,224],"S^2~S^1"],
	[[1,22,100,176,193],"L(3,1)"],
	[[1,22,100,179,196,226],"S^2~S^1"],
	[[1,23,34,37,81,210,229],"S^3"],
	[[1,23,34,45,73,154,229],"S^3"],
	[[1,23,73,81,203,229],"S^3"],
	[[1,23,74,81,101,228,232],"S^3"],
	[[1,23,83,91,180,187],"S^2xS^1"],
	[[1,24,54,58,101,106,124,155,212],"S^2~S^1"],
	[[1,24,77,80,90,91,101,104],"S^3"],
	[[1,24,93,100,187],"S^2xS^1"],
	[[1,25,47,58,75,170,194,213],"B2"],
	[[1,25,52,58,169,173,177,215],"B2"],
	[[1,25,52,58,169,194,213],"B2"],
	[[1,25,64,70,173,221],"L(3,1)"],
	[[1,25,64,70,178,195,219],"L(3,1)"],
	[[1,25,75,81,144,146,213,242],"S^2~S^1"],
	[[1,25,75,81,146,149,183],"S^2~S^1"],
	[[1,25,94,100],"S^2~S^1"],
	[[1,27,51,58,101,104,167,215],"S^3"],
	[[1,27,52,58,92,96,101,167,215],"S^3"],
	[[1,31,130,142,156,170,194],"S^3"],
	[[1,31,130,146,153,179,193],"S^3"],
	[[1,31,130,146,162,195],"S^3"],
	[[1,31,131,133,154,156,196],"S^3"],
	[[1,31,133,135,142,194],"S^3"],
	[[1,31,133,135,151,170,196],"S^3"],
	[[1,31,134,137,147,175,187,196,230],"S^2xS^1"],
	[[1,31,142,153,156,170,172,187,194],"S^2xS^1"],
	[[1,31,142,153,156,170,176,194],"L(3,1)"],
	[[1,31,142,156,160,165,242],"S^2~S^1"],
	[[1,31,142,156,161,170,194,243],"L(8,3)"],
	[[1,31,142,163,180],"S^3"],
	[[1,31,142,165,194,240],"S^2~S^1"],
	[[1,31,142,169,237,239,240],"S^2~S^1"],
	[[1,31,143,179,209,212,217,235],"RP^2xS^1"],
	[[1,31,146,153,161,179,193,243],"L(8,3)"],
	[[1,31,146,153,162,172,187,195],"S^2xS^1"],
	[[1,31,146,153,162,176,195],"L(3,1)"],
	[[1,31,146,156,162,242],"S^2~S^1"],
	[[1,31,146,160,162,194,240],"S^2~S^1"],
	[[1,31,146,161,162,195,243],"L(8,3)"],
	[[1,31,146,172,179,187,193],"S^2xS^1"],
	[[1,31,146,173,179,224],"S^2~S^1"],
	[[1,31,146,176,179,193],"L(3,1)"],
	[[1,31,146,181,183,187,192,193],"S^2xS^1"],
	[[1,31,146,196,226],"S^2~S^1"],
	[[1,31,152,212,217,218,235],"RP^2xS^1"],
	[[1,31,152,214,215,218,224,225,243],"S^3/Q28"],
	[[1,31,152,215,217,218,224,228],"RP^2xS^1"],
	[[1,31,152,221,226,227],"S^2~S^1"],
	[[1,31,152,222,237,239,240],"RP^2xS^1"],
	[[2,3,18,146,172,179,187,193],"S^2xS^1"],
	[[2,3,18,146,173,179,224],"S^2~S^1"],
	[[2,3,18,146,176,179,193],"L(3,1)"],
	[[2,3,18,146,181,183,187,192,193],"S^2xS^1"],
	[[2,3,18,146,196,226],"S^2~S^1"],
	[[2,3,18,147,173,175,187,188,196],"S^2xS^1"],
	[[2,3,18,149,172,173,187,196],"S^2xS^1"],
	[[2,3,18,149,173,176,196],"L(3,1)"],
	[[2,3,18,149,193,196,224],"S^2~S^1"],
	[[2,3,18,152,221,226,227],"S^2~S^1"],
	[[2,3,21,48,131,221,226,227],"S^2~S^1"],
	[[2,3,22,54,96,137,146,175],"S^2~S^1"],
	[[2,3,27,54,137,146,226],"S^2~S^1"],
	[[2,3,27,54,137,149,172,173,187],"S^2xS^1"],
	[[2,3,27,54,137,149,173,176],"L(3,1)"],
	[[2,3,27,54,137,149,193,224],"S^2~S^1"],
	[[2,5,27,40,59,120,137,173],"S^2~S^1"],
	[[2,5,27,41,61,72,138,164,187],"S^2xS^1"],
	[[2,5,28,41,59,138,164,187],"S^2xS^1"],
	[[2,5,28,42,62,73,138,164,187],"S^2xS^1"],
	[[2,6,18,40,54,71,173,196],"S^2~S^1"],
	[[2,6,27,40,71,137,173],"S^2~S^1"],
	[[2,8,27,41,72,73,138,187],"S^2xS^1"],
	[[2,8,27,42,59,62,72,138,187],"S^2xS^1"],
	[[2,8,28,41,59,61,73,138,187],"S^2xS^1"],
	[[2,8,28,42,60,61,83,139,187],"S^2xS^1"],
	[[2,8,28,42,61,62,138,187],"S^2xS^1"],
	[[2,10,18,41,54,74,75,187,192],"S^2xS^1"],
	[[2,11,18,33,47,60,72,82],"S^2xS^1"],
	[[2,12,18,33,47,60,71],"S^2xS^1"],
	[[2,12,18,40,54,172,173,187,196],"S^2xS^1"],
	[[2,12,18,40,54,173,176,196],"L(3,1)"],
	[[2,12,18,40,54,193,196,224],"S^2~S^1"],
	[[2,12,27,40,137,172,173,187],"S^2xS^1"],
	[[2,12,27,40,137,173,176],"L(3,1)"],
	[[2,13,18,33,47,59],"S^2xS^1"],
	[[2,14,18,33,46],"S^2xS^1"],[[2,15,18,32],"S^2xS^1"],
	[[2,16,17],"S^2xS^1"],[[3,4,19,20,34,211,229],"S^3"],
	[[3,4,19,23,34,73,154,229],"S^3"],
	[[3,4,33,142,156,170,194],"B2"],
	[[3,4,33,142,160,165,195],"B2"],
	[[3,4,33,146,153,179,193],"B2"],
	[[3,4,33,146,162,195],"B2"],
	[[3,4,33,149,153,173,196],"B2"],
	[[3,4,33,151,156,196],"B2"],
	[[3,4,35,60,146,160,193,212],
    "SFS[D:(3,1)(3,1)]U/mSFS[D:(3,1)(3,1)],m=[-4,5|-3,4]"],
	[[3,4,36,61,152,156,196],"B2"],
	[[3,4,36,63,93,147,156,220,229],"(S^2xS^1)#6"],
	[[3,4,36,63,93,150,156,197,229],"(S^2xS^1)#6"],
	[[3,4,36,69,92,101,116,127],"S^2~S^1"],
	[[3,4,37,59,78,101,110,147,157],"S^2~S^1"],
	[[3,4,37,59,80,101,110,127],"S^2~S^1"],
	[[3,4,37,69,101,111,127],"S^2~S^1"],
	[[3,4,38,59,71,90,101,103,127],"S^2~S^1"],
	[[3,4,38,59,78,101,104,147,157],"S^2~S^1"],
	[[3,4,38,59,80,101,104,127],"S^2~S^1"],
	[[3,4,38,60,84,149,156,194,229],"(S^2xS^1)#6"],
	[[3,4,38,63,150,156,194,229],"(S^2xS^1)#6"],
	[[3,4,39,60,85,146,160,193],"L(3,1)"],
	[[3,4,39,64,146,159,193],"L(3,1)"],
	[[3,4,40,59,71,84,101,107,127],"S^2~S^1"],
	[[3,4,41,59,73,101,113,127],"S^2~S^1"],
	[[3,4,41,59,74,101,107,127],"S^2~S^1"],
	[[3,4,42,61,92,101,118,127],"S^2~S^1"],
	[[3,4,42,62,101,113,127],"S^2~S^1"],
	[[3,4,44,61,92,101,118,144,154],"S^2~S^1"],
	[[3,4,44,62,101,109,122,144,155],"S^2~S^1"],
	[[3,4,44,62,101,113,144,154],"S^2~S^1"],
	[[3,6,32,72,89,99,149,175],"K^2xS^1"],
	[[3,6,32,79,90,146,172,187,193],"B4"],
	[[3,6,32,79,90,146,173,224],"B2"],
	[[3,6,32,79,90,146,176,193],"SFS[RP^2:(3,1)(3,2)]"],
	[[3,6,32,80,83,106,147,175,243],"G2"],
	[[3,6,32,80,89,147,174,243],"G2"],
	[[3,6,33,47,60,146,179,193],"B2"],
	[[3,6,33,47,60,149,173,196],"B2"],
	[[3,6,38,47,84,151,173,194,229],"(S^2xS^1)#6"],
	[[3,6,38,51,79,89,147,174,243],"G2"],
	[[3,6,39,52,80,90,147,174,243],"G2"],
	[[3,6,40,54,71,143,205,218],"T^3"],
	[[3,6,40,54,71,146,173,226],"T^3"],
	[[3,7,19,23,34,48,61,73,229],"S^3"],
	[[3,7,22,23,37,48,143,210,229],"S^3"],
	[[3,7,32,72,79,90,148,183],"T^3"],
	[[3,7,32,88,98,127,158,182],"T^3"],
	[[3,7,32,90,98,148,182],"T^3"],
	[[3,7,36,48,152,154,156,196],"B2"],
	[[3,7,42,48,92,101,118,127,154],"S^2~S^1"],
	[[3,7,44,48,61,62,101,113,144],"S^2~S^1"],
	[[3,7,44,48,92,101,115,125,143],"S^2~S^1"],
	[[3,7,44,48,92,101,118,144],"S^2~S^1"],
	[[3,8,32,72,80,106,149,188,243],"G2"],
	[[3,8,32,99,106,149,189,243],"G2"],
	[[3,8,37,51,96,110,145,220,229],"(S^2xS^1)#6"],
	[[3,8,37,51,96,110,150,190,229],"(S^2xS^1)#6"],
	[[3,8,38,51,96,104,145,220,229],"(S^2~S^1)#6"],
	[[3,8,38,51,96,104,150,190,229],"(S^2~S^1)#6"],
	[[3,8,38,52,92,145,220,229],"(S^2~S^1)#6"],
	[[3,8,38,52,92,150,190,229],"(S^2~S^1)#6"],
	[[3,8,38,55,92,116,145,225,229],"(S^2~S^1)#6"],
	[[3,8,40,53,98,106,149,189,243],"G2"],
	[[3,8,41,54,99,107,149,189,243],"G2"],
	[[3,8,41,55,92,110,116,145,229],"(S^2~S^1)#6"],
	[[3,8,42,49,101,109,122,127,154],"S^2~S^1"],
	[[3,8,42,49,101,113,127,155],"S^2~S^1"],
	[[3,8,44,49,101,109,122,144],"S^2~S^1"],
	[[3,8,44,49,101,113,144,154,155],"S^2~S^1"],
	[[3,9,36,47,73,93,151,197,229],"(S^2xS^1)#6"],
	[[3,9,36,50,93,147,220,229],"(S^2xS^1)#6"],
	[[3,9,36,50,93,150,197,229],"(S^2xS^1)#6"],
	[[3,9,36,50,96,110,146,220,229],"(S^2xS^1)#6"],
	[[3,9,38,47,73,151,194,229],"(S^2xS^1)#6"],
	[[3,9,38,50,146,225,229,241],"(S^2xS^1)#6"],
	[[3,9,38,50,150,194,229],"(S^2xS^1)#6"],
	[[3,9,38,52,102,145,220,229],"(S^2xS^1)#6"],
	[[3,9,38,52,102,150,190,229],"(S^2xS^1)#6"],
	[[3,9,38,52,105,116,146,220,229],"(S^2xS^1)#6"],
	[[3,9,38,55,102,116,145,225,229],"(S^2xS^1)#6"],
	[[3,9,38,55,105,146,225,229],"(S^2xS^1)#6"],
	[[3,9,41,50,93,96,147,229,241],"(S^2xS^1)#6"],
	[[3,9,41,50,110,146,229,241],"(S^2xS^1)#6"],
	[[3,9,41,55,93,96,105,147,229],"(S^2xS^1)#6"],
	[[3,9,41,55,102,110,116,145,229],"(S^2xS^1)#6"],
	[[3,9,41,55,105,110,146,229],"(S^2xS^1)#6"],
	[[3,10,32,74,80,85,89,147,243],"G2"],
	[[3,10,32,74,80,117,147,237,243],"S^2xS^1"],
	[[3,10,32,85,89,113,147,230,243],"G2"],
	[[3,10,32,109,115,120,124,127,187],"S^2xS^1"],
	[[3,10,32,113,117,147,230,237,243],"S^2xS^1"],
	[[3,10,32,113,117,151,198,243],"S^2xS^1"],
	[[3,10,36,47,73,92,151,197,229],"(S^2~S^1)#6"],
	[[3,10,36,50,92,147,220,229],"(S^2~S^1)#6"],
	[[3,10,36,50,92,150,197,229],"(S^2~S^1)#6"],
	[[3,10,37,50,111,116,147,220,229],"(S^2~S^1)#6"],
	[[3,10,37,50,111,116,150,197,229],"(S^2~S^1)#6"],
	[[3,10,37,51,105,116,147,220,229],"(S^2xS^1)#6"],
	[[3,10,37,51,105,116,150,197,229],"(S^2xS^1)#6"],
	[[3,10,41,50,92,96,147,229,241],"(S^2~S^1)#6"],
	[[3,10,41,55,92,96,105,147,229],"(S^2~S^1)#6"],
	[[3,11,25,27,40,54,146,226],"S^2~S^1"],
	[[3,11,32,75,80,103,106,148,243],"G2"],
	[[3,11,33,48,61,82,151,201],"T^3"],
	[[3,11,34,48,72,82,152,202],"T^3"],
	[[3,11,36,48,82,152,201],"T^3"],
	[[3,12,32,95,99,104,106,149,243],"G2"],
	[[3,12,32,104,106,122,149,235,243],"S^2xS^1"],
	[[3,12,32,120,124,127,172,187],"S^2xS^1"],
	[[3,12,32,120,124,127,176],"S^2xS^1"],
	[[3,12,34,48,71,152,202],"T^3"],
	[[3,12,36,48,71,72,152,201],"T^3"],
	[[3,12,39,53,64,66,146,193],"L(3,1)"],
	[[3,12,41,50,103,146,193,229,241],"(S^2xS^1)#6"],
	[[3,12,41,55,103,105,146,193,229],"(S^2xS^1)#6"],
	[[3,13,36,50,92,110,116,150,229],"(S^2~S^1)#6"],
	[[3,13,37,50,110,111,150,229],"(S^2~S^1)#6"],
	[[3,13,37,51,105,110,150,229],"(S^2xS^1)#6"],
	[[3,13,38,50,104,111,150,229],"(S^2xS^1)#6"],
	[[3,13,38,51,104,105,150,229],"(S^2~S^1)#6"],
	[[3,13,38,52,92,96,105,150,229],"(S^2~S^1)#6"],
	[[3,13,40,54,120,142,173,196],"B2"],
	[[3,13,42,54,142,165,196],"B2"],
	[[3,13,42,54,150,229,230,244],"(S^2xS^1)#6"],
	[[3,14,32,111,113,151,243],"S^2xS^1"],
	[[3,14,32,127,130],"S^2xS^1"],
	[[3,14,32,127,153,172,187],"S^2xS^1"],
	[[3,14,32,127,153,176],"S^2xS^1"],
	[[3,14,32,127,158,182,185],"T^3"],
	[[3,14,32,127,161,243],"S^2xS^1"],
	[[3,15,32,146,172,179,187,193],"B4"],
	[[3,15,32,146,173,179,224],"B2"],
	[[3,15,32,146,176,179,193],"SFS[RP^2:(3,1)(3,2)]"],
	[[3,15,32,146,181,183,187,192,193],"B4"],
	[[3,15,32,146,196,226],"B2"],
	[[3,15,32,152,221,226,227],"B2"],
	[[4,5,32,37,80,101,110,124],"S^2~S^1"],
	[[4,5,32,38,71,90,101,103,124],"S^2~S^1"],
	[[4,5,32,38,80,101,104,124],"S^2~S^1"],
	[[4,5,32,40,71,84,101,107,124],"S^2~S^1"],
	[[4,5,32,41,73,101,113,124],"S^2~S^1"],
	[[4,5,32,41,74,101,107,124],"S^2~S^1"],
	[[4,5,32,43,73,101,113,155,164],"S^2~S^1"],
	[[4,5,32,43,74,101,107,155,164],"S^2~S^1"],
	[[4,5,34,45,48,91,164,229],"S^3"],
	[[4,5,34,45,58,211,229],"S^3"],
	[[4,5,47,142,146,153,179,193],"B2"],
	[[4,5,47,142,146,162,195],"B2"],
	[[4,5,47,142,149,153,173,196],"B2"],
	[[4,5,47,142,151,156,196],"B2"],
	[[4,5,50,73,156,171,196,229],"(S^2xS^1)#6"],
	[[4,5,50,75,102,156,164,229,231],"(S^2xS^1)#6"],
	[[4,5,50,75,102,156,171,192,229],"(S^2xS^1)#6"],
	[[4,5,51,79,101,104,124],"S^2~S^1"],
	[[4,5,52,72,92,101,158,168],"S^3"],
	[[4,5,52,73,101,114,119,158,168],"S^3"],
	[[4,5,52,79,92,96,101,124],"S^2~S^1"],
	[[4,5,54,74,101,106,124],"S^2~S^1"],
	[[4,5,54,75,92,98,101,124],"S^2~S^1"],
	[[4,5,58,72,92,101,119,158,215],"S^3"],
	[[4,5,58,72,100,130,193,200],"S^3"],
	[[4,5,58,72,100,161,193,200,243],"L(8,3)"],
	[[4,5,58,73,101,114,158,215],"S^3"],
	[[4,6,48,70,81,154,218,224],"S^2~S^1"],
	[[4,7,35,44,49,62,101,113,154],"S^2~S^1"],
	[[4,7,46,90,98,127,148,182],"T^3"],
	[[4,7,47,60,81,100,130,193],"S^3"],
	[[4,7,47,60,81,100,161,193,243],"L(8,3)"],
	[[4,7,53,64,82,154,221,223],"S^2xS^1"],
	[[4,7,53,64,82,159,183],"S^2xS^1"],
	[[4,7,57,64,82,122,154,221],"G2"],
	[[4,8,32,41,61,101,113,124,164],"S^2~S^1"],
	[[4,8,32,43,61,101,109,122,154],"S^2~S^1"],
	[[4,8,32,43,61,101,113,155],"S^2~S^1"],
	[[4,8,34,45,48,61,73,91,229],"S^3"],
	[[4,8,35,44,48,61,101,113,155],"S^2~S^1"],
	[[4,8,36,37,57,69,155,243],"S^2xS^1"],
	[[4,8,58,61,101,114,158,164,215],"S^3"],
	[[4,9,54,69,107,110,156,229],"(S^2xS^1)#6"],
	[[4,10,46,109,115,120,124,187],"S^2xS^1"],
	[[4,10,52,59,78,101,104,157],"S^2~S^1"],
	[[4,10,52,59,80,101,104,127,147],"S^2~S^1"],
	[[4,10,54,66,74,75,157,243],"S^2xS^1"],
	[[4,10,58,70,109,115,121,124,187],"S^2xS^1"],
	[[4,11,18,20,60,72,82,141],"S^2xS^1"],
	[[4,11,25,27,46,120,124,173],"S^2~S^1"],
	[[4,11,33,35,49,82,162,200],"K^2xS^1"],
	[[4,11,46,116,121,158,217,228],"T^3"],
	[[4,11,47,60,72,82,130],"S^2xS^1"],
	[[4,11,47,60,72,82,161,243],"S^2xS^1"],
	[[4,11,47,61,72,119,123,130],"S^2xS^1"],
	[[4,11,47,61,72,119,123,161,243],"S^2xS^1"],
	[[4,11,48,61,82,158,211,215,229],"S^2~S^1"],
	[[4,11,53,64,146,148,159,193],"L(3,1)"],
	[[4,12,18,20,60,71,141],"S^2xS^1"],
	[[4,12,26,28,46,124,164,187],"S^2xS^1"],
	[[4,12,33,40,54,153,173,196],"B2"],
	[[4,12,40,41,51,60,84,160,243],"S^2xS^1"],
	[[4,12,46,120,124,172,187],"S^2xS^1"],
	[[4,12,46,120,124,173,193,224],"T^3"],
	[[4,12,46,120,124,176],"S^2xS^1"],
	[[4,12,47,60,71,130],"S^2xS^1"],
	[[4,12,47,60,71,161,243],"S^2xS^1"],
	[[4,12,48,60,123,126,130],"S^2xS^1"],
	[[4,12,48,60,123,126,161,243],"S^2xS^1"],
	[[4,12,48,61,82,119,126,130],"S^2xS^1"],
	[[4,12,48,61,82,119,126,161,243],"S^2xS^1"],
	[[4,12,51,60,71,74,160,243],"S^2xS^1"],
	[[4,12,52,64,95,96,159],"S^2xS^1"],
	[[4,12,54,69,103,107,156,193,229],"(S^2xS^1)#6"],
	[[4,12,58,64,119,121,124,221],"S^2xS^1"],
	[[4,12,58,70,121,124,172,187],"S^2xS^1"],
	[[4,12,58,70,121,124,176],"S^2xS^1"],
	[[4,13,18,20,59,141],"S^2xS^1"],
	[[4,13,25,29,46,115,154,173],"S^2~S^1"],
	[[4,13,47,59,130],"S^2xS^1"],
	[[4,13,47,59,153,172,187],"S^2xS^1"],
	[[4,13,47,59,153,173,193,224],"T^3"],
	[[4,13,47,59,153,176],"S^2xS^1"],
	[[4,13,47,59,161,243],"S^2xS^1"],
	[[4,13,51,59,74,160,243],"S^2xS^1"],
	[[4,14,46,130],"S^2xS^1"],
	[[4,14,46,153,172,187],"S^2xS^1"],
	[[4,14,46,153,173,193,224],"T^3"],
	[[4,14,46,153,176],"S^2xS^1"],
	[[4,14,46,154,185,215,217],"T^3"],
	[[4,14,46,158,182,185],"T^3"],
	[[4,14,46,161,243],"S^2xS^1"],
	[[5,6,21,22,61,218,224],"S^2~S^1"],
	[[5,6,21,22,69,99,218,233],"S^2~S^1"],
	[[5,6,21,25,64,173,221],"L(3,1)"],
	[[5,6,32,38,46,90,101,103,120],"S^2~S^1"],
	[[5,6,32,40,46,84,101,107,120],"S^2~S^1"],
	[[5,6,33,38,84,170,173,196,229],"(S^2xS^1)#6"],
	[[5,6,60,142,146,179,193],"B2"],
	[[5,6,60,142,149,173,196],"B2"],
	[[5,6,60,163,176,182,211,217,229],"SFS[S^2:(3,1)(3,1)(4,-3)]"],
	[[5,6,63,84,171,173,196,229],"(S^2xS^1)#6"],
	[[5,6,69,90,170,178,227,233],"L(3,1)"],
	[[5,6,69,90,171,179,225,229,241],"(S^2xS^1)#6"],
	[[5,7,34,45,61,91,154,164,229],"S^3"],
	[[5,8,33,38,75,92,170,192,229],"(S^2~S^1)#6"],
	[[5,8,37,41,59,97,164,227,243],"S^2xS^1"],
	[[5,8,59,97,104,164,225,227,243],"S^2xS^1"],
	[[5,8,59,97,104,170,191,243],"S^2xS^1"],
	[[5,8,62,74,98,112,164,229,231],"(S^2xS^1)#6"],
	[[5,8,62,74,98,112,171,192,229],"(S^2xS^1)#6"],
	[[5,8,63,74,98,106,164,229,231],"(S^2~S^1)#6"],
	[[5,8,63,74,98,106,171,192,229],"(S^2~S^1)#6"],
	[[5,8,63,75,92,164,229,231],"(S^2~S^1)#6"],
	[[5,8,63,75,92,171,192,229],"(S^2~S^1)#6"],
	[[5,8,63,80,92,118,164,229,244],"(S^2~S^1)#6"],
	[[5,8,68,80,92,112,118,164,229],"(S^2~S^1)#6"],
	[[5,9,33,38,73,170,196,229],"(S^2xS^1)#6"],
	[[5,9,33,38,75,102,170,192,229],"(S^2xS^1)#6"],
	[[5,9,61,73,93,166,229,231],"(S^2xS^1)#6"],
	[[5,9,61,73,93,171,199,229],"(S^2xS^1)#6"],
	[[5,9,63,73,171,196,229],"(S^2xS^1)#6"],
	[[5,9,63,75,102,164,229,231],"(S^2xS^1)#6"],
	[[5,9,63,75,102,171,192,229],"(S^2xS^1)#6"],
	[[5,9,63,80,102,118,164,229,244],"(S^2xS^1)#6"],
	[[5,9,68,73,93,98,166,229,230],"(S^2xS^1)#6"],
	[[5,9,68,80,93,98,107,166,229],"(S^2xS^1)#6"],
	[[5,9,68,80,102,112,118,164,229],"(S^2xS^1)#6"],
	[[5,10,46,58,81,109,116,121,187],"S^2xS^1"],
	[[5,10,59,109,115,120,187],"S^2xS^1"],
	[[5,10,59,111,115,133,134,187],"S^2xS^1"],
	[[5,10,61,73,92,166,229,231],"(S^2~S^1)#6"],
	[[5,10,61,73,92,171,199,229],"(S^2~S^1)#6"],
	[[5,10,62,73,113,118,166,229,231],"(S^2~S^1)#6"],
	[[5,10,62,73,113,118,171,199,229],"(S^2~S^1)#6"],
	[[5,10,62,74,107,118,166,229,231],"(S^2xS^1)#6"],
	[[5,10,62,74,107,118,171,199,229],"(S^2xS^1)#6"],
	[[5,10,64,79,101,104,124,157],"S^2~S^1"],
	[[5,10,68,73,92,98,166,229,230],"(S^2~S^1)#6"],
	[[5,10,68,80,92,98,107,166,229],"(S^2~S^1)#6"],
	[[5,10,69,80,110,115,133,134,187],"S^2xS^1"],
	[[5,10,70,81,109,116,120,187],"S^2xS^1"],
	[[5,10,70,81,111,116,133,134,187],"S^2xS^1"],
	[[5,11,25,27,59,120,173],"S^2~S^1"],
	[[5,11,59,115,120,182,211,217,229],"S^2~S^1"],
	[[5,12,26,27,61,72,164,187],"S^2xS^1"],
	[[5,12,26,28,59,164,187],"S^2xS^1"],
	[[5,12,59,120,130,153],"S^2xS^1"],
	[[5,12,59,120,153,161,243],"S^2xS^1"],
	[[5,12,59,120,172,187],"S^2xS^1"],
	[[5,12,59,120,173,193,224],"T^3"],
	[[5,12,59,120,176],"S^2xS^1"],
	[[5,12,60,71,163,182,211,217,229],"S^2~S^1"],
	[[5,13,59,163,182,211,217,229],"S^2~S^1"],
	[[5,13,59,169,212,217,234,239],"B2"],
	[[5,13,59,169,237,239,240],"B2"],
	[[6,7,22,23,34,37,210,229],"S^3"],
	[[6,7,22,23,74,101,228,232],"S^3"],
	[[6,7,34,44,92,101,115,125],"S^2~S^1"],
	[[6,7,61,70,81,218,224],"S^2~S^1"],
	[[6,7,69,70,81,99,218,233],"S^2~S^1"],
	[[6,7,72,176,182,211,217,229],"SFS[S^2:(3,1)(3,1)(4,-3)]"],
	[[6,7,78,97,175,182,211,217,229],"SFS[S^2:(3,1)(3,1)(4,-3)]"],
	[[6,7,79,98,176,183,211,217,229],"SFS[S^2:(3,1)(3,1)(4,-3)]"],
	[[6,7,81,100,130,153,193],"S^3"],
	[[6,7,81,100,153,161,193,243],"L(8,3)"],
	[[6,7,81,100,172,187,193],"S^2xS^1"],
	[[6,7,81,100,173,224],"S^2~S^1"],
	[[6,7,81,100,176,193],"L(3,1)"],
	[[6,10,32,38,52,80,89,174,243],"G2"],
	[[6,10,32,39,51,79,90,174,243],"G2"],
	[[6,10,38,39,80,90,174,243],"G2"],
	[[6,10,51,52,79,89,174,243],"G2"],
	[[6,10,59,70,91,110,116,187],"S^2xS^1"],
	[[6,10,69,70,80,91,111,116,187],"S^2xS^1"],
	[[6,10,71,109,115,187],"S^2xS^1"],
	[[6,10,81,91,110,115,187],"S^2xS^1"],
	[[6,11,18,22,47,82,141,200],"S^2xS^1"],
	[[6,11,20,21,116,120,130,200],"S^2xS^1"],
	[[6,11,20,21,116,120,161,200,243],"S^2xS^1"],
	[[6,11,20,22,82,130,200],"S^2xS^1"],
	[[6,11,20,22,82,139,141,162],"S^2xS^1"],
	[[6,11,20,22,82,161,200,243],"S^2xS^1"],
	[[6,11,52,54,71,175,211,217,229],"S^2~S^1"],
	[[6,11,71,115,182,211,217,229],"S^2~S^1"],
	[[6,11,72,82,130,153],"S^2xS^1"],
	[[6,11,72,82,153,161,243],"S^2xS^1"],
	[[6,11,72,82,172,187],"S^2xS^1"],
	[[6,11,72,82,176],"S^2xS^1"],
	[[6,11,78,82,97,175],"S^2xS^1"],
	[[6,12,71,130,153],"S^2xS^1"],
	[[6,12,71,153,161,243],"S^2xS^1"],
	[[6,12,71,172,187],"S^2xS^1"],
	[[6,12,71,173,193,224],"T^3"],[[6,12,71,176],"S^2xS^1"],
	[[7,8,32,35,57,101,109,121],"S^2~S^1"],
	[[7,8,32,43,48,101,109,122],"S^2~S^1"],
	[[7,8,32,43,48,101,113,154,155],"S^2~S^1"],
	[[7,8,34,37,81,91,210,229],"S^3"],
	[[7,8,34,44,71,101,109,125],"S^2~S^1"],
	[[7,8,35,42,101,113,127,144,155],"S^2~S^1"],
	[[7,8,35,44,101,109,122],"S^2~S^1"],
	[[7,8,35,44,101,113,154,155],"S^2~S^1"],
	[[7,10,82,109,187],"S^2xS^1"],
	[[7,11,82,182,211,217,229],"S^2~S^1"],
	[[8,9,93,145,147,220,229],"(S^2xS^1)#6"],
	[[8,9,93,145,150,197,229],"(S^2xS^1)#6"],
	[[8,9,93,147,150,190,229],"(S^2xS^1)#6"],
	[[8,9,93,187],"S^2xS^1"],
	[[8,9,93,190,197,220,229],"(S^2xS^1)#6"],
	[[8,10,92,145,147,220,229],"(S^2~S^1)#6"],
	[[8,10,92,145,150,197,229],"(S^2~S^1)#6"],
	[[8,10,92,147,150,190,229],"(S^2~S^1)#6"],
	[[8,10,92,164,166,229,231],"(S^2~S^1)#6"],
	[[8,10,92,187],"S^2xS^1"],
	[[17,20,32,58,71,82,92,101,124],"S^2~S^1"],
	[[17,20,35,80,91,155,243],"S^2xS^1"],
	[[17,21,32,46,70,82,92,101,120],"S^2~S^1"],
	[[17,21,32,58,82,92,101,121],"S^2~S^1"],
	[[17,21,42,61,70,82,118,120],"T^3"],
	[[17,21,44,48,58,221,226,227],"B2"],
	[[17,22,32,46,59,81,92,101,115],"S^2~S^1"],
	[[17,22,32,46,70,92,101,116],"S^2~S^1"],
	[[17,22,32,54,75,81,98,115],"B2"],
	[[17,22,32,58,92,101,116,120,121],"S^2~S^1"],
	[[17,22,37,49,57,62,70,243],"S^2xS^1"],
	[[17,22,37,62,70,99,219,243],"G2"],
	[[17,22,37,99,108,174,243],"G2"],
	[[17,22,42,59,61,81,115,118],"T^3"],
	[[17,22,42,61,70,116,118],"T^3"],
	[[17,23,32,46,59,71,91,101,109],"S^2~S^1"],
	[[17,23,32,46,59,81,101,110],"S^2~S^1"],
	[[17,23,32,46,70,101,110,115,116],"S^2~S^1"],
	[[17,24,32,46,59,71,91,101,103],"S^2~S^1"],
	[[17,24,35,74,80,91,212,243],"S^2xS^1"],
	[[17,24,35,91,113,212,230,243],"S^2xS^1"],
	[[17,24,39,74,80,85,91,243],"G2"],
	[[17,24,39,85,91,113,230,243],"G2"],
	[[17,31,38,84,90,152,218,229],"S^2~S^1"],
	[[17,31,44,152,211,221,226,227,229],"S^2~S^1"],
	[[18,20,49,79,98,141,160,239],"S^2xS^1"],
	[[18,20,50,74,101,132,179,212,241],"(S^2xS^1)#4"],
	[[18,20,50,74,101,141,155,190,237],"(S^2xS^1)#4"],
	[[18,20,50,74,101,141,160,241],"(S^2xS^1)#4"],
	[[18,20,52,79,85,89,141,160],"G2"],
	[[18,20,52,79,117,141,160,237],"S^2xS^1"],
	[[18,20,53,79,95,98,141,160],"G2"],
	[[18,20,54,59,67,132,179,212],"S^2xS^1"],
	[[18,20,54,59,67,141,160],"S^2xS^1"],
	[[18,20,54,64,66,75,141,160],"S^2xS^1"],
	[[18,20,54,78,96,97,141,160],"S^2xS^1"],
	[[18,20,55,74,101,105,132,179,212],"(S^2xS^1)#4"],
	[[18,20,55,74,101,105,141,160],"(S^2xS^1)#4"],
	[[18,20,56,79,132,179,212,240],"S^2xS^1"],
	[[18,20,56,79,141,160,240],"S^2xS^1"],
	[[18,21,50,84,101,133,186,220],"(S^2xS^1)#6"],
	[[18,21,56,89,133,186,219],"G2"],
	[[18,21,58,60,70,141,172,187],"S^2xS^1"],
	[[18,21,58,60,70,141,176],"S^2xS^1"],
	[[18,21,58,91,141,187,197],"S^2xS^1"],
	[[18,22,54,96,139,160,162,175],"S^2xS^1"],
	[[18,23,47,81,83,91,141,187,200],"S^2xS^1"],
	[[18,23,47,81,108,141,187,188,200],"S^2xS^1"],
	[[18,23,50,138,176,179,181,229,241],"(S^2xS^1)#4"],
	[[18,23,51,101,134,166,171,172,183],"(S^2~S^1)#6"],
	[[18,23,51,101,134,172,183,199,231],"(S^2~S^1)#6"],
	[[18,23,51,101,136,166,171,183,189],"(S^2xS^1)#6"],
	[[18,23,51,101,136,183,189,199,231],"(S^2xS^1)#6"],
	[[18,23,51,101,140,166,171,183,198],"(S^2xS^1)#4"],
	[[18,23,51,101,140,183,198,199,231],"(S^2xS^1)#4"],
	[[18,23,53,101,109,136,166,171],"(S^2xS^1)#6"],
	[[18,23,53,101,109,136,199,231],"(S^2xS^1)#6"],
	[[18,23,55,101,109,120,132,155,171],"(S^2xS^1)#4"],
	[[18,23,55,101,109,120,136,171,174],"(S^2xS^1)#6"],
	[[18,23,55,101,109,120,138,171,172],"(S^2~S^1)#6"],
	[[18,23,55,105,138,176,179,181,229],"(S^2xS^1)#4"],
	[[18,23,58,83,91,141,163,187],"S^2xS^1"],
	[[18,23,58,108,141,163,187,188],"S^2xS^1"],
	[[18,24,47,81,114,141,181,187,200],"S^2xS^1"],
	[[18,24,48,83,101,136,201,233],"(S^2xS^1)#6"],
	[[18,24,50,101,132,155,179,241],"(S^2xS^1)#4"],
	[[18,24,50,101,132,160,179,190,237],"(S^2xS^1)#4"],
	[[18,24,50,101,133,164,181,199],"(S^2xS^1)#6"],
	[[18,24,50,101,133,166,172,179,183],"(S^2~S^1)#6"],
	[[18,24,50,101,133,166,181,192],"(S^2~S^1)#6"],
	[[18,24,50,101,136,174,179,241],"(S^2xS^1)#6"],
	[[18,24,50,101,136,199,233],"(S^2xS^1)#6"],
	[[18,24,50,101,138,172,179,241],"(S^2~S^1)#6"],
	[[18,24,50,101,138,181,183,192,241],"(S^2~S^1)#6"],
	[[18,24,50,101,141,155,160,212,241],"(S^2xS^1)#4"],
	[[18,24,50,101,141,190,212,237],"(S^2xS^1)#4"],
	[[18,24,50,101,141,190,216,233],"(S^2xS^1)#6"],
	[[18,24,53,101,103,136,166,171],"(S^2xS^1)#6"],
	[[18,24,53,101,103,136,199,231],"(S^2xS^1)#6"],
	[[18,24,53,101,105,120,136,166,179],"(S^2xS^1)#6"],
	[[18,24,55,101,103,120,132,155,171],"(S^2xS^1)#4"],
	[[18,24,55,101,103,120,136,171,174],"(S^2xS^1)#6"],
	[[18,24,55,101,103,120,138,171,172],"(S^2~S^1)#6"],
	[[18,24,55,101,105,132,155,179],"(S^2xS^1)#4"],
	[[18,24,55,101,105,136,174,179],"(S^2xS^1)#6"],
	[[18,24,55,101,105,138,172,179],"(S^2~S^1)#6"],
	[[18,24,55,101,105,138,181,183,192],"(S^2~S^1)#6"],
	[[18,24,55,101,105,141,155,160,212],"(S^2xS^1)#4"],
	[[18,24,58,114,141,163,181,187],"S^2xS^1"],
	[[18,26,47,81,109,114,141,187,200],"S^2xS^1"],
	[[18,26,50,101,109,138,183,192,241],"(S^2xS^1)#6"],
	[[18,26,50,101,111,120,134,164,199],"(S^2~S^1)#6"],
	[[18,26,50,101,111,120,134,166,192],"(S^2xS^1)#6"],
	[[18,26,50,103,138,176,179,229,241],"(S^2xS^1)#4"],
	[[18,26,51,101,105,120,134,164,199],"(S^2~S^1)#6"],
	[[18,26,51,101,105,120,134,166,192],"(S^2xS^1)#6"],
	[[18,26,55,101,105,109,138,183,192],"(S^2xS^1)#6"],
	[[18,26,55,103,105,138,176,179,229],"(S^2xS^1)#4"],
	[[18,26,55,120,138,171,176,229],"(S^2xS^1)#4"],
	[[18,26,58,109,114,141,163,187],"S^2xS^1"],
	[[18,26,58,123,141,164,187],"S^2xS^1"],
	[[18,27,33,42,47,59,139],"S^2xS^1"],
	[[18,27,33,42,49,59,72,140],"S^2xS^1"],
	[[18,28,33,42,49,61,140],"S^2xS^1"],
	[[18,28,33,43,46,140],"S^2xS^1"],
	[[18,28,35,42,49,60,61,141],"S^2xS^1"],
	[[18,28,35,43,46,60,141],"S^2xS^1"],
	[[18,28,46,60,89,141,216],"G2"],
	[[18,28,48,83,101,111,134,172,201],"(S^2~S^1)#6"],
	[[18,28,48,83,101,111,136,189,201],"(S^2xS^1)#6"],
	[[18,28,48,83,101,111,140,198,201],"(S^2xS^1)#4"],
	[[18,28,50,101,111,134,172,199],"(S^2~S^1)#6"],
	[[18,28,50,101,111,136,189,199],"(S^2xS^1)#6"],
	[[18,28,50,101,111,140,198,199],"(S^2xS^1)#4"],
	[[18,28,50,101,111,141,189,190,216],"(S^2xS^1)#6"],
	[[18,28,51,101,105,134,172,199],"(S^2~S^1)#6"],
	[[18,28,51,101,105,136,189,199],"(S^2xS^1)#6"],
	[[18,28,51,101,105,140,198,199],"(S^2xS^1)#4"],
	[[18,28,51,101,105,141,189,190,216],"(S^2xS^1)#6"],
	[[18,28,53,101,105,109,136,183,199],"(S^2xS^1)#6"],
	[[18,28,58,120,123,141,172,187],"S^2xS^1"],
	[[18,28,58,120,123,141,176],"S^2xS^1"],
	[[18,29,32,44,141],"S^2xS^1"],
	[[18,29,33,43,48,60,140],"S^2xS^1"],
	[[18,29,35,43,48,141],"S^2xS^1"],
	[[18,29,48,89,141,216],"G2"],
	[[18,29,49,98,141,217,239],"S^2xS^1"],
	[[18,29,52,85,89,141,217],"G2"],
	[[18,29,52,117,141,217,237],"S^2xS^1"],
	[[18,29,53,95,98,141,217],"G2"],
	[[18,29,53,121,141,223],"G2"],
	[[18,29,56,132,160,179,234,237],"S^2xS^1"],
	[[18,29,56,141,212,234,237],"S^2xS^1"],
	[[18,29,56,141,213,219,223],"G2"],
	[[18,29,56,141,213,235,239],"S^2xS^1"],
	[[18,29,56,141,216,233,234],"G2"],
	[[18,29,56,141,217,240],"S^2xS^1"],
	[[18,29,57,121,122,141],"S^2xS^1"],
	[[18,31,48,91,131,164,229],"S^2~S^1"],
	[[18,31,52,82,89,95,129],"B2"],
	[[18,31,53,86,89,129],"B2"],
	[[18,31,58,131,211,229],"S^2~S^1"],
	[[20,21,47,58,91,130,187,197],"S^2xS^1"],
	[[20,21,57,58,80,91,155,243],"S^2xS^1"],
	[[20,21,75,85,130,170,217,223],"G2"],
	[[20,21,75,85,161,170,217,223,243],"G2"],
	[[20,21,81,91,130,163,187,197,200],"S^2xS^1"],
	[[20,22,47,54,96,160,175,200],"K^2xS^1"],
	[[20,22,72,162,172,187,201,217,240],"S^2xS^1"],
	[[20,22,72,162,176,201,217,240],"S^2xS^1"],
	[[20,23,35,37,81,91,155,243],"S^2xS^1"],
	[[20,23,47,58,83,91,130,163,187],"S^2xS^1"],
	[[20,23,47,58,108,130,163,187,188],"S^2xS^1"],
	[[20,23,81,83,91,130,187,200],"S^2xS^1"],
	[[20,23,81,108,130,187,188,200],"S^2xS^1"],
	[[20,24,32,35,51,79,90,155,243],"S^2xS^1"],
	[[20,24,35,38,80,90,155,243],"S^2xS^1"],
	[[20,24,47,50,101,130,190,212,237],"(S^2xS^1)#4"],
	[[20,24,47,50,101,130,190,216,233],"(S^2xS^1)#6"],
	[[20,24,47,58,114,130,163,181,187],"S^2xS^1"],
	[[20,24,73,101,144,145,154,220,234],"(S^2~S^1)#6"],
	[[20,24,73,101,144,150,154,190,234],"(S^2~S^1)#6"],
	[[20,26,35,40,74,84,155,187],"G2"],
	[[20,26,47,58,109,114,130,163,187],"S^2xS^1"],
	[[20,26,47,58,123,130,164,187],"S^2xS^1"],
	[[20,26,74,84,86,155,187,213,237],"G2"],
	[[20,26,79,95,96,124,164,187,216],"S^2xS^1"],
	[[20,26,81,123,130,163,164,187,200],"S^2xS^1"],
	[[20,27,33,42,59,162,200],"K^2xS^1"],
	[[20,27,35,41,73,83,155,187],"G2"],
	[[20,27,35,42,59,62,83,155,187],"G2"],
	[[20,27,35,43,73,83,124,164,187],"G2"],
	[[20,27,47,58,115,119,153,163],"T^3"],
	[[20,27,47,58,126,154,163],"T^3"],
	[[20,27,73,83,89,124,164,187,216],"S^2xS^1"],
	[[20,27,81,115,119,153,200],"T^3"],
	[[20,27,81,126,154,200],"T^3"],
	[[20,28,47,58,120,123,153],"K^2xS^1"],
	[[20,28,74,101,105,138,141,160,199],"(S^2xS^1)#4"],
	[[20,28,75,94,97,159,213,233],"S^2xS^1"],
	[[20,28,81,120,123,153,163,200],"K^2xS^1"],
	[[20,29,32,44,47,130],"S^2xS^1"],
	[[20,29,32,44,47,153,172,187],"S^2xS^1"],
	[[20,29,32,44,47,153,176],"S^2xS^1"],
	[[20,29,32,44,47,161,243],"S^2xS^1"],
	[[20,29,32,44,51,74,160,243],"S^2xS^1"],
	[[20,29,35,43,47,48,130],"S^2xS^1"],
	[[20,29,35,43,47,48,161,243],"S^2xS^1"],
	[[20,29,35,43,48,51,74,160,243],"S^2xS^1"],
	[[20,29,47,48,89,130,216],"G2"],
	[[20,29,47,48,89,161,216,243],"G2"],
	[[20,29,47,49,98,130,217,239],"S^2xS^1"],
	[[20,29,47,49,98,161,217,239,243],"S^2xS^1"],
	[[20,29,47,52,85,89,130,217],"G2"],
	[[20,29,47,52,85,89,161,217,243],"G2"],
	[[20,29,47,52,117,130,217,237],"S^2xS^1"],
	[[20,29,47,52,117,161,217,237,243],"S^2xS^1"],
	[[20,29,47,53,95,98,130,217],"G2"],
	[[20,29,47,53,95,98,161,217,243],"G2"],
	[[20,29,47,53,121,130,223],"G2"],
	[[20,29,47,53,121,161,223,243],"G2"],
	[[20,29,47,54,124,170,217],"T^3"],
	[[20,29,47,56,130,212,234,237],"S^2xS^1"],
	[[20,29,47,56,130,213,219,223],"G2"],
	[[20,29,47,56,130,213,235,239],"S^2xS^1"],
	[[20,29,47,56,130,216,233,234],"G2"],
	[[20,29,47,56,130,217,240],"S^2xS^1"],
	[[20,29,47,56,144,151,154],"T^3"],
	[[20,29,47,56,153,172,187,217,240],"S^2xS^1"],
	[[20,29,47,56,153,176,217,240],"S^2xS^1"],
	[[20,29,47,56,161,212,234,237,243],"S^2xS^1"],
	[[20,29,47,56,161,213,219,223,243],"G2"],
	[[20,29,47,56,161,213,235,239,243],"S^2xS^1"],
	[[20,29,47,56,161,216,233,234,243],"G2"],
	[[20,29,47,56,161,217,240,243],"S^2xS^1"],
	[[20,29,47,57,121,122,130],"S^2xS^1"],
	[[20,29,47,57,121,122,161,243],"S^2xS^1"],
	[[20,29,47,58,124,126],"T^3"],
	[[20,29,48,51,74,89,160,216,243],"G2"],
	[[20,29,51,53,74,121,160,223,243],"G2"],
	[[20,29,51,57,74,121,122,160,243],"S^2xS^1"],
	[[20,29,52,54,75,124,169,217],"T^3"],
	[[20,29,73,101,112,144,150,154],"(S^2~S^1)#6"],
	[[20,31,73,108,130,151,187,192,213],"S^2xS^1"],
	[[21,23,84,101,171,186,220],"(S^2xS^1)#6"],
	[[21,23,89,101,110,124,162,186,219],"(S^2xS^1)#6"],
	[[21,23,89,101,112,171,186,219],"(S^2xS^1)#6"],
	[[21,24,57,58,74,80,91,212,243],"S^2xS^1"],
	[[21,24,57,58,91,113,212,230,243],"S^2xS^1"],
	[[21,24,83,101,131,136,148,223],"(S^2xS^1)#6"],
	[[21,24,83,101,145,148,166],"(S^2xS^1)#6"],
	[[21,26,35,40,164,187,212,218],"G2"],
	[[21,26,35,41,121,124,155,187],"G2"],
	[[21,26,35,43,121,164,187],"G2"],
	[[21,26,36,39,82,164,187,233],"S^2xS^1"],
	[[21,26,36,41,82,117,124,155,187],"G2"],
	[[21,26,36,43,82,117,164,187],"G2"],
	[[21,26,39,40,85,164,187,218],"S^2xS^1"],
	[[21,26,48,53,164,187,223],"S^2xS^1"],
	[[21,26,48,57,122,164,187],"G2"],
	[[21,26,49,57,82,118,164,187],"G2"],
	[[21,26,82,94,164,187,218],"S^2xS^1"],
	[[21,26,82,99,118,164,187,219],"S^2xS^1"],
	[[21,26,86,164,187,222],"S^2xS^1"],
	[[21,26,86,169,197,229],"S^2~S^1"],
	[[21,26,89,121,164,187,216],"S^2xS^1"],
	[[21,26,90,95,99,164,187,218],"S^2xS^1"],
	[[21,26,90,122,164,187,218,235],"G2"],
	[[21,28,35,40,120,172,187,212,218],"G2"],
	[[21,28,35,40,120,176,212,218],"G2"],
	[[21,28,35,43,120,121,172,187],"G2"],
	[[21,28,35,43,120,121,176],"G2"],
	[[21,28,36,39,82,120,172,187,233],"S^2xS^1"],
	[[21,28,36,39,82,120,176,233],"S^2xS^1"],
	[[21,28,36,43,82,117,120,172,187],"G2"],
	[[21,28,36,43,82,117,120,176],"G2"],
	[[21,28,39,40,85,120,172,187,218],"S^2xS^1"],
	[[21,28,39,40,85,120,176,218],"S^2xS^1"],
	[[21,28,46,58,60,121,123],"T^3"],
	[[21,28,48,53,120,172,187,223],"S^2xS^1"],
	[[21,28,48,53,120,176,223],"S^2xS^1"],
	[[21,28,48,57,120,122,172,187],"G2"],
	[[21,28,48,57,120,122,176],"G2"],
	[[21,28,49,57,82,118,120,172,187],"G2"],
	[[21,28,49,57,82,118,120,176],"G2"],
	[[21,28,60,70,120,123],"T^3"],
	[[21,28,82,94,120,130,153,218],"S^2xS^1"],
	[[21,28,82,94,120,153,161,218,243],"S^2xS^1"],
	[[21,28,82,94,120,172,187,218],"S^2xS^1"],
	[[21,28,82,94,120,176,218],"S^2xS^1"],
	[[21,28,82,99,118,120,172,187,219],"S^2xS^1"],
	[[21,28,82,99,118,120,176,219],"S^2xS^1"],
	[[21,28,86,120,130,153,222],"S^2xS^1"],
	[[21,28,86,120,153,161,222,243],"S^2xS^1"],
	[[21,28,86,120,172,187,222],"S^2xS^1"],
	[[21,28,86,120,176,222],"S^2xS^1"],
	[[21,28,89,120,121,172,187,216],"S^2xS^1"],
	[[21,28,89,120,121,176,216],"S^2xS^1"],
	[[21,28,90,95,99,120,172,187,218],"S^2xS^1"],
	[[21,28,90,95,99,120,176,218],"S^2xS^1"],
	[[21,28,90,120,122,172,187,218,235],"G2"],
	[[21,28,90,120,122,176,218,235],"G2"],
	[[21,28,90,127,130,149,170,218],"S^2xS^1"],
	[[21,28,90,127,149,161,170,218,243],"S^2xS^1"],
	[[21,29,48,58,121,123],"T^3"],
	[[21,29,86,121,169,211,229],"S^2~S^1"],
	[[21,31,48,58,152,211,229],"S^2~S^1"],
	[[21,31,60,70,130,142,153],"S^2xS^1"],
	[[21,31,60,70,142,153,161,243],"S^2xS^1"],
	[[21,31,60,70,142,172,187],"S^2xS^1"],
	[[21,31,60,70,142,176],"S^2xS^1"],
	[[21,31,91,142,187,197],"S^2xS^1"],
	[[21,31,91,152,164,229],"S^2~S^1"],
	[[23,24,50,56,112,172,176,181,229],"(S^2~S^1)#9"],
	[[23,24,101,136,138,183,231,233,241],"(S^2xS^1)#6"],
	[[23,24,101,144,145,211,220,234],"(S^2~S^1)#6"],
	[[23,24,101,144,150,190,211,234],"(S^2~S^1)#6"],
	[[23,24,101,145,147,181,220],"(S^2~S^1)#6"],
	[[23,24,101,145,150,181,197],"(S^2xS^1)#6"],
	[[23,24,101,147,150,181,190],"(S^2~S^1)#6"],
	[[23,24,101,164,166,181,231],"(S^2xS^1)#6"],
	[[23,24,101,164,171,181,199],"(S^2xS^1)#6"],
	[[23,24,101,166,171,172,179,183],"(S^2~S^1)#6"],
	[[23,24,101,166,171,181,192],"(S^2~S^1)#6"],
	[[23,24,101,172,174,183,231,233,241],"(S^2~S^1)#6"],
	[[23,24,101,172,179,183,199,231],"(S^2~S^1)#6"],
	[[23,24,101,181,190,197,220],"(S^2xS^1)#6"],
	[[23,24,101,181,192,199,231],"(S^2~S^1)#6"],
	[[23,26,50,53,174,176,183,229,241],"(S^2xS^1)#4"],
	[[23,26,53,55,105,174,176,183,229],"(S^2xS^1)#4"],
	[[23,26,83,86,163,169,229],"S^2~S^1"],
	[[23,26,101,112,121,144,147,220],"(S^2xS^1)#6"],
	[[23,26,101,112,121,144,150,197],"(S^2~S^1)#6"],
	[[23,26,105,120,130,153,166,183,229],"(S^2xS^1)#4"],
	[[23,26,105,120,166,176,183,229],"(S^2xS^1)#4"],
	[[23,29,101,112,144,150,211],"(S^2~S^1)#6"],
	[[23,29,104,124,155,163,188,235,243],"S^2xS^1"],
	[[23,29,106,131,138,149,163,187,188],"S^2xS^1"],
	[[23,31,83,91,142,163,187],"S^2xS^1"],
	[[24,26,35,40,84,187,212],"G2"],
	[[24,26,35,43,84,121,187,218],"G2"],
	[[24,26,39,40,84,85,187],"S^2xS^1"],
	[[24,26,48,53,84,187,218,223],"S^2xS^1"],
	[[24,26,48,57,84,122,187,218],"G2"],
	[[24,26,50,53,101,164,166,233],"(S^2~S^1)#6"],
	[[24,26,50,53,101,174,179,192,241],"(S^2xS^1)#6"],
	[[24,26,50,53,101,192,199,233],"(S^2xS^1)#6"],
	[[24,26,50,55,101,120,164,174,233],"(S^2~S^1)#6"],
	[[24,26,50,56,103,112,172,176,229],"(S^2~S^1)#9"],
	[[24,26,53,55,101,105,174,179,192],"(S^2xS^1)#6"],
	[[24,26,82,84,94,187],"S^2xS^1"],
	[[24,26,84,86,187,212,213,237],"G2"],
	[[24,26,84,86,187,213,216,233],"S^2xS^1"],
	[[24,26,84,86,187,218,222],"S^2xS^1"],
	[[24,26,84,86,187,219,223,234],"S^2xS^1"],
	[[24,26,84,86,187,234,235,239],"G2"],
	[[24,26,84,89,121,187,216,218],"S^2xS^1"],
	[[24,26,84,90,95,99,187],"S^2xS^1"],
	[[24,26,84,90,122,187,235],"G2"],
	[[24,26,101,103,145,147,220],"(S^2xS^1)#6"],
	[[24,26,101,103,145,150,197],"(S^2~S^1)#6"],
	[[24,26,101,103,147,150,190],"(S^2xS^1)#6"],
	[[24,26,101,103,164,166,231],"(S^2~S^1)#6"],
	[[24,26,101,103,164,171,199],"(S^2~S^1)#6"],
	[[24,26,101,103,166,171,192],"(S^2xS^1)#6"],
	[[24,26,101,103,190,197,220],"(S^2~S^1)#6"],
	[[24,26,101,103,192,199,231],"(S^2xS^1)#6"],
	[[24,26,101,105,120,164,174,233,241],"(S^2~S^1)#6"],
	[[24,26,101,105,120,164,179,199],"(S^2~S^1)#6"],
	[[24,26,101,105,120,166,179,192],"(S^2xS^1)#6"],
	[[24,28,50,53,101,120,166,172,233],"(S^2~S^1)#6"],
	[[24,28,50,55,101,136,138,233],"(S^2xS^1)#6"],
	[[24,28,50,55,101,172,174,233],"(S^2~S^1)#6"],
	[[24,28,53,55,101,103,136,138,231],"(S^2xS^1)#6"],
	[[24,28,53,55,101,103,172,174,231],"(S^2~S^1)#6"],
	[[24,28,101,103,120,166,172,231],"(S^2~S^1)#6"],
	[[24,28,101,103,120,171,172,199],"(S^2~S^1)#6"],
	[[24,28,101,105,136,138,233,241],"(S^2xS^1)#6"],
	[[24,28,101,105,164,166,181,183],"(S^2xS^1)#6"],
	[[24,28,101,105,172,174,233,241],"(S^2~S^1)#6"],
	[[24,28,101,105,172,179,199],"(S^2~S^1)#6"],
	[[24,28,101,105,181,183,192,199],"(S^2~S^1)#6"],
	[[24,29,32,44,51,155,160,212,243],"S^2xS^1"],
	[[24,29,38,43,187,197,218,235],"G2"],
	[[24,29,60,62,89,101,136,138,234],"(S^2xS^1)#6"],
	[[24,29,60,62,89,101,172,174,234],"(S^2~S^1)#6"],
	[[24,29,101,103,121,145,150,211],"(S^2~S^1)#6"],
	[[24,29,101,103,121,190,211,220],"(S^2~S^1)#6"],
	[[24,31,84,91,152,218,229],"S^2~S^1"],
	[[26,28,50,53,101,111,164,166,189],"(S^2~S^1)#6"],
	[[26,28,50,53,101,111,189,192,199],"(S^2xS^1)#6"],
	[[26,28,50,55,103,174,176,229,233],"(S^2xS^1)#4"],
	[[26,28,51,53,101,105,164,166,189],"(S^2~S^1)#6"],
	[[26,28,51,53,101,105,189,192,199],"(S^2xS^1)#6"],
	[[26,28,53,55,174,176,229,231],"(S^2xS^1)#4"],
	[[26,28,84,86,111,187,189,213,216],"S^2xS^1"],
	[[26,28,103,105,130,153,179,199,229],"(S^2xS^1)#4"],
	[[26,28,103,105,174,176,229,233,241],"(S^2xS^1)#4"],
	[[26,28,103,105,176,179,199,229],"(S^2xS^1)#4"],
	[[26,28,120,130,153,166,229,231],"(S^2xS^1)#4"],
	[[26,28,120,130,153,171,199,229],"(S^2xS^1)#4"],
	[[26,28,120,166,176,229,231],"(S^2xS^1)#4"],
	[[26,28,120,171,176,199,229],"(S^2xS^1)#4"],
	[[26,29,82,85,98,197,223,229],"S^2~S^1"],
	[[26,29,86,89,197,223,229],"S^2~S^1"],
	[[26,29,101,103,112,147,150,234],"(S^2xS^1)#6"],
	[[26,29,101,103,112,197,220,234],"(S^2~S^1)#6"],
	[[32,35,57,82,92,101,121],"S^2~S^1"],
	[[32,36,57,92,101,117],"S^2~S^1"],
	[[33,38,62,101,138,140,166,171,239],"(S^2xS^1)#4"],
	[[33,38,62,101,138,140,199,231,239],"(S^2xS^1)#4"],
	[[33,38,63,155,161,192,229,239,244],"(S^2~S^1)#6"],
	[[33,38,63,191,192,229,238,239,244],"(S^2xS^1)#6"],
	[[33,38,68,112,155,161,192,229,239],"(S^2~S^1)#6"],
	[[33,38,68,112,191,192,229,238,239],"(S^2xS^1)#6"],
	[[33,39,64,164,171,191,229,233,241],"(S^2xS^1)#4"],
	[[33,39,64,191,192,229,231,233,241],"(S^2xS^1)#4"],
	[[33,40,67,120,142,149,153,160],"T^3"],
	[[33,43,62,101,112,138,140,199],"(S^2xS^1)#4"],
	[[33,43,63,101,106,138,140,199],"(S^2xS^1)#4"],
	[[33,43,64,117,164,171,191,229,241],"(S^2xS^1)#6"],
	[[33,43,64,117,191,192,229,231,241],"(S^2xS^1)#6"],
	[[33,43,68,155,161,164,171,229],"(S^2~S^1)#6"],
	[[33,43,68,155,161,192,229,231],"(S^2~S^1)#6"],
	[[33,43,68,164,171,191,229,238],"(S^2xS^1)#6"],
	[[33,43,68,191,192,229,231,238],"(S^2xS^1)#6"],
	[[35,44,48,49,92,101,118],"S^2~S^1"],
	[[35,44,64,67,82,97,122],"B2"],
	[[35,44,82,92,101,122],"S^2~S^1"],
	[[36,38,50,52,92,220,229],"(S^2~S^1)#6"],
	[[46,50,68,101,111],"S^2~S^1"],
	[[46,51,68,101,105],"S^2~S^1"],
	[[46,55,68,243],"S^2xS^1"],
	[[49,56,92,101,117],"S^2~S^1"],
	[[50,51,101,130,161,241],"(S^2xS^1)#4"],
	[[50,51,101,132,134,155,241],"(S^2xS^1)#4"],
	[[50,51,101,134,136,174,241],"(S^2xS^1)#6"],
	[[50,51,101,134,138,172,241],"(S^2~S^1)#6"],
	[[50,51,101,136,138,189,241],"(S^2xS^1)#6"],
	[[50,51,101,138,140,198,241],"(S^2xS^1)#4"],
	[[50,51,101,153,161,176,241],"(S^2xS^1)#4"],
	[[50,51,101,172,174,189,241],"(S^2~S^1)#6"],
	[[50,56,62,68,243],"S^2xS^1"],
	[[50,56,101,112],"S^2~S^1"],
	[[51,55,63,67,243],"S^2xS^1"],
	[[51,55,101,105,130,161],"(S^2xS^1)#4"],
	[[51,55,101,105,132,134,155],"(S^2xS^1)#4"],
	[[51,55,101,105,134,136,174],"(S^2xS^1)#6"],
	[[51,55,101,105,134,138,172],"(S^2~S^1)#6"],
	[[51,55,101,105,136,138,189],"(S^2xS^1)#6"],
	[[51,55,101,105,138,140,198],"(S^2xS^1)#4"],
	[[51,55,101,105,153,161,176],"(S^2xS^1)#4"],
	[[51,55,101,105,172,174,189],"(S^2~S^1)#6"],
	[[63,67,101,105],"S^2~S^1"],
	[[64,67,92,97,101],"S^2~S^1"]];
	
	dc[21]:=[[[1,18,34,49,63,76,87,97,107],"S^3"],
	[[1,18,34,49,69,87,93,101],"S^2~S^1"],
	[[1,18,34,49,70,92,93],"S^2~S^1"],
	[[1,18,34,49,72,279],"S^3"],
	[[1,18,34,54,79,84,275],"S^2~S^1"],
	[[1,18,34,54,120,276],"S^2~S^1"],
	[[1,18,34,55,76,84,87,97,111],"S^2~S^1"],
	[[1,18,34,55,87,94,97,110],"S^2~S^1"],
	[[1,18,34,56,69,70,92,94],"S^2~S^1"],
	[[1,18,34,56,87,94,101],"S^2~S^1"],
	[[1,18,34,56,100,104],"S^2~S^1"],
	[[1,18,34,60,273,276,277],"S^2~S^1"],
	[[1,18,36,78,114,235,243,248,260],"RP^3"],
	[[1,18,36,85,235,243,248,256],"RP^3"],
	[[1,18,37,90,127,237,260,276,279],"L(3,1)"],
	[[1,18,37,95,237,272,274,278],"S^2~S^1"],
	[[1,18,37,95,237,275,276,279],"L(3,1)"],
	[[1,18,38,52,61,273,276,277],"S^2~S^1"],
	[[1,18,38,58,61,102,268,273,276],"S^2~S^1"],
	[[1,18,38,65,68,127,248,273],"G3"],
	[[1,18,38,65,74,227,248],"S^3"],
	[[1,18,38,65,74,248,250],"S^3/Q8"],
	[[1,18,38,65,74,248,253],"G3"],
	[[1,18,38,68,70,101,127,247,273],"G3"],
	[[1,18,38,70,74,101,227,247],"S^3"],
	[[1,18,38,70,74,101,229,232,252],"S^3"],
	[[1,18,38,70,74,101,247,250],"S^3/Q8"],
	[[1,18,38,70,74,101,247,253],"G3"],
	[[1,18,38,71,74,102,248,254],"G3"],
	[[1,18,38,80,85,103,125,248,259],"G3"],
	[[1,18,38,105,244,271],"S^2~S^1"],
	[[1,18,38,105,248,263],"S^3/Q8"],
	[[1,18,38,105,249,264,274],"S^2~S^1"],
	[[1,18,39,54,58,111,121,252],"S^2~S^1"],
	[[1,18,39,58,61,111,246,252,268],"S^2~S^1"],
	[[1,18,39,70,74,92,93,107,125],"S^2~S^1"],
	[[1,18,39,78,85,227,243,248,256],"RP^3"],
	[[1,18,39,78,85,243,248,250,256],"S^3/Q8"],
	[[1,18,39,78,85,243,248,253,256],"SFS[S^2:(3,2)(3,2)(3,-1)]"],
	[[1,18,39,78,85,246,248,254,258],"(RP^2)#3xS^1"],
	[[1,18,39,78,85,247,248,252,259],"(RP^2)#3xS^1"],
	[[1,18,39,81,85,112,130,248,259],"G3"],
	[[1,18,39,114,227,239,241,247,248],"RP^3"],
	[[1,18,39,114,227,243,248,260],"RP^3"],
	[[1,18,39,114,239,241,247,248,250],"S^3/Q8"],
	[[1,18,39,114,239,241,247,248,253],"SFS[S^2:(3,2)(3,2)(3,-1)]"],
	[[1,18,39,114,243,248,250,260],"S^3/Q8"],
	[[1,18,39,114,243,248,253,260],"SFS[S^2:(3,2)(3,2)(3,-1)]"],
	[[1,18,39,114,251,266,276,279],"L(3,1)"],
	[[1,18,39,114,255,265,269,276,279],"L(3,1)"],
	[[1,18,40,79,85,275],"S^2~S^1"],
	[[1,18,40,84,85,120,276],"S^2~S^1"],
	[[1,18,40,89,95,256],"S^2~S^1"],
	[[1,18,40,121,257],"S^2~S^1"],
	[[1,18,41,55,56,69,70,92,95],"S^2~S^1"],
	[[1,18,41,55,56,87,95,101],"S^2~S^1"],
	[[1,18,41,80,85,258,260,266,279],"L(3,1)"],
	[[1,18,41,87,95,97,110],"S^2~S^1"],
	[[1,18,41,90,95,266,275,276,279],"L(3,1)"],
	[[1,18,41,97,105,109],"S^2~S^1"],
	[[1,18,41,99,105,243,248],"S^3/Q8"],
	[[1,18,42,69,74,238,241,254],"S^2~S^1"],
	[[1,18,42,100,105],"S^2~S^1"],
	[[1,18,43,70,74,227,245,247,272],"S^3"],
	[[1,18,43,70,74,245,247,250,272],"S^3/Q8"],
	[[1,18,43,70,74,245,247,253,272],"G3"],
	[[1,18,44,52,61,102,246,276,277],"S^2~S^1"],
	[[1,18,44,58,61,246,268,276],"S^2~S^1"],
	[[1,18,44,71,74,246,248,254,273],"G3"],
	[[1,18,47,227,236,246,248,254,258],"RP^3"],
	[[1,18,47,227,240,248,249,254],"RP^3"],
	[[1,18,47,228,231,248,249,251,254],"RP^3"],
	[[1,18,47,229,232,236,248,259],"RP^3"],
	[[1,18,47,229,232,241,248,252,255],"RP^3"],
	[[1,18,47,236,243,248,256],"RP^3"],
	[[1,18,47,236,246,248,250,254,258],"S^3/Q8"],
	[[1,18,47,236,246,248,253,254,258],"SFS[S^2:(3,2)(3,2)(3,-1)]"],
	[[1,18,47,237,262,268,276,279],"L(3,1)"],
	[[1,18,47,238,244,249,271],"S^2~S^1"],
	[[1,18,47,238,248,249,263],"S^3/Q8"],
	[[1,18,47,238,264,274],"S^2~S^1"],
	[[1,18,47,239,251,255,266,276,279],"L(3,1)"],
	[[1,18,47,240,248,249,250,254],"S^3/Q8"],
	[[1,18,47,240,248,249,253,254],"SFS[S^2:(3,2)(3,2)(3,-1)]"],
	[[1,18,47,242,272,274,278],"S^2~S^1"],
	[[1,18,47,242,275,276,279],"L(3,1)"],
	[[1,19,55,80,86,149,216,252],"S^2~S^1"],
	[[1,19,55,128,149,201,216],"S^2~S^1"],
	[[1,19,56,91,96,150,183,188],"S^3"],
	[[1,19,56,91,96,150,220,258,279],"S^3"],
	[[1,19,56,133,148,190,211,213],"S^2~S^1"],
	[[1,19,56,133,148,194,211,239,240],"S^2~S^1"],
	[[1,19,56,133,150,183,188,208,258],"S^3"],
	[[1,19,56,133,150,189,192,220],"S^2~S^1"],
	[[1,19,56,133,150,194,195,209,220],"S^2~S^1"],
	[[1,19,56,133,150,208,220,279],"S^3"],
	[[1,19,56,133,156,189,192,209,240],"S^2~S^1"],
	[[1,19,56,133,156,190,195,213,239],"S^2~S^1"],
	[[1,19,56,133,156,194,195,240],"S^2~S^1"],
	[[1,19,56,133,156,208,209,240,279],"S^3"],
	[[1,19,57,103,106,134,151,184,197],"S^3"],
	[[1,19,57,123,128,149,207,216],"S^2~S^1"],
	[[1,19,58,129,133,150,189,200,220],"S^2~S^1"],
	[[1,19,62,148,185,190,211,213,240],"S^2~S^1"],
	[[1,19,62,148,185,194,211,239],"S^2~S^1"],
	[[1,19,62,150,185,189,192,220,240],"S^2~S^1"],
	[[1,19,62,150,185,208,220,240,279],"S^3"],
	[[1,19,62,156,180,193,194,202],"S^2~S^1"],
	[[1,19,62,156,184,189,195,208],"S^2~S^1"],
	[[1,19,62,156,184,192,195,279],"S^3"],
	[[1,19,62,156,185,189,192,209],"S^2~S^1"],
	[[1,19,62,156,185,194,195],"S^2~S^1"],
	[[1,19,62,156,185,208,209,279],"S^3"],
	[[1,20,64,96,165,209,231],"S^3"],
	[[1,20,65,106,158,190,237,242,265],"RP^2xS^1"],
	[[1,20,66,110,137,158,191,218,271],"S^2~S^1"],
	[[1,20,66,115,158,191,238,241,274],"S^2~S^1"],
	[[1,20,66,115,158,191,263,271],"S^2~S^1"],
	[[1,20,66,115,162,228,238,274],"S^2~S^1"],
	[[1,20,66,115,162,228,241,263,271],"S^2~S^1"],
	[[1,20,70,101,106,166,214,232],"S^3"],
	[[1,20,70,137,158,195],"S^3"],
	[[1,20,70,137,166,232,261],"S^3"],
	[[1,20,75,158,237,242,243,265],"RP^2xS^1"],
	[[1,20,75,158,238,241,244,274],"S^2~S^1"],
	[[1,20,75,158,244,263,271],"S^2~S^1"],
	[[1,20,75,158,248],"S^3"],
	[[1,20,75,158,249,263,264,274],"S^2~S^1"],
	[[1,20,75,162,191,228,238,244,274],"S^2~S^1"],
	[[1,20,75,163,192,229],"S^3"],
	[[1,21,36,48,238,244,249,271],"S^2~S^1"],
	[[1,21,36,48,238,248,249,263],"S^3/Q8"],
	[[1,21,36,48,238,264,274],"S^2~S^1"],
	[[1,21,36,48,241,244,248,264],"S^3/Q8"],
	[[1,21,36,48,241,263,264,271],"S^2~S^1"],
	[[1,21,36,48,242,272,274,278],"S^2~S^1"],
	[[1,21,41,48,80,229,238,244,271],"S^2~S^1"],
	[[1,21,41,48,80,229,238,248,263],"S^3/Q8"],
	[[1,21,50,62,169,180,189,198,205],"S^2~S^1"],
	[[1,21,52,60,79,106,120,172,237],"S^2~S^1"],
	[[1,21,54,62,82,98,102,173,184],"S^2~S^1"],
	[[1,21,54,62,82,118,174,184],"S^2~S^1"],
	[[1,21,54,62,84,118,138],"S^2~S^1"],
	[[1,21,55,62,76,94,107,110,138],"S^2~S^1"],
	[[1,21,55,62,82,107,111,174,184],"S^2~S^1"],
	[[1,21,55,62,84,107,111,138],"S^2~S^1"],
	[[1,21,56,62,79,116,171,182,270],"S^3"],
	[[1,21,57,62,76,90,107,113,138],"S^2~S^1"],
	[[1,21,58,62,79,120,138],"S^2~S^1"],
	[[1,21,58,62,80,107,113,138],"S^2~S^1"],
	[[1,21,60,62,79,116,131,171,182],"S^2~S^1"],
	[[1,21,60,62,79,120,172,182],"S^2~S^1"],
	[[1,21,63,71,98,103,122,173,239],"S^2~S^1"],
	[[1,21,76,90,107,111,137,177,241],"S^2~S^1"],
	[[1,21,79,104,106,120,172,237,267],"S^3"],
	[[1,21,80,128,175,236,248,249,259],"(RP^2)#3xS^1"],
	[[1,21,81,133,176,241,247,248],"G3"],
	[[1,21,82,98,102,122,173,239],"S^2~S^1"],
	[[1,21,82,118,122,174,239],"S^2~S^1"],
	[[1,21,84,109,115,131,177,218,237],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,21,84,110,113,137,177,238,263],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,21,84,113,115,177,218,238],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,21,84,118,122,138,184,239],"S^2~S^1"],
	[[1,21,86,159,161,169,189,205,219],"S^2~S^1"],
	[[1,21,86,169,189,198,205,224],"S^2~S^1"],
	[[1,21,86,170,236,246,248,254,258],"(RP^2)#3xS^1"],
	[[1,21,86,170,236,247,248,252,259],"(RP^2)#3xS^1"],
	[[1,21,86,170,240,248,249,254],"(RP^2)#3xS^1"],
	[[1,21,86,170,241,247,248,255],"(RP^2)#3xS^1"],
	[[1,21,86,172,204,216,238,252],"S^2~S^1"],
	[[1,21,86,174,200,242,248,249,254],"(RP^2)#3xS^1"],
	[[1,21,86,179,200,223,241,271,272],"S^2~S^1"],
	[[1,21,86,179,204,238,274],"S^2~S^1"],
	[[1,21,86,179,204,241,263,271],"S^2~S^1"],
	[[1,22,37,38,52,62,273,276,277],"S^2~S^1"],
	[[1,22,38,41,90,106,182,271],"S^2~S^1"],
	[[1,22,38,41,93,106,125,185,213],"S^3/Q8"],
	[[1,22,38,45,90,106,125,182,260],"S^2~S^1"],
	[[1,22,41,48,90,172,182,229,271],"S^2~S^1"],
	[[1,22,42,45,93,133,185,244],"S^3/Q8"],
	[[1,22,49,55,75,94,107,110,134],"S^2~S^1"],
	[[1,22,49,57,75,90,107,113,134],"S^2~S^1"],
	[[1,22,51,53,115,183,249,263],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,22,51,62,227,236,246,254,258],"S^2~S^1"],
	[[1,22,51,62,227,236,247,252,259],"S^2~S^1"],
	[[1,22,51,62,227,239,243,255,260],"S^2~S^1"],
	[[1,22,51,62,227,240,243,251,262],"S^2~S^1"],
	[[1,22,51,62,227,240,249,254],"S^2~S^1"],
	[[1,22,51,62,227,241,247,255],"S^2~S^1"],
	[[1,22,51,62,228,231,243,262],"S^2~S^1"],
	[[1,22,51,62,228,231,249,251,254],"S^2~S^1"],
	[[1,22,51,62,229,232,236,259],"S^2~S^1"],
	[[1,22,51,62,229,232,241,252,255],"S^2~S^1"],
	[[1,22,51,62,231,234,236,246],"S^2~S^1"],
	[[1,22,51,62,231,234,240,249,258],"S^2~S^1"],
	[[1,22,51,62,232,235,239,243],"S^2~S^1"],
	[[1,22,51,62,232,235,241,247,260],"S^2~S^1"],
	[[1,22,51,62,236,243,256],"S^2~S^1"],
	[[1,22,51,62,236,246,250,254,258],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,22,51,62,236,246,253,254,258],"SFS[RP^2:(3,1)(3,1)(3,2)]"],
	[[1,22,51,62,236,247,250,252,259],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,22,51,62,236,247,252,253,259],"SFS[RP^2:(3,1)(3,1)(3,2)]"],
	[[1,22,51,62,238,249,263],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,22,51,62,239,243,250,255,260],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,22,51,62,239,243,253,255,260],"SFS[RP^2:(3,1)(3,1)(3,2)]"],
	[[1,22,51,62,240,243,250,251,262],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,22,51,62,240,243,251,253,262],"SFS[RP^2:(3,1)(3,1)(3,2)]"],
	[[1,22,51,62,240,249,250,254],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,22,51,62,240,249,253,254],"SFS[RP^2:(3,1)(3,1)(3,2)]"],
	[[1,22,51,62,241,244,264],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,22,51,62,241,247,250,255],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,22,51,62,241,247,253,255],"SFS[RP^2:(3,1)(3,1)(3,2)]"],
	[[1,22,54,62,76,84,118,135],"S^2~S^1"],
	[[1,22,54,62,94,110,111,118,135],"S^2~S^1"],
	[[1,22,55,62,76,84,107,111,135],"S^2~S^1"],
	[[1,22,55,62,90,227,242,247,255],"S^2~S^1"],
	[[1,22,55,62,90,242,244,264],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,22,55,62,90,242,247,250,255],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,22,55,62,90,242,247,253,255],"SFS[RP^2:(3,1)(3,1)(3,2)]"],
	[[1,22,55,62,92,107,110,240,247],"S^2~S^1"],
	[[1,22,55,62,94,107,110,135],"S^2~S^1"],
	[[1,22,56,62,93,130,227,240],"S^3"],
	[[1,22,56,62,93,130,228,231,251],"S^3"],
	[[1,22,56,62,93,130,240,250],"S^3/Q8"],
	[[1,22,56,62,93,130,240,253],"G3"],
	[[1,22,57,62,79,80,90,120,135],"S^2~S^1"],
	[[1,22,57,62,90,107,113,135],"S^2~S^1"],
	[[1,22,58,62,76,79,120,135],"S^2~S^1"],
	[[1,22,58,62,76,80,107,113,135],"S^2~S^1"],
	[[1,22,59,62,93,227,240,245,251],"S^3"],
	[[1,22,59,62,93,228,231,245],"S^3"],
	[[1,22,59,62,93,240,245,250,251],"S^3/Q8"],
	[[1,22,59,62,93,240,245,251,253],"G3"],
	[[1,22,64,75,147,149,189,205,218],"S^2~S^1"],
	[[1,22,64,75,148,150,180,198,213],"S^2~S^1"],
	[[1,22,64,75,149,151,180,189,207],"S^2~S^1"],
	[[1,22,64,75,150,152,180,189,200],"S^2~S^1"],
	[[1,22,64,75,151,153,180,191,198],"S^2~S^1"],
	[[1,22,64,75,152,154,182,189,205],"S^2~S^1"],
	[[1,22,64,75,180,189,198,205],"S^2~S^1"],
	[[1,22,90,128,187,207,210,229,232],"S^2~S^1"],
	[[1,22,93,109,115,130,185,217],"S^3/Q8"],
	[[1,22,93,130,133,185,227],"S^3"],
	[[1,22,93,130,133,185,250],"S^3/Q8"],
	[[1,22,93,130,133,185,253],"G3"],
	[[1,22,96,147,149,172,174,209,221],"S^2~S^1"],
	[[1,22,96,147,149,205,218,221],"S^2~S^1"],
	[[1,22,96,149,151,180,207,221],"S^2~S^1"],
	[[1,22,96,149,151,187,203,209,221],"S^2~S^1"],
	[[1,22,96,150,152,175,177,184,221],"S^2~S^1"],
	[[1,22,96,150,152,180,200,221],"S^2~S^1"],
	[[1,22,96,152,154,182,205,221],"S^2~S^1"],
	[[1,22,96,152,154,184,210,219,221],"S^2~S^1"],
	[[1,22,96,180,198,205,221],"S^2~S^1"],
	[[1,22,96,181,227,236,240,249,254],"S^2~S^1"],
	[[1,22,96,181,227,236,241,247,255],"S^2~S^1"],
	[[1,22,96,181,227,246,254,258],"S^2~S^1"],
	[[1,22,96,181,227,247,252,259],"S^2~S^1"],
	[[1,22,96,181,229,232,259],"S^2~S^1"],
	[[1,22,96,181,231,234,246],"S^2~S^1"],
	[[1,22,96,181,236,238,249,263],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,22,96,181,236,240,249,250,254],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,22,96,181,236,240,249,253,254],"SFS[RP^2:(3,1)(3,1)(3,2)]"],
	[[1,22,96,181,236,241,244,264],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,22,96,181,236,241,247,250,255],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,22,96,181,236,241,247,253,255],"SFS[RP^2:(3,1)(3,1)(3,2)]"],
	[[1,22,96,181,238,240,246,258,263],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,22,96,181,243,256],"S^2~S^1"],
	[[1,22,96,181,244,252,255,259,264],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,22,96,181,246,250,254,258],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,22,96,181,246,253,254,258],"SFS[RP^2:(3,1)(3,1)(3,2)]"],
	[[1,22,96,181,247,250,252,259],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,22,96,181,247,252,253,259],"SFS[RP^2:(3,1)(3,1)(3,2)]"],
	[[1,22,96,185,211,226,231],"S^3"],
	[[1,22,96,187,210,227,247,252,257],"S^2~S^1"],
	[[1,22,96,187,210,229,232,257],"S^2~S^1"],
	[[1,22,96,187,210,247,250,252,257],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,22,96,187,210,247,252,253,257],"SFS[RP^2:(3,1)(3,1)(3,2)]"],
	[[1,22,96,188,211,245],"S^3"],
	[[1,23,52,62,182,190,227,242,265],"RP^2xS^1"],
	[[1,23,52,62,182,190,242,250,265],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,23,52,62,182,190,242,253,265],"SFS[RP^2:(3,1)(3,1)(3,1)]"],
	[[1,23,65,75,227,237,242,243,265],"RP^2xS^1"],
	[[1,23,65,75,227,248],"S^3"],
	[[1,23,65,75,237,242,243,250,265],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,23,65,75,237,242,243,253,265],"SFS[RP^2:(3,1)(3,1)(3,1)]"],
	[[1,23,65,75,248,250],"S^3/Q8"],
	[[1,23,65,75,248,253],"G3"],
	[[1,23,66,70,101,115,191,255,264],"S^3/Q8"],
	[[1,23,66,74,105,115,191,271],"S^2~S^1"],
	[[1,23,66,75,105,114,244,251],"S^2~S^1"],
	[[1,23,69,75,100,246,252,266,279],"L(3,1)"],
	[[1,23,70,74,105,110,115,195,218],"S^3/Q8"],
	[[1,23,70,75,101,227,247],"S^3"],
	[[1,23,70,75,101,229,232,252],"S^3"],
	[[1,23,70,75,101,244,255,264],"S^3/Q8"],
	[[1,23,70,75,101,247,250],"S^3/Q8"],
	[[1,23,70,75,101,247,253],"G3"],
	[[1,23,71,75,102,248,254],"G3"],
	[[1,23,74,75,105,244,271],"S^2~S^1"],
	[[1,23,74,75,105,248,263],"S^3/Q8"],
	[[1,23,74,75,105,249,264,274],"S^2~S^1"],
	[[1,23,77,86,148,150,198,213],"S^2~S^1"],
	[[1,23,77,86,149,151,189,207],"S^2~S^1"],
	[[1,23,77,86,150,152,189,200],"S^2~S^1"],
	[[1,23,77,86,151,153,191,198],"S^2~S^1"],
	[[1,23,77,86,189,198,205],"S^2~S^1"],
	[[1,23,79,86,98,191,200,266],"S^3"],
	[[1,23,84,86,104,149,151,189,208],"S^2~S^1"],
	[[1,23,84,86,104,149,151,192,279],"S^3"],
	[[1,23,98,122,191,251,266],"S^3"],
	[[1,23,98,122,191,255,265,269],"S^3"],
	[[1,23,98,122,195,218,255,265,267],"S^3"],
	[[1,23,104,120,122,192,255,265,269],"S^3"],
	[[1,23,105,114,115,191,251],"S^2~S^1"],
	[[1,23,106,145,151,163,185,194,255],"S^2~S^1"],
	[[1,23,106,145,153,161,184,194,262],"S^2~S^1"],
	[[1,23,106,148,150,213,217],"S^2~S^1"],
	[[1,23,106,151,153,191,217],"S^2~S^1"],
	[[1,23,106,151,153,194,255,262],"S^2~S^1"],
	[[1,23,106,158,161,234,237,242,265],"RP^2xS^1"],
	[[1,23,106,161,163,184,185,194],"S^2~S^1"],
	[[1,23,106,161,163,189,208],"S^2~S^1"],
	[[1,23,106,163,166,192,214,252],"S^3"],
	[[1,23,106,189,205,217],"S^2~S^1"],
	[[1,23,106,190,227,237,242,265],"RP^2xS^1"],
	[[1,23,106,190,237,242,250,265],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,23,106,190,237,242,253,265],"SFS[RP^2:(3,1)(3,1)(3,1)]"],
	[[1,23,106,196,215,248,249,254,262],"G3"],
	[[1,24,39,40,79,86,275],"S^2~S^1"],
	[[1,24,39,40,84,86,120,276],"S^2~S^1"],
	[[1,24,39,40,122,200,251,275],"S^2~S^1"],
	[[1,24,39,46,84,86,259,269,276],"S^2~S^1"],
	[[1,24,40,43,79,86,110,251],"S^2~S^1"],
	[[1,24,40,43,110,122,200],"S^2~S^1"],
	[[1,24,43,48,110,174,200,231],"S^2~S^1"],
	[[1,24,78,86,236,240,248,249,254],"(RP^2)#3xS^1"],
	[[1,24,78,86,236,241,247,248,255],"(RP^2)#3xS^1"],
	[[1,24,88,96,149,151,207],"S^2~S^1"],
	[[1,24,88,96,150,152,200],"S^2~S^1"],
	[[1,24,88,96,198,205],"S^2~S^1"],
	[[1,24,94,96,113,150,152,201],"S^2~S^1"],
	[[1,24,115,158,159,191,228,263,271],"S^2~S^1"],
	[[1,24,115,159,162,238,274],"S^2~S^1"],
	[[1,24,115,159,162,241,263,271],"S^2~S^1"],
	[[1,24,115,159,167,219,238,258],"S^2~S^1"],
	[[1,24,115,162,164,201],"S^2~S^1"],
	[[1,24,115,162,167,203,240,241,263],"S^2~S^1"],
	[[1,24,115,198,212],"S^2~S^1"],
	[[1,24,115,199,257,275,276,279],"SFS[S^2:(2,1)(2,1)(2,1)(3,-5)]"],
	[[1,24,115,203,219,231,234,244,248],"S^3/Q8"],
	[[1,24,115,203,219,231,234,263,271],"S^2~S^1"],
	[[1,24,115,203,219,238,240,258,271],"S^2~S^1"],
	[[1,25,50,58,79,102,106,209,215],"S^2~S^1"],
	[[1,25,52,54,106,182,185,208,268],"S^3"],
	[[1,25,52,54,106,208,213,270],"S^3"],
	[[1,25,52,56,87,96,117,270],"S^3"],
	[[1,25,52,56,106,116,270],"S^3"],
	[[1,25,52,58,106,120,138,171,182],"S^2~S^1"],
	[[1,25,52,60,87,96,117,131],"S^2~S^1"],
	[[1,25,52,60,106,116,131],"S^2~S^1"],
	[[1,25,52,60,106,120,171,172],"S^2~S^1"],
	[[1,25,54,62,182,208,213,237,270],"S^3"],
	[[1,25,54,62,185,208,237,268],"S^3"],
	[[1,25,56,62,116,182,237,270],"S^3"],
	[[1,25,56,62,116,185,213,237,268],"S^3"],
	[[1,25,58,62,116,131,138,172,237],"S^2~S^1"],
	[[1,25,58,62,120,138,171,237],"S^2~S^1"],
	[[1,25,60,62,116,131,182,237],"S^2~S^1"],
	[[1,25,60,62,120,171,172,182,237],"S^2~S^1"],
	[[1,25,63,69,83,102,106,116],"S^2~S^1"],
	[[1,25,63,71,79,103,106,210,215],"S^2~S^1"],
	[[1,25,63,71,81,103,106,116],"S^2~S^1"],
	[[1,25,79,80,90,96,258,260,276],"S^2~S^1"],
	[[1,25,79,82,102,106,210,215],"S^2~S^1"],
	[[1,25,79,85,90,96,127,260],"S^2~S^1"],
	[[1,25,79,85,95,96,275],"S^2~S^1"],
	[[1,25,79,86,203,210,255,262],"S^2~S^1"],
	[[1,25,80,84,95,96,120,127,258],"S^2~S^1"],
	[[1,25,81,82,102,106,116],"S^2~S^1"],
	[[1,25,84,85,95,96,120,276],"S^2~S^1"],
	[[1,25,87,96,100,117,266],"S^3"],
	[[1,25,87,96,104,117,131,267],"S^3"],
	[[1,25,89,96,256],"S^2~S^1"],
	[[1,25,95,96,121,257],"S^2~S^1"],
	[[1,25,98,106,205],"S^2~S^1"],
	[[1,25,100,106,116,266],"S^3"],
	[[1,25,104,106,116,131,267],"S^3"],
	[[1,25,104,106,120,171,172,267],"S^3"],
	[[1,25,122,206],"S^2~S^1"],
	[[1,26,36,38,80,106,229,264,274],"S^2~S^1"],
	[[1,26,36,48,80,172,244,249,271],"S^2~S^1"],
	[[1,26,36,48,80,172,264,274],"S^2~S^1"],
	[[1,26,38,41,106,244,271],"S^2~S^1"],
	[[1,26,38,41,106,248,263],"S^3/Q8"],
	[[1,26,38,41,106,249,264,274],"S^2~S^1"],
	[[1,26,38,45,106,125,244,260],"S^2~S^1"],
	[[1,26,41,42,100,106],"S^2~S^1"],
	[[1,26,41,45,103,106,247,249,264],"S^2~S^1"],
	[[1,26,41,48,172,229,244,271],"S^2~S^1"],
	[[1,26,41,48,172,229,248,263],"S^3/Q8"],
	[[1,26,41,48,172,229,249,264,274],"S^2~S^1"],
	[[1,26,51,62,90,182,241,264],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,26,52,55,106,237,242,264],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,26,52,55,106,262,264,268,275],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,26,52,62,99,182,239,269,275],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,26,52,62,99,182,242,265],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,26,55,62,182,242,264],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,26,70,75,99,101,190,247],"S^3/Q8"],
	[[1,26,80,86,172,179,204,274],"S^2~S^1"],
	[[1,26,80,86,204,216,252],"S^2~S^1"],
	[[1,26,90,96,207,212,275,276,279],"SFS[S^2:(2,1)(2,1)(2,1)(3,-5)]"],
	[[1,26,97,106,109],"S^2~S^1"],
	[[1,26,99,106,237,239,269,275],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,26,99,106,237,242,265],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,26,99,106,256,257,275,276,279],"SFS[S^2:(2,1)(2,1)(2,1)(3,-5)]"],
	[[1,26,99,106,262,265,268,275],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,26,103,106,125,232,235,264,274],"S^2~S^1"],
	[[1,26,103,106,125,237,238,269,275],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,26,103,106,125,244,247,260,274],"S^2~S^1"],
	[[1,26,108,115],"S^2~S^1"],
	[[1,27,56,62,148,150,185,194,239],"S^2~S^1"],
	[[1,27,56,62,184,219,222,239,270],"S^3"],
	[[1,27,59,62,91,93,183,228,231],"S^3"],
	[[1,27,60,62,131,184,219,222,239],"S^2~S^1"],
	[[1,27,64,69,96,162,165,209],"S^3"],
	[[1,27,66,75,114,132,191,248],"S^3/Q8"],
	[[1,27,69,75,148,150,239,240,247],"S^2~S^1"],
	[[1,27,91,96,183,188,211],"S^3"],
	[[1,28,43,48,174,231,257],"S^2~S^1"],
	[[1,28,43,48,177,221,231,246],"S^2~S^1"],
	[[1,28,55,57,80,86,149,151,252],"S^2~S^1"],
	[[1,28,57,62,184,227,236,255,258],"S^2~S^1"],
	[[1,28,57,62,184,236,250,255,258],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,28,57,62,184,236,253,255,258],"SFS[RP^2:(3,1)(3,1)(3,2)]"],
	[[1,28,70,75,192,227,247,252],"S^3"],
	[[1,28,70,75,192,229,232],"S^3"],
	[[1,28,70,75,192,244,252,255,264],"S^3/Q8"],
	[[1,28,70,75,192,247,250,252],"S^3/Q8"],
	[[1,28,70,75,192,247,252,253],"G3"],
	[[1,29,54,62,118,174,175,184,240],"S^2~S^1"],
	[[1,33,145,147,193,194,202,225],"S^2~S^1"],
	[[1,33,145,147,194,197,202,262,273],"S^2~S^1"],
	[[1,33,145,153,184,191,197,217,255],"S^2~S^1"],
	[[1,33,145,153,184,193,194,225,273],"S^2~S^1"],
	[[1,33,145,153,184,194,197,262],"S^2~S^1"],
	[[1,33,148,150,161,197,213,217],"S^2~S^1"],
	[[1,33,148,150,165,190,213,220],"S^2~S^1"],
	[[1,33,148,150,165,194,220,239,240],"S^2~S^1"],
	[[1,33,157,169,172,187,189,205,218],"S^2~S^1"],
	[[1,33,157,169,173,180,196,198,213],"S^2~S^1"],
	[[1,33,157,169,174,180,189,203,207],"S^2~S^1"],
	[[1,33,157,169,174,187,189,209],"S^2~S^1"],
	[[1,33,157,169,175,180,196,202],"S^2~S^1"],
	[[1,33,157,169,176,187,193,207,209],"S^2~S^1"],
	[[1,33,157,171,176,180,202,207,269],"SFS[S^2:(2,1)(2,1)(2,1)(3,-5)]"],
	[[1,33,157,171,178,180,198,205,222],"S^2~S^1"],
	[[1,33,157,173,178,180,191,193,203],"S^2~S^1"],
	[[1,33,157,173,178,180,198,215],"S^2~S^1"],
	[[1,33,157,180,193,194,202],"S^2~S^1"],
	[[1,33,157,184,189,195,208],"S^2~S^1"],
	[[1,33,157,185,194,195],"S^2~S^1"],
	[[1,33,157,186,273,274,276,277],"RP^2xS^1"],
	[[1,33,158,197,234,237,242,265],"RP^2xS^1"],
	[[1,33,159,169,179,189,198,205,224],"S^2~S^1"],
	[[1,33,159,200,223,241,271,272],"S^2~S^1"],
	[[1,33,159,204,238,274],"S^2~S^1"],
	[[1,33,159,204,241,263,271],"S^2~S^1"],
	[[1,33,160,175,179,207,249,255,278],"S^2~S^1"],
	[[1,33,160,211,245],"S^3"],
	[[1,33,161,169,179,182,184,189,210],"S^2~S^1"],
	[[1,33,161,169,179,189,205,219],"S^2~S^1"],
	[[1,33,161,189,197,205,217],"S^2~S^1"],
	[[1,33,161,190,197,227,237,242,265],"RP^2xS^1"],
	[[1,33,161,190,197,237,242,250,265],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[1,33,161,190,197,237,242,253,265],"SFS[RP^2:(3,1)(3,1)(3,1)]"],
	[[1,33,162,198,204,212],"S^2~S^1"],
	[[1,33,163,184,185,194,197],"S^2~S^1"],
	[[1,33,163,189,197,208],"S^2~S^1"],
	[[1,33,163,206,211],"S^2~S^1"],
	[[1,33,164,175,179,244,249,255],"S^2~S^1"],
	[[1,33,164,201,204],"S^2~S^1"],
	[[1,33,168,237,239,243,269,275],"RP^2xS^1"],
	[[1,33,168,237,241,247,260,269,275],"RP^2xS^1"],
	[[1,33,168,237,242,243,265],"RP^2xS^1"],
	[[1,33,168,238,241,244,274],"S^2~S^1"],
	[[1,33,168,244,263,271],"S^2~S^1"],
	[[1,33,168,248],"S^3"],
	[[2,3,19,157,180,193,194,202],"S^2~S^1"],
	[[2,3,19,157,184,189,195,208],"S^2~S^1"],
	[[2,3,19,157,185,189,192,209],"S^2~S^1"],
	[[2,3,19,157,185,194,195],"S^2~S^1"],
	[[2,3,19,160,211,245],"S^3"],
	[[2,3,19,161,189,197,205,217],"S^2~S^1"],
	[[2,3,19,162,198,204,212],"S^2~S^1"],
	[[2,3,19,163,184,185,194,197],"S^2~S^1"],
	[[2,3,19,163,189,197,208],"S^2~S^1"],
	[[2,3,19,163,206,211],"S^2~S^1"],
	[[2,3,19,163,223,261],"S^3"],
	[[2,3,19,164,198,199,216],"S^2~S^1"],
	[[2,3,19,164,201,204],"S^2~S^1"],
	[[2,3,19,165,189,190,205,220],"S^2~S^1"],
	[[2,3,19,165,189,192,211],"S^2~S^1"],
	[[2,3,19,165,194,195,209,211],"S^2~S^1"],
	[[2,3,22,56,91,145,249],"S^3"],
	[[2,3,22,56,91,150,160,183],"S^3"],
	[[2,3,23,57,103,134,145,255],"S^3"],
	[[2,3,23,57,103,134,151,161,184],"S^3"],
	[[2,3,23,59,103,145,183,191,255],"S^3"],
	[[2,3,26,55,149,164,201],"S^2~S^1"],
	[[2,3,26,57,123,149,164,207],"S^2~S^1"],
	[[2,3,27,56,145,183,245,249],"S^3"],
	[[2,3,27,56,145,185,217,251,262],"S^2~S^1"],
	[[2,3,27,56,150,160,245],"S^3"],
	[[2,3,27,56,150,163,206],"S^2~S^1"],
	[[2,3,27,56,150,165,189,192],"S^2~S^1"],
	[[2,3,27,56,150,165,194,195,209],"S^2~S^1"],
	[[2,3,27,56,153,163,217,251],"S^2~S^1"],
	[[2,3,27,58,129,150,165,189,200],"S^2~S^1"],
	[[2,3,27,59,130,145,183,249,251],"S^3"],
	[[2,3,27,59,130,150,160,251],"S^3"],
	[[2,3,28,57,151,164,198,199],"S^2~S^1"],
	[[2,3,29,58,152,165,189,190,205],"S^2~S^1"],
	[[2,3,30,51,93,145,185],"S^3"],
	[[2,3,30,51,93,153,163,262],"S^3"],
	[[2,3,30,59,145,183,217,254,262],"S^3"],
	[[2,3,30,59,145,185,261,262],"S^3"],
	[[2,3,30,59,153,163,261],"S^3"],
	[[2,5,24,38,65,78,147,243,256],"S^2~S^1"],
	[[2,6,22,36,145,163,192,229],"S^3"],
	[[2,6,22,41,80,145,163,192,249],"S^3"],
	[[2,8,19,35,53,70,101,210,272],"S^3"],
	[[2,8,19,43,53,101,211,272],"S^3"],
	[[2,8,24,38,147,181,229,232,259],"S^2~S^1"],
	[[2,8,24,38,147,181,231,234,246],"S^2~S^1"],
	[[2,8,24,38,147,181,243,256],"S^2~S^1"],
	[[2,8,24,38,147,185,211,226,231],"S^3"],
	[[2,11,19,35,50,64,77,88,97],"S^2~S^1"],
	[[2,11,19,35,56,70,123,200,210],"S^2~S^1"],
	[[2,11,19,41,56,184,185,194,197],"S^2~S^1"],
	[[2,11,19,41,56,189,197,208],"S^2~S^1"],
	[[2,11,19,41,56,206,211],"S^2~S^1"],
	[[2,11,19,41,56,223,261],"S^3"],
	[[2,11,19,43,56,123,200,211],"S^2~S^1"],
	[[2,11,19,43,58,123,129,192,211],"S^2~S^1"],
	[[2,11,22,36,80,91,145,163,229],"S^3"],
	[[2,11,22,41,91,145,163,249],"S^3"],
	[[2,11,22,41,91,153,185,249,262],"S^3"],
	[[2,11,27,41,150,206],"S^2~S^1"],
	[[2,11,27,41,153,217,251],"S^2~S^1"],
	[[2,11,27,43,123,150,200],"S^2~S^1"],
	[[2,11,27,44,124,153,162,165,252],"S^2~S^1"],
	[[2,11,27,44,124,153,217,256],"S^2~S^1"],
	[[2,11,30,41,51,56,93,153,262],"S^3"],
	[[2,11,30,41,56,59,153,261],"S^3"],
	[[2,11,30,41,130,153,243,251],"S^2~S^1"],
	[[2,11,30,41,130,153,249,254,262],"S^2~S^1"],
	[[2,11,30,44,124,130,153,243,256],"S^2~S^1"],
	[[2,11,31,41,64,69,94,153,262],"S^3"],
	[[2,12,19,35,50,64,77,87],"S^2~S^1"],
	[[2,12,19,35,55,69,123,203,207],"S^2~S^1"],
	[[2,12,19,35,57,69,201,203],"S^2~S^1"],
	[[2,12,19,42,55,123,204,207],"S^2~S^1"],
	[[2,12,19,42,57,198,199,216],"S^2~S^1"],
	[[2,12,19,42,57,201,204],"S^2~S^1"],
	[[2,12,19,44,57,129,191,198,216],"S^2~S^1"],
	[[2,12,26,42,55,57,149,201],"S^2~S^1"],
	[[2,12,26,42,123,149,207],"S^2~S^1"],
	[[2,12,28,35,69,151,201,203,216],"S^2~S^1"],
	[[2,12,28,42,151,198,199],"S^2~S^1"],
	[[2,12,28,42,151,201,204,216],"S^2~S^1"],
	[[2,12,28,44,129,151,191,198],"S^2~S^1"],
	[[2,13,19,35,50,64,76],"S^2~S^1"],
	[[2,13,19,41,56,123,129,197,208],"S^2~S^1"],
	[[2,13,19,43,53,111,208,211,272],"S^3"],
	[[2,13,19,43,56,129,189,200,211],"S^2~S^1"],
	[[2,13,19,43,58,189,190,205,220],"S^2~S^1"],
	[[2,13,19,43,58,189,192,211],"S^2~S^1"],
	[[2,13,27,43,56,58,150,189,192],"S^2~S^1"],
	[[2,13,27,43,129,150,189,200],"S^2~S^1"],
	[[2,13,29,43,152,189,190,205],"S^2~S^1"],
	[[2,13,29,43,152,189,192,211,220],"S^2~S^1"],
	[[2,13,29,45,134,152,182,189,205],"S^2~S^1"],
	[[2,14,19,35,50,63],"S^2~S^1"],
	[[2,14,27,44,130,150,160,166,251],"S^3"],
	[[2,14,27,44,130,153,217,221,256],"S^2~S^1"],
	[[2,14,28,44,134,151,180,191,198],"S^2~S^1"],
	[[2,14,30,41,124,153,221,243,251],"S^2~S^1"],
	[[2,14,30,44,153,180,181,198,205],"S^2~S^1"],
	[[2,14,30,44,153,221,243,256],"S^2~S^1"],
	[[2,15,19,35,49],"S^2~S^1"],
	[[2,16,19,34],"S^2~S^1"],
	[[2,17,18],"S^2~S^1"],
	[[3,4,20,24,43,110,174,200,231],"S^2~S^1"],
	[[3,4,20,28,43,174,231,257],"S^2~S^1"],
	[[3,4,27,28,43,69,162,174,257],"S^2~S^1"],
	[[3,4,35,157,169,180,189,198,205],"S^2~S^1"],
	[[3,4,38,75,106,163,172,192],"S^3"],
	[[3,4,40,63,76,95,117,141],"S^2~S^1"],
	[[3,4,40,63,83,98,102,162,174],"S^2~S^1"],
	[[3,4,40,63,83,118,162,173],"S^2~S^1"],
	[[3,4,40,63,85,118,141],"S^2~S^1"],
	[[3,4,40,69,81,82,118,162,173],"S^2~S^1"],
	[[3,4,40,74,118,124,125,141],"S^2~S^1"],
	[[3,4,42,69,149,151,164,174,201],"S^2~S^1"],
	[[3,4,43,63,76,89,121,141],"S^2~S^1"],
	[[3,4,43,70,150,152,163,175,206],"S^2~S^1"],
	[[3,4,43,70,158,177,195,234],"S^3"],
	[[3,4,43,70,161,177,190,195,227],"S^3"],
	[[3,4,43,70,161,177,190,195,250],"SFS[S^2:(5,1)(5,1)(5,-4)]"],
	[[3,4,43,70,161,177,190,195,253],"SFS[S^2:(4,1)(4,1)(4,-3)]"],
	[[3,4,44,63,79,121,141],"S^2~S^1"],
	[[3,4,45,66,78,79,121,141],"S^2~S^1"],
	[[3,4,48,64,96,165,209,231],"S^3"],
	[[3,4,48,70,101,106,166,214,232],"S^3"],
	[[3,4,48,70,137,158,195],"S^3"],
	[[3,4,48,75,163,192,229],"S^3"],
	[[3,5,20,22,36,56,91,249],"S^3"],
	[[3,5,20,22,41,56,80,91,229],"S^3"],
	[[3,6,34,77,88,104,114,165,192],"B2"],
	[[3,6,34,77,95,104,163,189,207],"B2"],
	[[3,6,34,84,95,161,189,205,217],"B2"],
	[[3,6,34,84,95,163,189,208],"B2"],
	[[3,6,41,55,84,94,163,189,208],"B2"],
	[[3,6,43,53,76,111,158,234,272],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[3,6,43,57,80,90,158,229,243],"B2"],
	[[3,6,43,58,76,158,229,243],"B2"],
	[[3,7,34,77,85,94,162,198,213],"B2"],
	[[3,7,34,88,103,113,141,175,200],"B2"],
	[[3,7,34,88,105,113,164,200],"B2"],
	[[3,7,34,94,103,141,175,201],"B2"],
	[[3,7,34,94,105,162,198,212],"B2"],
	[[3,7,34,94,105,164,201],"B2"],
	[[3,7,41,55,95,103,141,175,201],"B2"],
	[[3,7,41,55,95,105,164,201],"B2"],
	[[3,7,43,58,87,129,158,229,243],"B2"],
	[[3,8,34,77,84,88,95,163,208],"B2"],
	[[3,8,34,88,93,104,141,174,207],"B2"],
	[[3,8,34,88,95,104,163,207],"B2"],
	[[3,8,34,104,114,163,206],"B2"],
	[[3,8,34,104,114,165,189,192],"B2"],
	[[3,8,34,104,114,165,194,195,209],"B2"],
	[[3,8,35,55,69,97,157,187,209],"B2"],
	[[3,8,39,52,161,182,185,208,268],"S^3"],
	[[3,8,39,57,103,134,145,148,255],"S^3"],
	[[3,8,43,57,104,113,165,189,192],"B2"],
	[[3,8,45,52,78,79,121,141,171],"S^2~S^1"],
	[[3,11,20,28,43,55,69,231,257],"S^2~S^1"],
	[[3,11,27,28,43,55,162,257],"S^2~S^1"],
	[[3,11,27,28,43,58,124,162,252],"S^2~S^1"],
	[[3,11,34,91,93,126,141,174,278],"S^2~S^1"],
	[[3,11,34,91,95,126,163,278],"S^2~S^1"],
	[[3,11,35,50,65,78,97,166,223],"B2"],
	[[3,11,35,51,65,88,97,167,225],"B2"],
	[[3,11,35,53,65,97,167,223],"B2"],
	[[3,11,36,51,77,88,97,168,226],"B2"],
	[[3,11,38,51,88,97,168,225],"B2"],
	[[3,11,38,53,97,157,188,221],"B2"],
	[[3,11,38,53,97,168,223],"B2"],
	[[3,11,39,51,88,125,130,168,225],"B2"],
	[[3,11,39,53,125,130,157,188,221],"B2"],
	[[3,11,39,53,125,130,168,223],"B2"],
	[[3,11,42,55,149,151,164,201],"S^2~S^1"],
	[[3,11,45,51,93,127,132,163,262],"S^2~S^1"],
	[[3,11,45,59,91,95,127,163,278],"S^2~S^1"],
	[[3,12,35,51,65,87,167,225],"B2"],
	[[3,12,35,53,65,87,88,167,223],"B2"],
	[[3,12,36,51,77,87,165,229,243],"B2"],
	[[3,12,36,51,77,87,168,226],"B2"],
	[[3,12,38,51,87,168,225],"B2"],
	[[3,12,38,53,87,88,168,223],"B2"],
	[[3,12,41,56,128,137,158,195,232],"S^3"],
	[[3,12,43,56,150,152,165,189,192],"S^2~S^1"],
	[[3,12,43,56,158,175,177,195,234],"S^3"],
	[[3,12,48,56,137,158,175,195],"S^3"],
	[[3,13,26,29,41,57,164,213,260],"S^2~S^1"],
	[[3,13,36,51,76,165,229,243],"B2"],
	[[3,13,36,51,76,168,226],"B2"],
	[[3,13,38,51,76,77,168,225],"B2"],
	[[3,13,38,53,76,77,88,168,223],"B2"],
	[[3,13,39,57,80,85,114,124,165],"B2"],
	[[3,13,39,57,111,161,213,271,274],"S^2~S^1"],
	[[3,13,41,51,76,80,165,243,249],"B2"],
	[[3,13,41,57,124,162,217,271,274],"S^2~S^1"],
	[[3,13,42,57,111,115,133,162,258],"S^2~S^1"],
	[[3,13,44,57,161,191,197,216,217],"SFS[K^2:(2,1)]"],
	[[3,13,44,57,165,251,252,271,274],"S^2~S^1"],
	[[3,14,35,50,63,163,206,211],"B2"],
	[[3,14,38,53,97,124,130,157,188],"B2"],
	[[3,14,39,53,124,125,157,188],"B2"],
	[[3,14,39,53,124,125,168,221,223],"B2"],
	[[3,14,41,55,110,111,134,157,197],"B2"],
	[[3,14,45,54,118,167,218,219],"S^2~S^1"],
	[[3,15,35,49,162,198,204,212],"B2"],
	[[3,15,35,49,163,206,211],"B2"],
	[[3,15,35,49,164,198,199,216],"B2"],
	[[3,15,35,49,164,201,204],"B2"],
	[[3,15,38,49,70,101,160,210,272],"S^2~S^1"],
	[[3,15,38,49,70,101,167,247],"S^2~S^1"],
	[[3,15,42,49,69,162,198,203,212],"B2"],
	[[3,15,42,49,69,164,201,203],"B2"],
	[[3,16,34,157,180,193,194,202],"B2"],
	[[3,16,34,157,184,189,195,208],"B2"],
	[[3,16,34,157,185,194,195],"B2"],
	[[3,16,34,161,189,197,205,217],"B2"],
	[[3,16,34,162,198,204,212],"B2"],
	[[3,16,34,163,206,211],"B2"],
	[[3,16,34,164,201,204],"B2"],
	[[4,5,21,22,51,236,243,256],"S^2~S^1"],
	[[4,5,21,22,51,238,249,263],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[4,5,21,22,51,241,244,264],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[4,5,21,22,55,90,242,244,264],"SFS[RP^2:(2,1)(2,1)(2,1)]"],	
	[[4,5,21,26,51,90,182,241,264],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[4,5,21,26,55,182,242,264],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[4,5,22,24,51,78,170,243,256],"S^2~S^1"],
	[[4,5,22,26,51,80,172,249,263],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[4,5,34,40,76,95,117,138],"S^2~S^1"],
	[[4,5,34,40,85,118,138],"S^2~S^1"],
	[[4,5,34,43,76,89,121,138],"S^2~S^1"],
	[[4,5,34,44,79,121,138],"S^2~S^1"],
	[[4,5,34,46,79,121,171,181],"S^2~S^1"],
	[[4,5,40,41,55,76,94,117,138],"S^2~S^1"],
	[[4,5,50,169,180,189,198,205],"S^2~S^1"],
	[[4,5,54,82,118,174,184],"S^2~S^1"],
	[[4,5,54,84,118,138],"S^2~S^1"],
	[[4,5,57,76,89,113,114,121,138],"S^2~S^1"],
	[[4,5,58,79,120,138],"S^2~S^1"],
	[[4,5,60,79,120,172,182],"S^2~S^1"],
	[[4,5,62,82,118,122,174,239],"S^2~S^1"],
	[[4,5,62,86,169,189,198,205,224],"S^2~S^1"],
	[[4,6,36,48,56,75,91,249],"S^3"],
	[[4,6,41,48,56,75,80,91,229],"S^3"],
	[[4,7,49,88,103,113,175,200],"B2"],
	[[4,7,49,88,105,113,141,164,200],"B2"],
	[[4,7,49,94,103,170,247,249],"B2"],
	[[4,7,49,94,103,175,201],"B2"],
	[[4,7,49,94,105,141,162,198,212],"B2"],
	[[4,7,49,94,105,141,164,201],"B2"],
	[[4,8,34,46,65,78,79,121,171],"S^2~S^1"],
	[[4,8,49,88,93,104,174,207],"B2"],
	[[4,8,49,93,95,104,114,174,206],"B2"],
	[[4,8,50,68,81,97,177,210],"B2"],
	[[4,8,55,69,97,171,237,242,262],"B2"],
	[[4,8,55,69,97,171,268,275],"B2"],
	[[4,11,19,21,64,77,88,97,156],"S^2~S^1"],
	[[4,11,36,48,56,75,80,192,249],"S^3"],
	[[4,11,41,48,56,75,192,229],"S^3"],
	[[4,11,41,48,58,75,129,200,229],"S^3"],
	[[4,11,51,65,87,127,136,178,222],"B2"],
	[[4,11,54,68,81,82,118,174],"S^2~S^1"],
	[[4,12,19,21,64,77,87,156],"S^2~S^1"],
	[[4,12,19,28,69,149,174,201,216],"S^2~S^1"],
	[[4,12,43,48,55,69,128,231,257],"S^2~S^1"],
	[[4,12,51,66,88,132,136,178,222],"B2"],
	[[4,13,19,21,64,76,156],"S^2~S^1"],
	[[4,13,35,41,55,110,111,169,197],"B2"],
	[[4,13,55,68,110,111,169,196],"B2"],
	[[4,13,55,68,111,114,136,176,265],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[4,14,19,21,63,156],"S^2~S^1"],
	[[4,15,49,169,194,231,232,256],"B2"],
	[[4,15,49,170,203,231,235,256,260],"B2"],
	[[4,15,49,175,193,196,231,232,256],"B2"],
	[[4,15,49,178,228,232,239,256],"B2"],
	[[4,15,49,178,228,235,236],"B2"],
	[[5,6,20,22,158,244,263,271],"S^2~S^1"],
	[[5,6,20,26,90,158,182,263,271],"S^2~S^1"],
	[[5,6,22,23,66,105,114,244,251],"S^2~S^1"],
	[[5,6,22,23,74,105,244,271],"S^2~S^1"],
	[[5,6,23,26,74,90,105,182,271],"S^2~S^1"],
	[[5,6,34,40,49,95,117,134],"S^2~S^1"],
	[[5,6,34,43,49,89,121,134],"S^2~S^1"],
	[[5,6,34,45,49,89,121,182,190],"S^2~S^1"],
	[[5,6,38,41,75,90,106,182,271],"S^2~S^1"],
	[[5,6,39,43,74,90,114,125,134],"B2"],
	[[5,6,40,41,49,55,94,117,134],"S^2~S^1"],
	[[5,6,41,45,66,95,110,127,134],"B2"],
	[[5,6,49,57,89,113,114,121,134],"S^2~S^1"],
	[[5,6,51,62,65,106,190,236,256],"S^2~S^1"],
	[[5,6,51,62,66,115,191,241,264],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[5,6,51,62,75,236,243,256],"S^2~S^1"],
	[[5,6,51,62,75,238,249,263],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[5,6,51,62,75,241,244,264],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[5,6,55,62,75,90,242,244,264],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[5,6,64,180,189,198,205],"S^2~S^1"],
	[[5,6,65,90,99,182,190,263,271],"S^2~S^1"],
	[[5,6,65,96,106,181,190,256],"S^2~S^1"],
	[[5,6,75,96,180,198,205,221],"S^2~S^1"],
	[[5,6,75,96,181,236,238,249,263],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[5,6,75,96,181,236,241,244,264],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[5,6,75,96,181,243,256],"S^2~S^1"],
	[[5,7,21,22,65,170,236,243,256],"S^2~S^1"],
	[[5,7,22,24,65,78,243,256],"S^2~S^1"],
	[[5,7,22,24,65,85,114,243,260],"S^2~S^1"],
	[[5,8,20,32,77,114,165,187,207],"S^3"],
	[[5,8,38,39,74,85,181,243,256],"B2"],
	[[5,8,50,55,69,97,187,209],"B2"],
	[[5,8,66,82,102,182,210,215],"S^2~S^1"],
	[[5,8,66,82,106,140,182,210,262],"S^2~S^1"],
	[[5,8,66,85,95,96,106,182,275],"S^2~S^1"],
	[[5,8,69,80,97,181,252,254],"B2"],
	[[5,8,69,80,97,186,209],"B2"],
	[[5,8,74,80,97,132,181,252,274],"B2"],
	[[5,11,21,26,69,174,182,242,264],"SFS[RP^2:(2,1)(2,1)(2,1)]"],
	[[5,11,54,55,69,82,118,184],"S^2~S^1"],
	[[5,11,65,78,97,157,166,223],"B2"],
	[[5,11,66,78,127,132,157,166,223],"B2"],
	[[5,11,69,79,121,127,184,246,247],"S^2~S^1"],
	[[5,14,63,157,168,244,263,271],"B2"],
	[[5,14,63,160,166,181,244,263,271],"B2"],
	[[6,7,21,23,172,204,216,238,252],"S^2~S^1"],
	[[6,7,21,23,179,204,238,274],"S^2~S^1"],
	[[6,7,23,24,39,40,79,275],"S^2~S^1"],
	[[6,7,23,24,39,40,84,120,276],"S^2~S^1"],
	[[6,7,23,26,80,172,179,204,274],"S^2~S^1"],
	[[6,7,23,26,80,204,216,252],"S^2~S^1"],
	[[6,7,65,66,78,191,199,263,271],"S^2~S^1"],
	[[6,7,66,74,86,105,115,191,271],"S^2~S^1"],
	[[6,7,66,75,86,105,114,244,251],"S^2~S^1"],
	[[6,7,74,75,86,105,244,271],"S^2~S^1"],
	[[6,7,77,189,198,205],"S^2~S^1"],
	[[6,7,80,99,170,172,190,264,274],"S^2~S^1"],
	[[6,7,86,105,114,115,191,251],"S^2~S^1"],
	[[6,7,86,106,189,205,217],"S^2~S^1"],
	[[6,8,65,75,78,96,243,256],"S^2~S^1"],
	[[6,8,65,75,85,96,114,243,260],"S^2~S^1"],
	[[6,8,78,96,106,190,256],"S^2~S^1"],
	[[6,8,78,96,106,197,216,258],"S^2~S^1"],
	[[6,8,84,96,106,113,197,216,257],"S^2~S^1"],
	[[6,8,85,96,106,114,190,260],"S^2~S^1"],
	[[6,11,19,23,50,88,97,156,224],"S^2~S^1"],
	[[6,11,36,38,88,97,158,168,225],"B2"],
	[[6,11,76,123,129,160,168,223],"B2"],
	[[6,11,77,88,97,158,168,226],"B2"],
	[[6,12,19,23,50,87,156,224],"S^2~S^1"],
	[[6,12,36,38,87,158,168,225],"B2"],
	[[6,12,36,38,87,197,225,234,243],"B2"],
	[[6,12,77,87,158,165,229,243],"B2"],
	[[6,12,77,87,158,168,226],"B2"],
	[[6,12,77,87,160,168,189,223],"B2"],
	[[6,12,77,87,197,226,234,243],"B2"],
	[[6,13,76,158,165,229,243],"B2"],
	[[6,13,76,158,168,226],"B2"],
	[[6,13,76,160,168,189,223],"B2"],
	[[6,13,76,189,206,243,244,263,271],"K^2xS^1"],
	[[7,8,20,24,64,165,209,231],"S^3"],
	[[7,8,20,32,64,114,165,203,209],"S^3"],
	[[7,8,22,24,181,229,232,259],"S^2~S^1"],
	[[7,8,22,24,181,231,234,246],"S^2~S^1"],
	[[7,8,22,24,181,243,256],"S^2~S^1"],
	[[7,8,22,24,187,210,229,232,257],"S^2~S^1"],
	[[7,8,22,32,114,181,203,234,246],"S^2~S^1"],
	[[7,8,24,32,95,181,187,232,259],"S^2~S^1"],
	[[7,8,24,32,95,210,232,257],"S^2~S^1"],
	[[7,8,64,66,159,165,209,228,231],"S^3"],
	[[7,8,88,198,205],"S^2~S^1"],
	[[7,8,96,115,159,162,238,274],"S^2~S^1"],
	[[7,8,96,115,159,167,219,238,258],"S^2~S^1"],
	[[7,8,96,115,162,164,201],"S^2~S^1"],
	[[7,8,96,115,198,212],"S^2~S^1"],
	[[7,11,87,123,158,168,189,226],"B2"],
	[[7,11,87,123,160,168,223],"B2"],
	[[7,11,87,123,171,178,222],"B2"],
	[[7,12,87,159,161,215,224,276,277],"B2"],
	[[7,12,87,159,164,238,273,276,277],"B2"],
	[[7,12,87,172,173,198,273,276],"B2"],
	[[7,12,87,173,178,198,215],"B2"],
	[[7,12,87,201,273,274,276,277],"B2"],
	[[8,11,97,158,168,189,226],"B2"],
	[[8,11,97,160,168,223],"B2"],
	[[8,11,97,174,177,209,275,276],"B2"],
	[[8,11,97,181,184,257,259,275,276],"B2"],
	[[18,22,41,95,128,174,175,184,246],"B2"],
	[[18,22,43,65,75,87,132,182,190],"T^3"],
	[[18,22,43,87,106,132,182,243],"T^3"],
	[[18,22,45,65,75,87,132,134],"T^3"],
	[[18,22,45,66,75,87,97,127,134],"T^3"],
	[[18,22,45,87,106,132,134,190,243],"T^3"],
	[[18,22,48,95,174,184,232,246],"B2"],
	[[18,23,34,58,80,86,97,113,129],"B2"],
	[[18,23,39,57,62,103,114,134,239],"S^2~S^1"],
	[[18,23,39,65,75,78,85,243,256],"B2"],
	[[18,23,39,65,75,114,239,241,247],"B2"],
	[[18,23,39,65,75,114,243,260],"B2"],
	[[18,23,39,78,85,106,190,256],"B2"],
	[[18,23,39,78,85,106,197,216,258],"B2"],
	[[18,23,39,106,114,190,260],"B2"],
	[[18,23,43,106,130,132,134,182,243],"T^3"],
	[[18,23,44,63,75,78,97,127,130],"T^3"],
	[[18,23,44,65,66,78,86,129,132],"T^3"],
	[[18,23,44,78,86,97,127,129],"T^3"],
	[[18,23,45,63,65,86,129,132],"T^3"],
	[[18,23,45,63,66,86,97,127,129],"T^3"],
	[[18,23,45,65,75,130,132],"T^3"],
	[[18,23,45,66,75,97,127,130],"T^3"],
	[[18,23,45,106,130,132,190,243],"T^3"],
	[[18,23,47,57,62,103,134,255],"S^2~S^1"],
	[[18,23,47,65,75,236,247,252,259],"B2"],
	[[18,23,47,65,75,241,247,255],"B2"],
	[[18,23,47,106,190,239,255,260],"B2"],
	[[18,23,47,106,190,241,243,247,255],"B2"],
	[[18,24,34,57,76,86,90,113,124],"B2"],
	[[18,24,34,57,90,96,113,123],"B2"],
	[[18,24,34,57,92,96,113,203,210],"B2"],
	[[18,24,34,58,76,80,96,113,123],"B2"],
	[[18,24,34,58,80,86,113,124],"B2"],
	[[18,24,34,58,82,86,113,254,259],"B2"],
	[[18,24,41,78,86,127,251,256],"T^3"],
	[[18,24,43,57,58,80,86,114,124],"B2"],
	[[18,24,43,76,86,90,114,124],"B2"],
	[[18,24,43,90,96,114,123],"B2"],
	[[18,24,43,92,96,114,203,210],"B2"],
	[[18,24,44,76,78,96,123,127],"T^3"],
	[[18,24,44,78,86,124,127],"T^3"],
	[[18,24,45,63,65,86,97,124,132],"T^3"],
	[[18,24,45,63,66,76,96,123,127],"T^3"],
	[[18,24,45,63,66,86,124,127],"T^3"],
	[[18,24,45,66,75,124,127,129,130],"T^3"],
	[[18,28,48,88,95,110,175,232],"B2"],
	[[18,28,48,92,95,174,232],"B2"],
	[[18,33,39,78,85,161,190,197,256],"B2"],
	[[18,33,39,78,85,161,216,258],"B2"],
	[[18,33,39,78,85,168,246,254,258],"B2"],
	[[18,33,39,114,161,190,197,260],"B2"],
	[[18,33,39,114,161,216,256,258,260],"B2"],
	[[18,33,47,161,190,197,239,255,260],"B2"],
	[[20,30,64,90,125,158,191,218,275],"SFS[K^2:(2,1)(2,1)(2,1)]"],
	[[20,31,64,94,165,228,236,251,252],"S^2~S^1"],
	[[20,33,35,43,90,96,123,203,231],"B2"],
	[[20,33,35,43,92,96,210,231],"B2"],
	[[20,33,49,55,87,94,97,110,142],"B2"],
	[[21,22,55,57,84,90,135,177,186],"T^3"],
	[[21,22,55,58,80,94,138,242,249],"T^3"],
	[[21,22,80,94,124,138,236,249,252],"SFS[T^2:(1,7)]"],
	[[21,22,84,90,123,135,177,180,207],"SFS[T^2:(1,7)]"],
	[[21,23,77,159,165,179,190,198,220],"T^3"],
	[[21,23,86,106,170,190,236,256],"B2"],
	[[21,23,86,106,170,197,216,236,258],"B2"],
	[[21,24,51,57,76,111,170,241,258],"SFS[T^2:(1,7)]"],
	[[21,24,78,159,161,179,190,197,256],"B2"],
	[[21,24,78,159,161,179,216,258],"B2"],
	[[21,24,78,159,168,179,246,254,258],"B2"],
	[[21,24,78,175,201,240,252,254,259],"T^3"],
	[[21,27,86,133,165,167,172,198,224],"T^3"],
	[[21,28,35,44,66,76,88,179,224],"B2"],
	[[21,28,39,44,76,88,178,224],"B2"],
	[[21,28,51,57,78,88,170,201,241],"SFS[T^2:(1,7)]"],
	[[21,28,51,62,78,88,137,170,180],"T^3"],
	[[21,28,76,88,111,178,224,252,276],"B2"],
	[[21,28,78,110,174,227,238,254,263],"(S^2xS^1)#12"],
	[[21,28,78,110,174,238,253,254,263],"(S^2xS^1)#12"],
	[[21,28,80,124,135,138,186,221,252],"SFS[T^2:(1,7)]"],
	[[21,28,83,110,112,174,227,240,252],"(S^2xS^1)#12"],
	[[21,28,83,110,112,174,240,250,252],"(S^2xS^1)#12"],
	[[21,28,86,124,128,135,138,180,221],"T^3"],
	[[21,29,35,44,66,78,179,224],"B2"],
	[[21,29,35,45,63,179,224],"B2"],
	[[21,29,39,44,78,178,224],"B2"],
	[[21,29,39,45,63,66,178,224],"B2"],
	[[21,29,50,58,170,180,187,242],"T^3"],
	[[21,29,50,58,174,180,187,200,240],"T^3"],
	[[21,29,50,62,140,170,180],"T^3"],
	[[21,29,51,57,76,170,201,241],"SFS[T^2:(1,7)]"],
	[[21,29,51,62,76,123,128,170,181],"T^3"],
	[[21,29,51,62,76,137,170,180],"T^3"],
	[[21,29,55,57,76,90,170,201,242],"SFS[T^2:(1,7)]"],
	[[21,29,55,58,80,170,201,242],"SFS[T^2:(1,7)]"],
	[[21,29,55,58,80,174,200,201,240],"SFS[T^2:(1,7)]"],
	[[21,29,57,58,80,123,170,207,242],"SFS[T^2:(1,7)]"],
	[[21,29,76,90,123,170,207,242],"SFS[T^2:(1,7)]"],
	[[21,29,76,96,123,128,170,236],"T^3"],
	[[21,29,76,96,137,170,180,181,236],"T^3"],
	[[21,29,78,111,178,224,252,276],"B2"],
	[[21,29,80,124,170,201,236,252],"T^3"],
	[[21,29,82,170,180,181,209,236,259],"SFS[T^2:(1,7)]"],
	[[21,29,82,170,180,187,209,210,242],"T^3"],
	[[21,29,82,170,201,236,252,254,259],"T^3"],
	[[21,29,82,170,203,207,210,242],"T^3"],
	[[21,29,82,170,203,224,255],"T^3"],
	[[21,29,82,174,206,207,239,260],"B2"],
	[[21,29,82,178,224,246,247,252,276],"B2"],
	[[21,29,82,178,224,255,260,262],"T^3"],
	[[21,29,84,138,184,206,207,239,260],"B2"],
	[[21,29,86,140,170,224],"T^3"],
	[[21,30,35,45,65,77,179,224],"B2"],
	[[21,30,39,45,65,66,77,178,224],"B2"],
	[[21,31,50,58,138,187,242],"T^3"],
	[[21,31,50,60,159,167,170],"B2"],
	[[21,31,50,60,172,182,187,242],"T^3"],
	[[21,31,50,62,138,140],"T^3"],
	[[21,31,51,57,76,138,186,241],"T^3"],
	[[21,31,51,62,76,137,138],"T^3"],
	[[21,31,55,57,76,90,138,186,242],"T^3"],
	[[21,31,76,90,123,138,180,207,242],"SFS[T^2:(1,7)]"],
	[[21,31,76,96,123,128,138,180,236],"T^3"],
	[[21,31,76,96,137,138,181,236],"T^3"],
	[[21,31,82,138,181,209,236,259],"SFS[T^2:(1,7)]"],
	[[21,31,82,138,187,207,209,224,255],"T^3"],
	[[21,31,82,138,187,209,210,242],"T^3"],
	[[21,31,84,159,167,170,218,219],"B2"],
	[[21,31,86,123,124,137,138,181,236],"T^3"],
	[[21,33,77,106,167,172,217,220,224],"T^3"],
	[[21,33,86,161,170,190,197,236,256],"B2"],
	[[21,33,86,161,170,216,236,258],"B2"],
	[[21,33,86,167,170,224,225],"T^3"],
	[[21,33,86,167,172,198,220,224],"T^3"],
	[[21,33,86,168,170,236,246,254,258],"B2"],
	[[22,27,64,75,133,180,190,198],"B2"],
	[[22,28,39,44,88,135,138,178,224],"B2"],
	[[22,28,50,55,76,124,187,221,252],"SFS[T^2:(1,7)]"],
	[[22,28,50,58,76,187,221,257],"SFS[T^2:(1,7)]"],
	[[22,28,51,62,137,221,236],"T^3"],
	[[22,28,57,58,90,124,186,221,252],"SFS[T^2:(1,7)]"],
	[[22,28,64,75,137,181,189],"T^3"],
	[[22,28,65,75,87,133,181,189],"T^3"],
	[[22,28,90,123,180,207,221,257],"T^3"],
	[[22,28,96,137,181,221],"T^3"],
	[[22,29,64,75,129,133,180],"B2"],
	[[22,29,96,129,133,180,189,221],"B2"],
	[[22,30,64,65,103,184,221,258],"B2"],
	[[22,30,64,75,134,137],"T^3"],
	[[22,30,65,75,87,133,134],"T^3"],
	[[22,31,49,51,65,103,184,221,258],"B2"],
	[[23,27,38,45,130,190,198,201,274],"B2"],
	[[23,27,77,86,133,190,198],"T^3"],
	[[23,27,100,190,213,217,260],"T^3"],
	[[23,27,100,194,217,239,240,260],"T^3"],
	[[23,27,103,130,190,198,201,247],"B2"],
	[[23,27,106,133,190,217],"T^3"],
	[[23,29,35,45,63,77,197,217,219],"B2"],
	[[23,29,39,45,97,125,129,201,274],"B2"],
	[[23,29,63,75,77,130,133],"T^3"],
	[[23,29,77,86,129,133],"T^3"],
	[[23,29,100,129,198,213,217,260],"T^3"],
	[[23,29,103,129,130,201,247],"B2"],
	[[23,30,35,45,65,197,217,219],"B2"],
	[[23,30,65,75,130,133],"T^3"],
	[[23,33,77,86,165,170,172,190,225],"T^3"],
	[[27,31,97,103,113,125,218,239,242],"B2"]];

	dc[22]:=[[[1,19,36,52,67,81,94,106,117,127],"S^3"],
	[[1,19,36,52,67,81,100],"S^3"],
	[[1,19,36,52,73,94,102,106,121],"S^2~S^1"],
	[[1,19,36,52,73,100,102,117,121,127],"S^3"],
	[[1,19,36,52,74,94,102,111],"S^2~S^1"],
	[[1,19,36,52,74,110,113],"S^2~S^1"],
	[[1,19,36,52,77,335],"S^3"],
	[[1,19,36,55,114,319],"S^3"],
	[[1,19,36,57,71,78,335],"S^3"],
	[[1,19,36,57,81,91,100,127,138],"S^3"],
	[[1,19,36,57,84,91,314,325,329],"S^3"],
	[[1,19,36,57,84,91,327,329],"S^3/P24"],
	[[1,19,36,57,133,314,325,329,330],"S^3"],
	[[1,19,36,57,133,326,333,335],"S^3"],
	[[1,19,36,57,133,327,329,330],"S^3/P24"],
	[[1,19,36,58,81,91,100,127,131],"S^3"],
	[[1,19,36,58,85,91,333,335],"S^3"],
	[[1,19,36,58,97,103,314,329],"S^3"],
	[[1,19,36,58,97,103,325,327,329],"S^3/P24"],
	[[1,19,36,58,100,103,127,130],"S^3"],
	[[1,19,36,59,73,74,94,103,111],"S^2~S^1"],
	[[1,19,36,59,94,103,106,121],"S^2~S^1"],
	[[1,19,36,59,100,103,117,121,127],"S^3"],
	[[1,19,36,59,106,114,120],"S^2~S^1"],
	[[1,19,36,59,109,114,314],"S^3"],
	[[1,19,36,59,109,114,325,327],"S^3/P24"],
	[[1,19,36,60,110,114],"S^2~S^1"],
	[[1,19,36,64,314,322,325,328,329,330],"S^3"],
	[[1,19,36,64,320],"S^3"],
	[[1,19,36,64,322,326,328,333,335],"S^3"],
	[[1,19,36,64,322,327,328,329,330],"S^3/P24"],
	[[1,19,38,87,152,267,301,312,333],"RP^3"],
	[[1,19,38,92,267,284,289,328,332],"RP^3"],
	[[1,19,38,92,267,312,328,333],"RP^3"],
	[[1,19,38,92,272,284,288,297,309],"RP^2xS^1"],
	[[1,19,38,92,272,299,333],"RP^3"],
	[[1,19,39,102,123,125,276,319,328],"S^2~S^1"],
	[[1,19,39,104,274,329],"S^3"],
	[[1,19,40,55,65,320],"S^3"],
	[[1,19,40,55,65,322,326,328,333,335],"S^3"],
	[[1,19,40,55,65,322,327,328,329,330],"S^3/P24"],
	[[1,19,40,55,65,323,326,327,329,331],"S^3/P24"],
	[[1,19,40,55,65,323,330,331,333,335],"S^3"],
	[[1,19,40,69,79,263,282,303],"S^3"],
	[[1,19,40,69,79,263,286,299,301],"S^2~S^1"],
	[[1,19,40,77,79,94,100,150,335],"S^3"],
	[[1,19,40,115,263,282,290,303,311],"S^3"],
	[[1,19,40,115,275,278,286,333],"RP^3"],
	[[1,19,40,115,283,314,320],"S^3"],
	[[1,19,40,115,283,320,325,327],"S^3/P24"],
	[[1,19,40,115,289,310,320,327,328],"S^3/P24"],
	[[1,19,41,56,57,71,79,335],"S^3"],
	[[1,19,41,56,57,134,292,321,335],"S^3"],
	[[1,19,41,57,61,121,134,292,333,335],"S^3"],
	[[1,19,41,67,79,81,100,117,127,145],"S^3"],
	[[1,19,41,70,79,314,320,325],"S^3"],
	[[1,19,41,70,79,320,327],"S^3/P24"],
	[[1,19,41,70,79,322,328,329,330],"SFS[K^2:(2,1)]"],
	[[1,19,41,70,79,323,326,329,331],"SFS[K^2:(2,1)]"],
	[[1,19,41,72,79,122,138,322,329],"S^3/P24"],
	[[1,19,41,73,79,100,102,121,145],"RP^2xS^1"],
	[[1,19,41,75,79,98,100,123,145],"RP^2xS^1"],
	[[1,19,41,75,79,121,324,326,329,331],"SFS[K^2:(2,1)]"],
	[[1,19,41,76,79,122,320,328],"S^3/P24"],
	[[1,19,41,77,79,94,100,106,145,335],"S^3"],
	[[1,19,41,83,92,263,299,333],"RP^3"],
	[[1,19,41,83,92,290,297],"S^2~S^1"],
	[[1,19,41,85,92,121,137,292,303,333],"Sigma^3"],
	[[1,19,41,102,104,123,292,303,328,331],"SFS[RP^2:(2,1)(3,1)]"],
	[[1,19,41,122,138,141,293,329],"S^3/P24"],
	[[1,19,41,125,276,279,294],"S^2~S^1"],
	[[1,19,41,125,290,308],"S^2~S^1"],
	[[1,19,41,125,291,314,325],"S^3"],
	[[1,19,41,125,291,327],"S^3/P24"],
	[[1,19,42,67,74,92,110,113,127,138],"S^2~S^1"],
	[[1,19,42,71,78,91,92,335],"S^3"],
	[[1,19,42,71,79,292,298,324,329,332],"SFS[K^2:(2,1)]"],
	[[1,19,42,73,79,131,144,322,329],"S^3/P24"],
	[[1,19,42,81,92,100,127,138],"S^3"],
	[[1,19,42,84,92,314,325,329],"S^3"],
	[[1,19,42,84,92,321,324,325,329,332],"SFS[K^2:(2,1)]"],
	[[1,19,42,84,92,327,329],"S^3/P24"],
	[[1,19,42,90,92,132,326,329],"S^3/P24"],
	[[1,19,42,96,104,293,294,300,303],"SFS[RP^2:(2,1)(3,1)]"],
	[[1,19,42,96,104,299,303,333],"Sigma^3"],
	[[1,19,42,100,104,127,137],"S^3"],
	[[1,19,42,102,104,132,292,294,300,303],"SFS[RP^2:(2,1)(3,1)]"],
	[[1,19,42,134,293,294,300,303,304],"SFS[RP^2:(2,1)(3,1)]"],
	[[1,19,42,134,298,314,329],"S^3"],
	[[1,19,42,134,298,321,324,329,332],"SFS[K^2:(2,1)]"],
	[[1,19,42,134,298,325,327,329],"S^3/P24"],
	[[1,19,42,134,299,303,304,333],"Sigma^3"],
	[[1,19,43,69,79,108,263,303,311],"S^3"],
	[[1,19,43,72,79,294,298,300,303,322],"SFS[RP^2:(2,1)(3,1)]"],
	[[1,19,43,85,91,98,104,146,332,335],"S^3"],
	[[1,19,43,85,91,103,104,333,335],"S^3"],
	[[1,19,43,85,92,298,303,326,333],"Sigma^3"],
	[[1,19,43,87,88,111,115,127,129],"S^3"],
	[[1,19,43,87,89,112,115,127,129],"S^2~S^1"],
	[[1,19,43,94,100,115,127,129],"S^3"],
	[[1,19,43,97,104,314,329],"S^3"],
	[[1,19,43,97,104,321,324,329,332],"SFS[K^2:(2,1)]"],
	[[1,19,43,97,104,325,327,329],"S^3/P24"],
	[[1,19,43,98,104,140,146,330,332,335],"S^3"],
	[[1,19,43,99,101,112,115,127,129],"S^3"],
	[[1,19,43,100,104,127,130],"S^3"],
	[[1,19,43,102,104,139,322,324,329,332],"SFS[K^2:(2,1)]"],
	[[1,19,43,103,104,140,326,327,329],"S^3/P24"],
	[[1,19,43,103,104,140,330,333,335],"S^3"],
	[[1,19,43,108,115,297,299,303,333],"Sigma^3"],
	[[1,19,44,55,59,115,319],"S^3"],
	[[1,19,44,55,65,109,283],"S^3"],
	[[1,19,44,59,60,110,115],"S^2~S^1"],
	[[1,19,44,73,79,296,310,323],"S^2~S^1"],
	[[1,19,44,86,92,300,308],"S^3/P24"],
	[[1,19,44,106,115,120],"S^2~S^1"],
	[[1,19,44,109,115,314],"S^3"],
	[[1,19,44,109,115,325,327],"S^3/P24"],
	[[1,19,44,119,125],"S^2~S^1"],
	[[1,19,45,87,92,302,312,328,333],"RP^3"],
	[[1,19,46,70,79,121,292,323,329,331],"SFS[K^2:(2,1)]"],
	[[1,19,46,71,79,130,292,324,329,332],"SFS[K^2:(2,1)]"],
	[[1,19,46,72,79,137,294,300,303,322],"SFS[RP^2:(2,1)(3,1)]"],
	[[1,19,46,75,79,292,324,329,331],"SFS[K^2:(2,1)]"],
	[[1,19,46,75,79,294,303,305,322],"SFS[RP^2:(2,1)(3,1)]"],
	[[1,19,46,85,92,137,303,326,333],"Sigma^3"],
	[[1,19,47,70,79,122,293,320],"S^3/P24"],
	[[1,19,47,76,79,293,322,329,330],"S^3/P24"],
	[[1,19,50,264,267,284,288,309,310],"RP^2xS^1"],
	[[1,19,50,273,284,288,297,309],"RP^2xS^1"],
	[[1,19,50,273,285,287,290,292,297,335],"S^3/Q32"],
	[[1,19,50,273,287,288,290,295,297],"RP^2xS^1"],
	[[1,19,50,273,293,294,300],"S^2~S^1"],
	[[1,19,50,273,299,333],"RP^3"],
	[[1,19,50,275,283,289,314,320],"S^3"],
	[[1,19,50,275,283,289,320,325,327],"S^3/P24"],
	[[1,19,50,275,301,302,309,321],"RP^3"],
	[[1,19,50,275,310,314,320,325,328],"S^3"],
	[[1,19,50,275,310,320,327,328],"S^3/P24"],
	[[1,19,50,276,290,296,308],"S^2~S^1"],
	[[1,19,50,276,291,296,314,325],"S^3"],
	[[1,19,50,276,291,296,327],"S^3/P24"],
	[[1,19,50,278,286,289,333],"RP^3"],
	[[1,19,50,279,283,289,308,314,328],"S^3"],
	[[1,19,50,279,294,296],"S^2~S^1"],
	[[1,19,50,281,329],"S^3"],
	[[1,20,53,82,116,177,241,253,260],"S^3"],
	[[1,20,53,93,177,224,241,260],"S^3"],
	[[1,20,56,121,157,177,280,321],"S^3"],
	[[1,20,57,135,168,220,240,241,244],"L(3,1)"],
	[[1,20,57,135,177,220,221,241,244,276],"L(3,1)"],
	[[1,20,61,121,126,172,227,232],"S^2~S^1"],
	[[1,20,61,157,177,275,309,321],"S^3"],
	[[1,20,61,157,177,280,333],"S^3"],
	[[1,20,66,167,227,259,275,309],"S^3"],
	[[1,20,66,167,232,275,333],"S^3"],
	[[1,20,66,167,232,280,309,321],"S^3"],
	[[1,20,66,168,207,220,240,241,244,276],"L(3,1)"],
	[[1,20,66,177,203,224,241],"S^3"],
	[[1,20,66,177,207,214,221,233,241,244],"S^2xS^1"],
	[[1,20,66,177,207,220,221,241,244],"L(3,1)"],
	[[1,20,66,177,210,214,217,233,241,245],"S^2xS^1"],
	[[1,20,66,177,210,217,220,241,245],"L(3,1)"],
	[[1,21,69,116,179,215,303],"S^3"],
	[[1,21,70,121,157,179,221,320,323],"RP^3"],
	[[1,21,73,98,105,186,236,267],"S^2~S^1"],
	[[1,21,73,148,179,219,273,280,302],"S^2~S^1"],
	[[1,21,73,148,179,219,278,280,314],"S^3/Q16"],
	[[1,21,73,148,179,219,299,301],"S^2~S^1"],
	[[1,21,73,148,186,267,299],"S^2~S^1"],
	[[1,21,73,148,186,272,312,328],"S^2~S^1"],
	[[1,21,80,179,273,275,289,297,311],"S^2~S^1"],
	[[1,21,80,179,273,278,289,301],"S^2~S^1"],
	[[1,21,80,179,273,280,283,297,312],"S^2~S^1"],
	[[1,21,80,179,273,280,286,302],"S^2~S^1"],
	[[1,21,80,179,275,280,283,320,333],"RP^3"],
	[[1,21,80,179,278,280,286,314],"S^3/Q16"],
	[[1,21,80,179,282,303],"S^3"],
	[[1,21,80,179,283,309,320,321],"RP^3"],
	[[1,21,80,179,284,311,320,332],"RP^3"],
	[[1,21,80,179,286,299,301],"S^2~S^1"],
	[[1,21,80,179,289,301,302,314],"S^3/Q16"],
	[[1,21,80,179,289,311,312,320,333],"RP^3"],
	[[1,21,80,186,219,267,273,278,289],"S^2~S^1"],
	[[1,21,80,186,219,267,286,299],"S^2~S^1"],
	[[1,21,80,186,219,267,289,302,314],"S^3/Q16"],
	[[1,21,80,186,219,272,286,312,328],"S^2~S^1"],
	[[1,21,80,188,221,269,283,320,321],"RP^3"],
	[[1,22,38,51,273,284,288,297,309],"RP^2xS^1"],
	[[1,22,38,51,273,299,333],"RP^3"],
	[[1,22,38,51,275,283,289,314,320],"S^3"],
	[[1,22,38,51,275,301,302,309,321],"RP^3"],
	[[1,22,38,51,278,284,286,312,332],"RP^3"],
	[[1,22,38,51,280,311,312,314,320],"S^3"],
	[[1,22,38,51,281,329],"S^3"],
	[[1,22,39,42,84,105,193,329],"S^3"],
	[[1,22,42,51,84,264,329],"S^3"],
	[[1,22,53,55,116,200,241,255,274],"B2"],
	[[1,22,53,56,121,157,164,280,321],"S^3"],
	[[1,22,53,57,107,116,200,241,258,274],"B2"],
	[[1,22,53,57,135,191,221,241,244,276],"B2"],
	[[1,22,53,57,135,200,241,242,258,274],"B2"],
	[[1,22,53,57,135,200,241,246,276],"B2"],
	[[1,22,53,61,157,164,275,309,321],"S^3"],
	[[1,22,53,61,157,164,280,333],"S^3"],
	[[1,22,53,66,164,203,224,241],"S^3"],
	[[1,22,53,66,191,203,214,224,233,241],"S^2xS^1"],
	[[1,22,53,66,191,203,220,224,241],"L(3,1)"],
	[[1,22,53,66,191,207,221,241,244],"B2"],
	[[1,22,53,66,191,210,217,241,245],"B2"],
	[[1,22,53,66,194,212,241,254],"B2"],
	[[1,22,53,66,196,212,241,245],"B2"],
	[[1,22,53,66,197,203,218,222,224,241],"L(3,1)"],
	[[1,22,53,66,197,203,224,235,239,241],"B2"],
	[[1,22,53,66,199,203,224,237,239,241],"L(3,1)"],
	[[1,22,53,66,200,205,241,255],"B2"],
	[[1,22,53,66,200,207,241,246],"B2"],
	[[1,22,53,66,201,203,224,241,314,325],"L(15,4)"],
	[[1,22,53,66,201,203,224,241,327],"L(8,3)"],
	[[1,22,55,57,84,116,201,241,255,274],"B2"],
	[[1,22,55,64,84,116,127,140,194,274],"S^2~S^1"],
	[[1,22,55,64,85,116,127,133,194,274],"S^2~S^1"],
	[[1,22,55,66,84,107,197,205,241,318],"L(5,1)"],
	[[1,22,55,66,84,107,201,205,241,258],"B2"],
	[[1,22,57,66,84,201,205,241,255],"B2"],
	[[1,22,57,66,84,201,207,241,246],"B2"],
	[[1,22,57,66,89,127,138,196,207],"S^2~S^1"],
	[[1,22,57,66,91,127,138,158],"S^2~S^1"],
	[[1,22,58,61,85,157,194,241,254,280],"B2"],
	[[1,22,58,61,85,157,196,241,245,280],"B2"],
	[[1,22,58,66,81,103,127,130,158],"S^2~S^1"],
	[[1,22,58,66,85,194,211,241,254],"B2"],
	[[1,22,58,66,85,196,211,241,245],"B2"],
	[[1,22,58,66,87,136,196,211,238,241],"B2"],
	[[1,22,58,66,88,107,111,127,196,208],"S^3"],
	[[1,22,58,66,88,127,130,197,208],"S^3"],
	[[1,22,58,66,89,107,112,127,196,208],"S^2~S^1"],
	[[1,22,58,66,89,127,131,196,207],"S^2~S^1"],
	[[1,22,58,66,91,127,131,158],"S^2~S^1"],
	[[1,22,59,66,81,103,117,121,127,158],"S^2~S^1"],
	[[1,22,59,66,86,191,207,222,241,244],"B2"],
	[[1,22,59,66,86,199,207,241,244,314],"HS"],
	[[1,22,59,66,88,117,121,127,197,208],"S^3"],
	[[1,22,61,66,81,97,127,133,158],"S^2~S^1"],
	[[1,22,61,66,81,98,117,124,127,158],"S^2~S^1"],
	[[1,22,62,66,84,127,140,158],"S^2~S^1"],
	[[1,22,62,66,85,127,133,158],"S^2~S^1"],
	[[1,22,64,66,84,127,136,151,193,205],"S^2~S^1"],
	[[1,22,64,66,84,127,140,194,205],"S^2~S^1"],
	[[1,22,64,66,85,127,133,194,205],"S^2~S^1"],
	[[1,22,81,97,127,131,157,200,280],"S^2~S^1"],
	[[1,22,82,116,164,241,253,260],"S^3"],
	[[1,22,82,116,175,177,202,241,253],"S^3"],
	[[1,22,82,116,201,241,253,260,327],"L(8,3)"],
	[[1,22,84,107,116,197,241,274,318],"L(5,1)"],
	[[1,22,84,107,116,201,241,258,274],"B2"],
	[[1,22,84,135,194,241,274,306,311,318],"L(5,1)"],
	[[1,22,84,135,194,241,276,323,326],"B2"],
	[[1,22,84,135,194,241,280,311,325,326],"B2"],
	[[1,22,84,135,196,264,267,329],"S^3"],
	[[1,22,84,135,197,241,242,274,318],"L(5,1)"],
	[[1,22,84,135,201,241,242,258,274],"B2"],
	[[1,22,84,135,201,241,246,276],"B2"],
	[[1,22,85,127,135,196,276,328,331],"S^2~S^1"],
	[[1,22,85,142,194,208,211,241,254,277],"B2"],
	[[1,22,85,142,196,208,211,241,245,277],"B2"],
	[[1,22,88,107,111,127,142,196,277],"S^3"],
	[[1,22,88,108,111,127,135,196,277],"S^3"],
	[[1,22,88,127,130,142,197,277],"S^3"],
	[[1,22,89,107,112,127,142,196,277],"S^2~S^1"],
	[[1,22,89,108,112,127,135,196,277],"S^2~S^1"],
	[[1,22,89,127,131,142,196,207,208,277],"S^2~S^1"],
	[[1,22,89,127,135,138,196,276],"S^2~S^1"],
	[[1,22,90,132,135,194,241,280,311,314],"HS"],
	[[1,22,91,107,111,127,142,155,196,285],"S^3"],
	[[1,22,91,107,112,127,142,158,207,277],"S^2~S^1"],
	[[1,22,91,108,111,127,135,155,196,285],"S^3"],
	[[1,22,91,108,112,127,135,158,207,277],"S^2~S^1"],
	[[1,22,91,127,130,142,155,197,285],"S^3"],
	[[1,22,91,127,131,142,158,208,277],"S^2~S^1"],
	[[1,22,91,127,135,138,158,207,276],"S^2~S^1"],
	[[1,22,93,164,180,182,241,255],"S^3"],
	[[1,22,93,164,224,241,260],"S^3"],
	[[1,22,93,175,177,202,224,241],"S^3"],
	[[1,22,93,180,182,191,214,233,241,255],"S^2xS^1"],
	[[1,22,93,180,182,191,220,241,255],"L(3,1)"],
	[[1,22,93,180,182,194,320,335],"Sigma^3"],
	[[1,22,93,180,182,197,218,222,241,255],"L(3,1)"],
	[[1,22,93,180,182,197,235,239,241,255],"B2"],
	[[1,22,93,180,182,199,237,239,241,255],"L(3,1)"],
	[[1,22,93,180,182,201,241,255,314,325],"L(15,4)"],
	[[1,22,93,180,182,201,241,255,327],"L(8,3)"],
	[[1,22,93,191,214,224,233,241,260],"S^2xS^1"],
	[[1,22,93,191,220,224,241,260],"L(3,1)"],
	[[1,22,93,192,263,273,299,333],"RP^3"],
	[[1,22,93,192,263,281,329],"S^3"],
	[[1,22,93,192,273,290,297],"S^2~S^1"],
	[[1,22,93,192,281,284,288,290,309,329],"SFS[RP^2:(3,1)(4,3)]"],
	[[1,22,93,192,281,287,288,295,329],"SFS[RP^2:(3,1)(4,3)]"],
	[[1,22,93,197,218,222,224,241,260],"L(3,1)"],
	[[1,22,93,197,224,235,236,241,280,295],"B2"],
	[[1,22,93,197,224,235,239,241,260],"B2"],
	[[1,22,93,197,227,281,288,289,295,329],"SFS[RP^2:(3,1)(4,3)]"],
	[[1,22,93,199,224,236,237,241,280,295],"L(3,1)"],
	[[1,22,93,199,224,237,239,241,260],"L(3,1)"],
	[[1,22,93,201,224,241,260,314,325],"L(15,4)"],
	[[1,22,93,201,224,241,260,327],"L(8,3)"],
	[[1,22,93,202,227,259,275,309,320],"RP^3"],
	[[1,22,93,202,232,273,283,286,302],"S^2~S^1"],
	[[1,22,93,202,232,273,297,312],"S^2~S^1"],
	[[1,22,93,202,232,275,320,333],"RP^3"],
	[[1,22,93,202,232,278,283,286,314],"S^3/Q16"],
	[[1,22,93,202,232,280,309,320,321],"RP^3"],
	[[1,23,39,44,55,66,109,283],"S^3"],
	[[1,23,52,58,80,103,127,130,154],"S^2~S^1"],
	[[1,23,52,59,77,103,148,209,286,335],"S^3"],
	[[1,23,52,59,80,103,117,121,127,154],"S^2~S^1"],
	[[1,23,52,61,80,97,127,133,154],"S^2~S^1"],
	[[1,23,52,61,80,98,117,124,127,154],"S^2~S^1"],
	[[1,23,52,66,73,102,117,121,127,155],"S^2~S^1"],
	[[1,23,52,66,75,98,117,123,127,155],"S^2~S^1"],
	[[1,23,52,66,77,100,155,335],"S^3"],
	[[1,23,52,66,77,103,278,286,335],"S^3"],
	[[1,23,54,66,273,282,293,294,300,303],"SFS[RP^2:(2,1)(3,1)]"],
	[[1,23,54,66,276,282,290,296,303,308],"SFS[RP^2:(2,1)(3,1)]"],
	[[1,23,54,66,276,282,291,296,303,327],"SFS[D:(2,1)(3,1)]U/mSFS[D:(3,1)(3,1)],m=[-8,11|-5,7]"],
	[[1,23,54,66,279,282,290,291,303,310],"SFS[RP^2:(2,1)(3,1)]"],
	[[1,23,54,66,279,282,294,296,303],"SFS[RP^2:(2,1)(3,1)]"],
	[[1,23,54,66,279,282,303,308,310,327],"SFS[D:(2,1)(3,1)]U/mSFS[D:(3,1)(3,1)],m=[-8,11|-5,7]"],
	[[1,23,55,65,104,116,205,329],"S^3"],
	[[1,23,57,66,71,75,100,276,284,335],"S^3"],
	[[1,23,57,66,71,78,100,155,335],"S^3"],
	[[1,23,57,66,81,91,127,138,155],"S^2~S^1"],
	[[1,23,57,66,97,127,275,283,322,328],"S^2~S^1"],
	[[1,23,57,66,101,108,112,127,276,285],"S^3"],
	[[1,23,57,66,101,127,138,277,285],"S^3"],
	[[1,23,58,59,73,80,102,127,130,154],"S^2~S^1"],
	[[1,23,58,66,81,91,127,131,155],"S^2~S^1"],
	[[1,23,58,66,96,127,277,286],"S^3"],
	[[1,23,58,66,101,127,131,277,285],"S^3"],
	[[1,23,58,66,103,127,130,155],"S^2~S^1"],
	[[1,23,59,66,73,77,102,278,286,335],"S^3"],
	[[1,23,59,66,103,117,121,127,155],"S^2~S^1"],
	[[1,23,61,66,97,127,133,155],"S^2~S^1"],
	[[1,23,61,66,98,117,124,127,155],"S^2~S^1"],
	[[1,23,61,66,100,279,283,308,328],"RP^3"],
	[[1,23,62,66,81,84,127,140,155],"S^2~S^1"],
	[[1,23,62,66,81,85,127,133,155],"S^2~S^1"],
	[[1,23,62,66,104,159,265,329],"S^3"],
	[[1,23,63,66,100,139,140,155,335],"S^3"],
	[[1,23,64,66,97,127,133,275,283],"S^2~S^1"],
	[[1,23,64,66,100,155,320],"RP^3"],
	[[1,23,64,66,103,278,286,320],"RP^3"],
	[[1,23,65,66,104,274,329],"S^3"],
	[[1,23,68,71,135,210,241,245,284],"B2"],
	[[1,23,68,80,164,191,203,224,241],"S^3"],
	[[1,23,68,80,167,169,214,233,241,254],"S^2xS^1"],
	[[1,23,68,80,167,169,220,241,254],"L(3,1)"],
	[[1,23,68,80,168,170,203,224,241,249],"S^2xS^1"],
	[[1,23,68,80,172,174,203,216,224,241],"S^2xS^1"],
	[[1,23,68,80,173,175,205,214,233,241],"S^2xS^1"],
	[[1,23,68,80,173,175,205,220,241],"L(3,1)"],
	[[1,23,68,80,191,194,212,241,254],"B2"],
	[[1,23,68,80,191,196,212,241,245],"B2"],
	[[1,23,68,80,191,200,205,241,255],"B2"],
	[[1,23,68,80,191,200,207,241,246],"B2"],
	[[1,23,68,80,191,201,203,224,241,327],"L(8,3)"],
	[[1,23,68,80,203,214,224,233,241],"S^2xS^1"],
	[[1,23,68,80,203,214,224,235,241,314],"L(5,1)"],
	[[1,23,68,80,203,216,224,241,249,325],"K^2xS^1"],
	[[1,23,68,80,203,220,224,241],"L(3,1)"],
	[[1,23,68,80,207,221,241,244],"B2"],
	[[1,23,68,80,210,217,241,245],"B2"],
	[[1,23,70,80,95,173,174,205,220,241],"L(3,1)"],
	[[1,23,71,75,100,135,207,284,335],"S^3"],
	[[1,23,71,78,100,135,155,207,276,335],"S^3"],
	[[1,23,72,80,97,191,199,205,241,255],"B2"],
	[[1,23,72,80,97,191,199,207,241,246],"B2"],
	[[1,23,72,80,97,205,222,241,255,314],"HS"],
	[[1,23,72,80,97,207,222,241,246,314],"HS"],
	[[1,23,73,77,102,148,209,286,335],"S^3"],
	[[1,23,73,80,95,117,127,208,219],"S^3"],
	[[1,23,73,80,98,164,191,210,333],"S^3"],
	[[1,23,73,80,98,205,217,241,326,328],"B2"],
	[[1,23,73,80,98,210,220,333],"S^3/P24"],
	[[1,23,73,80,98,211,217,241,245],"B2"],
	[[1,23,73,80,102,117,121,127,154],"S^2~S^1"],
	[[1,23,73,80,103,146,210,220,332],"S^3/P24"],
	[[1,23,75,80,97,123,124,127,133,154],"S^2~S^1"],
	[[1,23,75,80,98,117,123,127,154],"S^2~S^1"],
	[[1,23,75,80,100,207,217,335],"S^3"],
	[[1,23,77,80,102,209,219,335],"S^3"],
	[[1,23,78,80,98,146,210,220,328],"S^3/P24"],
	[[1,23,81,91,127,135,138,155,207,276],"S^2~S^1"],
	[[1,23,95,117,127,148,208,286],"S^3"],
	[[1,23,96,127,142,208,286],"S^3"],
	[[1,23,97,142,205,241,255,258,285,316],"L(5,1)"],
	[[1,23,97,142,207,241,246,258,285,316],"L(5,1)"],
	[[1,23,99,153,207,241,244,287],"L(3,1)"],
	[[1,23,100,130,135,207,284,331,332,335],"S^3"],
	[[1,23,100,130,135,212,255,284,332,335],"S^3"],
	[[1,23,101,107,112,127,142,207,285],"S^3"],
	[[1,23,101,108,112,127,135,207,285],"S^3"],
	[[1,23,101,127,131,142,208,285],"S^3"],
	[[1,23,101,127,135,138,207,276,277,285],"S^3"],
	[[1,23,102,145,148,209,286,322,326,335],"S^3"],
	[[1,23,103,127,131,142,158,197,285],"S^3"],
	[[1,23,103,133,135,205,241,289,312],"B2"],
	[[1,23,104,115,116,205,283,314,329],"S^3"],
	[[1,23,104,115,116,205,283,325,327,329],"S^3/P24"],
	[[1,23,105,164,191,203,214,224,241,257],"S^3"],
	[[1,23,105,164,194,212,233,241,254,257],"S^2xS^1"],
	[[1,23,105,164,196,212,233,241,245,257],"S^2xS^1"],
	[[1,23,105,164,200,205,233,241,255,257],"S^2xS^1"],
	[[1,23,105,164,200,207,233,241,246,257],"S^2xS^1"],
	[[1,23,105,167,169,214,220,241,254,257],"L(3,1)"],
	[[1,23,105,167,169,233,241,254,257],"S^2xS^1"],
	[[1,23,105,173,175,205,214,220,241,257],"L(3,1)"],
	[[1,23,105,173,175,205,233,241,257],"S^2xS^1"],
	[[1,23,105,191,194,212,214,241,254,257],"B2"],
	[[1,23,105,191,196,212,214,241,245,257],"B2"],
	[[1,23,105,191,200,205,214,241,255,257],"B2"],
	[[1,23,105,191,200,207,214,241,246,257],"B2"],
	[[1,23,105,193,197,203,224,241,257,318],"L(5,1)"],
	[[1,23,105,193,201,203,224,241,257,258],"B2"],
	[[1,23,105,194,200,205,233,257,320,335],
    "SFS[D:(2,1)(2,1)]U/mSFS[D:(2,1)(3,1)],m=[-5,11|-4,9]"],
	[[1,23,105,203,214,220,224,241,257],"L(3,1)"],
	[[1,23,105,203,224,233,241,257],"S^2xS^1"],
	[[1,23,105,203,224,235,241,257,314],"L(5,1)"],
	[[1,23,105,203,224,235,241,257,325,327],"L(7,1)"],
	[[1,23,105,203,224,239,241,257,258,316],"L(5,1)"],
	[[1,23,105,204,273,279,282,294,296,303],"SFS[RP^2:(2,1)(3,1)]"],
	[[1,23,105,204,282,293,294,300,303],"SFS[RP^2:(2,1)(3,1)]"],
	[[1,23,105,205,233,245,257,320,322,335],
    "SFS[D:(2,1)(2,1)]U/mSFS[D:(2,1)(3,1)],m=[-5,11|-4,9]"],
	[[1,23,105,205,234,241,283,298,314],"L(5,1)"],
	[[1,23,105,205,234,241,283,298,325,327],"L(7,1)"],
	[[1,23,105,205,239,241,255,285,298,314],"L(5,1)"],
	[[1,23,105,207,214,221,241,244,257],"B2"],
	[[1,23,105,207,233,254,257,320,331,335],
    "SFS[D:(2,1)(2,1)]U/mSFS[D:(2,1)(3,1)],m=[-5,11|-4,9]"],
	[[1,23,105,207,237,241,244,287,300],"L(3,1)"],
	[[1,23,105,207,239,241,246,285,298,314],"L(5,1)"],
	[[1,23,105,210,214,217,241,245,257],"B2"],
	[[1,23,105,212,233,254,255,257,320,335],
    "SFS[D:(2,1)(2,1)]U/mSFS[D:(2,1)(3,1)],m=[-5,11|-4,9]"],
	[[1,24,40,41,56,57,71,80,335],"S^3"],
	[[1,24,40,41,70,80,320,327],"S^3/P24"],
	[[1,24,40,41,70,80,322,328,329,330],"SFS[K^2:(2,1)]"],
	[[1,24,40,41,70,80,323,326,329,331],"SFS[K^2:(2,1)]"],
	[[1,24,40,41,72,80,122,138,322,329],"S^3/P24"],
	[[1,24,40,41,76,80,122,320,328],"S^3/P24"],
	[[1,24,40,42,73,80,131,144,322,329],"S^3/P24"],
	[[1,24,40,44,73,80,296,310,323],"S^2~S^1"],
	[[1,24,40,47,70,80,122,293,320],"S^3/P24"],
	[[1,24,40,47,76,80,293,322,329,330],"S^3/P24"],
	[[1,24,52,58,80,94,103,127,130,150],"S^2~S^1"],
	[[1,24,52,61,80,94,97,127,133,150],"S^2~S^1"],
	[[1,24,55,65,70,80,125,291],"S^3"],
	[[1,24,69,80,263,273,278,289,301],"S^2~S^1"],
	[[1,24,69,80,263,273,280,286,302],"S^2~S^1"],
	[[1,24,69,80,263,278,280,286,314],"S^3/Q16"],
	[[1,24,69,80,263,282,303],"S^3"],
	[[1,24,69,80,263,286,299,301],"S^2~S^1"],
	[[1,24,69,80,263,289,301,302,314],"S^3/Q16"],
	[[1,24,77,80,94,100,150,335],"S^3"],
	[[1,24,82,93,164,191,224,241],"S^3"],
	[[1,24,82,93,168,170,224,241,249],"S^2xS^1"],
	[[1,24,82,93,172,174,216,224,241],"S^2xS^1"],
	[[1,24,82,93,191,201,224,241,327],"L(8,3)"],
	[[1,24,82,93,214,224,233,241],"S^2xS^1"],
	[[1,24,82,93,214,224,235,241,314],"L(5,1)"],
	[[1,24,82,93,216,224,241,249,325],"K^2xS^1"],
	[[1,24,82,93,220,224,241],"L(3,1)"],
	[[1,24,107,127,142,221,254,292,320,331],"S^3"],
	[[1,24,108,127,135,221,254,292,320,331],"S^3"],
	[[1,24,109,148,198,202,223,304,331,335],"Sigma^3"],
	[[1,24,116,164,191,241,253],"S^3"],
	[[1,24,116,164,202,223,241,255],"S^3"],
	[[1,24,116,168,170,241,249,253],"S^2xS^1"],
	[[1,24,116,172,174,216,241,253],"S^2xS^1"],
	[[1,24,116,173,175,200,202,223,241],"S^3"],
	[[1,24,116,175,177,223,241,255,260],"S^3"],
	[[1,24,116,179,182,271,303],"S^3"],
	[[1,24,116,191,201,241,253,327],"L(8,3)"],
	[[1,24,116,194,198,223,261,296,335],"Sigma^3"],
	[[1,24,116,194,202,223,320,335],"Sigma^3"],
	[[1,24,116,196,202,223,241,327,328,330],"L(8,3)"],
	[[1,24,116,198,202,223,243,331,335],"Sigma^3"],
	[[1,24,116,201,202,223,241,255,327],"L(8,3)"],
	[[1,24,116,208,211,214,238,241,253],"B2"],
	[[1,24,116,214,233,241,253],"S^2xS^1"],
	[[1,24,116,214,235,241,253,314],"L(5,1)"],
	[[1,24,116,215,263,303],"S^3"],
	[[1,24,116,215,290,297,299,303,333],"Sigma^3"],
	[[1,24,116,216,241,249,253,325],"K^2xS^1"],
	[[1,24,116,217,241,295,306],"L(3,1)"],
	[[1,24,116,220,241,253],"L(3,1)"],
	[[1,25,38,41,93,272,299,333],"RP^3"],
	[[1,25,38,51,83,192,299,333],"RP^3"],
	[[1,25,41,42,71,78,91,93,335],"S^3"],
	[[1,25,41,42,84,93,314,325,329],"S^3"],
	[[1,25,41,42,84,93,327,329],"S^3/P24"],
	[[1,25,41,42,90,93,132,326,329],"S^3/P24"],
	[[1,25,41,43,85,93,298,303,326,333],"Sigma^3"],
	[[1,25,41,46,85,93,137,303,326,333],"Sigma^3"],
	[[1,25,42,46,90,93,121,132,292,329],"S^3/P24"],
	[[1,25,52,59,77,103,105,121,143,335],"S^2~S^1"],
	[[1,25,52,61,77,98,105,124,143,335],"S^2~S^1"],
	[[1,25,56,57,71,79,92,93,335],"S^3"],
	[[1,25,56,57,92,93,134,292,321,335],"S^3"],
	[[1,25,56,61,85,93,137,292,303,321],"Sigma^3"],
	[[1,25,56,61,137,142,227,303,321],"Sigma^3"],
	[[1,25,68,70,105,229,238,330,331,335],"S^3"],
	[[1,25,68,77,105,123,229,238,335],"S^3"],
	[[1,25,68,80,95,164,191,224,241,257],"S^3"],
	[[1,25,68,80,95,220,224,241,257],"L(3,1)"],
	[[1,25,70,75,98,105,143,330,331,335],"S^2~S^1"],
	[[1,25,70,80,174,175,220,224,241,257],"L(3,1)"],
	[[1,25,70,80,221,227,257,330,331,335],"S^3"],
	[[1,25,73,77,102,105,121,143,335],"S^2~S^1"],
	[[1,25,75,77,98,105,123,143,335],"S^2~S^1"],
	[[1,25,77,80,123,221,227,257,335],"S^3"],
	[[1,25,83,93,273,281,287,288,295,329],"SFS[RP^2:(3,1)(4,3)]"],
	[[1,25,83,93,290,297],"S^2~S^1"],
	[[1,25,85,93,121,137,292,303,333],"Sigma^3"],
	[[1,25,90,93,121,130,132,292,298,329],"S^3/P24"],
	[[1,25,92,93,125,290,308],"S^2~S^1"],
	[[1,25,92,93,125,291,314,325],"S^3"],
	[[1,25,92,93,125,291,327],"S^3/P24"],
	[[1,25,95,105,164,191,214,224,241],"S^3"],
	[[1,25,95,105,193,197,224,241,318],"L(5,1)"],
	[[1,25,95,105,193,201,224,241,258],"B2"],
	[[1,25,95,105,214,220,224,241],"L(3,1)"],
	[[1,25,95,105,224,233,241],"S^2xS^1"],
	[[1,25,95,105,224,235,241,314],"L(5,1)"],
	[[1,25,95,105,224,235,241,325,327],"L(7,1)"],
	[[1,25,95,105,224,239,241,258,316],"L(5,1)"],
	[[1,25,97,105,124,127,133,230,238],"S^3"],
	[[1,25,98,105,117,127,230,238],"S^3"],
	[[1,25,102,105,121,143,145,322,326,335],"S^2~S^1"],
	[[1,25,103,105,124,224,236,241,314],"L(5,1)"],
	[[1,25,103,105,124,224,236,241,325,327],"L(7,1)"],
	[[1,25,126,224,241,248],"S^2xS^1"],
	[[1,25,126,225,297],"S^2~S^1"],
	[[1,25,126,232,252,299],"S^2~S^1"],
	[[1,26,39,42,105,274,329],"S^3"],
	[[1,26,42,48,102,105,292,294,303,331],"SFS[RP^2:(2,1)(3,1)]"],
	[[1,26,42,48,102,105,305,322,324,329],"SFS[K^2:(2,1)]"],
	[[1,26,42,51,193,264,274,329],"S^3"],
	[[1,26,52,64,67,81,105,127,137,320],"L(3,1)"],
	[[1,26,53,55,84,116,193,200,241,255],"B2"],
	[[1,26,53,61,85,111,116,127,239,245],"S^3"],
	[[1,26,53,62,85,112,116,127,238,246],"S^2~S^1"],
	[[1,26,54,66,96,204,279,294,296,303],"SFS[RP^2:(2,1)(3,1)]"],
	[[1,26,55,57,116,193,196,241,328,330],"B2"],
	[[1,26,55,57,116,193,201,241,255],"B2"],
	[[1,26,55,57,116,239,241,255,316],"L(5,1)"],
	[[1,26,55,58,116,127,193,194,323,331],"S^2~S^1"],
	[[1,26,55,62,116,127,140,158,193,205],"S^2~S^1"],
	[[1,26,55,64,94,105,127,137,151],"S^2~S^1"],
	[[1,26,55,64,116,127,136,151],"S^2~S^1"],
	[[1,26,55,64,116,127,140,193,194],"S^2~S^1"],
	[[1,26,57,59,102,105,145,298,303,322],"Sigma^3"],
	[[1,26,57,61,111,116,239,241,255,328],"L(5,1)"],
	[[1,26,57,66,193,201,205,241,255,274],"B2"],
	[[1,26,57,66,193,201,207,241,246,274],"B2"],
	[[1,26,57,66,205,239,241,255,274,316],"L(5,1)"],
	[[1,26,57,66,207,239,241,246,274,316],"L(5,1)"],
	[[1,26,58,66,127,204,280,304],"S^3"],
	[[1,26,58,66,127,212,236,260,295,306],"S^3"],
	[[1,26,58,66,127,212,239,280,306],"S^3"],
	[[1,26,62,66,127,136,151,158,194,274],"S^2~S^1"],
	[[1,26,62,66,127,140,158,193,274],"S^2~S^1"],
	[[1,26,64,66,127,136,151,205,274],"S^2~S^1"],
	[[1,26,64,66,127,140,193,194,205,274],"S^2~S^1"],
	[[1,26,71,78,91,92,104,105,335],"S^3"],
	[[1,26,71,78,103,105,298,303,326],"Sigma^3"],
	[[1,26,73,79,104,105,131,144,322,329],"S^3/P24"],
	[[1,26,84,86,98,105,300,305,329,330],"S^3/P24"],
	[[1,26,84,86,104,105,147,300,329],"S^3/P24"],
	[[1,26,84,92,104,105,327,329],"S^3/P24"],
	[[1,26,85,88,111,116,127,238,245],"S^3"],
	[[1,26,85,89,112,116,127,238,245],"S^2~S^1"],
	[[1,26,85,93,127,230,238,295,306],"S^3"],
	[[1,26,86,91,98,105,133,300,305,329],"S^3/P24"],
	[[1,26,87,88,111,116,127,136],"S^3"],
	[[1,26,87,89,112,116,127,136],"S^2~S^1"],
	[[1,26,90,92,104,105,132,326,329],"S^3/P24"],
	[[1,26,94,100,116,127,136],"S^3"],
	[[1,26,96,105,273,279,294,296,303],"SFS[RP^2:(2,1)(3,1)]"],
	[[1,26,96,105,293,294,300,303],"SFS[RP^2:(2,1)(3,1)]"],
	[[1,26,96,105,299,303,333],"Sigma^3"],
	[[1,26,97,101,112,116,127,239,246],"S^3"],
	[[1,26,97,105,127,302,307],"S^3"],
	[[1,26,98,105,133,146,298,303,332],"Sigma^3"],
	[[1,26,99,101,112,116,127,136],"S^3"],
	[[1,26,102,105,132,292,294,300,303],"SFS[RP^2:(2,1)(3,1)]"],
	[[1,26,107,116,164,191,214,241],"S^3"],
	[[1,26,107,116,193,197,241,318],"L(5,1)"],
	[[1,26,107,116,193,201,241,258],"B2"],
	[[1,26,107,116,208,211,238,241],"B2"],
	[[1,26,107,116,214,220,241],"L(3,1)"],
	[[1,26,107,116,233,241],"S^2xS^1"],
	[[1,26,107,116,235,241,314],"L(5,1)"],
	[[1,26,107,116,235,241,325,327],"L(7,1)"],
	[[1,26,107,116,239,241,258,316],"L(5,1)"],
	[[1,26,111,116,130,193,196,241,318],"L(5,1)"],
	[[1,26,112,116,127,138,238,245,328,331],"S^2~S^1"],
	[[1,26,112,116,131,207,211,238,241],"B2"],
	[[1,26,112,116,131,238,241,254,323,330],"B2"],
	[[1,26,112,116,133,158,200,207,238,241],"B2"],
	[[1,26,135,164,191,214,241,242],"S^3"],
	[[1,26,135,164,191,223,240,241,244],"S^3"],
	[[1,26,135,164,200,221,223,240,241,246],"S^3"],
	[[1,26,135,193,194,241,274,276,323,326],"B2"],
	[[1,26,135,193,194,241,306,311,318],"L(5,1)"],
	[[1,26,135,193,196,264,267,274,329],"S^3"],
	[[1,26,135,193,197,241,242,318],"L(5,1)"],
	[[1,26,135,193,201,241,242,258],"B2"],
	[[1,26,135,193,201,241,246,274,276],"B2"],
	[[1,26,135,205,211,238,241,307,312],"B2"],
	[[1,26,135,208,211,238,241,242],"B2"],
	[[1,26,135,214,220,241,242],"L(3,1)"],
	[[1,26,135,220,223,240,241,244],"L(3,1)"],
	[[1,26,135,221,223,240,254,320,331,335],"Sigma^3"],
	[[1,26,135,233,241,242],"S^2xS^1"],
	[[1,26,135,234,297,299,303,333],"Sigma^3"],
	[[1,26,135,235,241,242,314],"L(5,1)"],
	[[1,26,135,235,241,242,325,327],"L(7,1)"],
	[[1,26,135,238,241,245,307,312,326,328],"B2"],
	[[1,26,135,239,241,242,258,316],"L(5,1)"],
	[[1,26,135,239,241,246,274,276,316],"L(5,1)"],
	[[1,26,135,239,241,246,276,280,306,326],"L(5,1)"],
	[[1,26,135,240,256,263,292,303],"S^3"],
	[[1,27,52,61,80,127,133,154,205,283],"S^2~S^1"],
	[[1,27,53,66,85,212,234,238,241,277],"B2"],
	[[1,27,53,66,85,212,241,254,275],"B2"],
	[[1,27,55,57,116,127,274,275,322,328],"S^2~S^1"],
	[[1,27,55,64,116,127,129,151],"S^2~S^1"],
	[[1,27,55,64,116,127,133,274,275],"S^2~S^1"],
	[[1,27,55,65,97,104,116,283,329],"S^3"],
	[[1,27,55,65,116,141,265,269,329,331],"S^3"],
	[[1,27,55,65,116,141,283,304,329],"S^3"],
	[[1,27,57,66,127,205,275,322,328],"S^2~S^1"],
	[[1,27,57,66,127,210,244,275,324,328],"S^2~S^1"],
	[[1,27,58,62,112,116,265,269,286,329],"S^3"],
	[[1,27,58,66,207,241,275,323,330],"B2"],
	[[1,27,58,66,211,234,238,241,277],"B2"],
	[[1,27,58,66,211,241,254,275],"B2"],
	[[1,27,61,66,127,133,155,205,283],"S^2~S^1"],
	[[1,27,62,66,127,129,151,158,194,274],"S^2~S^1"],
	[[1,27,62,66,127,133,158,194,275],"S^2~S^1"],
	[[1,27,64,66,127,129,151,205,274],"S^2~S^1"],
	[[1,27,64,66,127,133,205,275],"S^2~S^1"],
	[[1,27,68,80,97,173,175,220,241,283],"L(3,1)"],
	[[1,27,68,80,97,191,200,241,255,283],"B2"],
	[[1,27,72,80,191,199,241,255,283],"B2"],
	[[1,27,72,80,222,241,255,283,314],"HS"],
	[[1,27,84,88,111,116,127,307,310],"S^3"],
	[[1,27,84,89,112,116,127,307,310],"S^2~S^1"],
	[[1,27,87,88,111,116,127,129],"S^3"],
	[[1,27,87,89,112,116,127,129],"S^2~S^1"],
	[[1,27,94,100,116,127,129],"S^3"],
	[[1,27,97,104,115,116,314,329],"S^3"],
	[[1,27,97,104,115,116,325,327,329],"S^3/P24"],
	[[1,27,97,105,234,241,298,314],"L(5,1)"],
	[[1,27,97,105,234,241,298,325,327],"L(7,1)"],
	[[1,27,99,101,112,116,127,129],"S^3"],
	[[1,27,103,104,115,116,140,326,327,329],"S^3/P24"],
	[[1,27,103,105,140,234,241,298,326,327],"L(7,1)"],
	[[1,27,108,116,297,299,303,333],"Sigma^3"],
	[[1,27,112,116,127,131,307,310,322,323],"S^2~S^1"],
	[[1,27,112,116,138,265,269,286,326,329],"S^3"],
	[[1,27,112,116,138,269,272,307,329],"S^3"],
	[[1,27,115,116,141,298,299,303,333],"Sigma^3"],
	[[1,27,115,116,141,304,314,329],"S^3"],
	[[1,27,115,116,141,304,325,327,329],"S^3/P24"],
	[[1,27,118,126,241],"S^2xS^1"],
	[[1,28,73,80,168,170,289,312,328],"S^2~S^1"],
	[[1,28,73,80,220,249,289,312,328],"SFS[D_:(3,1)(3,1)]"],
	[[1,28,98,105,183,186,236],"S^2~S^1"],
	[[1,28,119,126],"S^2~S^1"],
	[[1,29,45,51,196,267,298,314,329],"S^3"],
	[[1,29,45,51,196,267,299,303,304,333],"Sigma^3"],
	[[1,29,45,51,196,271,311,314,322,329],"S^3"],
	[[1,29,45,51,200,254,271,311,314,329],"S^3"],
	[[1,29,49,51,151,196,267,298,311,329],"S^3"],
	[[1,29,49,51,151,196,271,322,329],"S^3"],
	[[1,29,49,51,151,200,254,271,329],"S^3"],
	[[1,29,53,66,85,136,212,238,241,277],"B2"],
	[[1,29,53,66,87,212,241,245,277],"B2"],
	[[1,29,57,66,87,89,127,138,207,277],"S^2~S^1"],
	[[1,29,57,66,99,101,127,138,207,277],"S^3"],
	[[1,29,58,66,85,87,211,241,245,277],"B2"],
	[[1,29,58,66,87,89,127,131,207,277],"S^2~S^1"],
	[[1,29,58,66,99,101,127,131,207,277],"S^3"],
	[[1,29,58,66,136,211,238,241,277],"B2"],
	[[1,29,74,80,215,241,287],"L(3,1)"],
	[[1,29,97,105,136,241,298,314],"L(5,1)"],
	[[1,29,97,105,136,241,298,325,327],"L(7,1)"],
	[[1,29,99,105,239,241,246,298,314],"L(5,1)"],
	[[1,29,103,105,136,140,241,298,326,327],"L(7,1)"],
	[[1,30,52,64,80,154,208,286,320],"RP^3"],
	[[1,30,52,66,77,155,208,286,335],"S^3"],
	[[1,30,58,66,127,130,208,278],"S^3"],
	[[1,30,59,66,117,121,127,208,278],"S^3"],
	[[1,30,64,66,155,208,286,320],"RP^3"],
	[[1,30,75,80,207,208,217,286,335],"S^3"],
	[[1,30,77,80,154,208,286,335],"S^3"],
	[[1,31,57,66,127,138,209,277,285,287],"S^3"],
	[[1,31,58,66,127,131,209,277,285,287],"S^3"],
	[[1,31,74,80,149,224,241,287],"L(3,1)"],
	[[1,31,76,80,192,197,218,224,241,289],"L(3,1)"],
	[[1,31,76,80,218,224,227,241,287],"L(3,1)"],
	[[1,32,52,66,73,117,124,127,210,280],"S^2~S^1"],
	[[1,32,56,66,121,154,220,280,321],"S^3/P24"],
	[[1,32,61,66,121,123,207,241,244,280],"B2"],
	[[1,32,61,66,154,220,275,309,321],"S^3/P24"],
	[[1,32,63,66,167,174,227,275,309],"S^3"],
	[[1,33,38,51,91,200,329],"S^3"],
	[[1,33,49,51,200,271,313,329],"S^3"],
	[[1,35,164,178,191,207,221,241,244],"S^3"],
	[[1,35,164,178,194,212,241,254],"S^3"],
	[[1,35,164,178,196,212,241,245],"S^3"],
	[[1,35,164,180,202,224,241,260],"S^3"],
	[[1,35,164,182,191,223,241,253],"S^3"],
	[[1,35,164,182,202,241,255],"S^3"],
	[[1,35,164,184,191,214,240,241,242],"S^3"],
	[[1,35,164,184,191,223,241,244],"S^3"],
	[[1,35,164,184,200,221,223,241,246],"S^3"],
	[[1,35,164,185,191,214,224,232,236,241],"S^3"],
	[[1,35,165,167,192,194,241,256],"S^3"],
	[[1,35,165,167,224,241,261],"S^3"],
	[[1,35,167,169,178,241,254],"S^3"],
	[[1,35,167,169,189,212,241,256],"S^3"],
	[[1,35,168,170,178,207,221,241,244,249],"S^2xS^1"],
	[[1,35,168,170,178,210,217,241,245,249],"S^2xS^1"],
	[[1,35,168,170,182,223,241,249,253],"S^2xS^1"],
	[[1,35,168,170,183,215,241,256,298,326],"S^2xS^1"],
	[[1,35,168,170,183,217,241,247,298,326],"S^2xS^1"],
	[[1,35,168,170,184,223,241,244,249],"S^2xS^1"],
	[[1,35,168,170,187,215,241,249,256],"S^2xS^1"],
	[[1,35,168,170,187,217,241,247,249],"S^2xS^1"],
	[[1,35,169,172,184,216,223,227,241,330],"S^2xS^1"],
	[[1,35,169,172,184,227,241,252,304],"S^2xS^1"],
	[[1,35,178,191,194,212,214,233,241,254],"S^2xS^1"],
	[[1,35,178,191,194,212,220,241,254],"L(3,1)"],
	[[1,35,178,191,196,212,214,233,241,245],"S^2xS^1"],
	[[1,35,178,191,196,212,220,241,245],"L(3,1)"],
	[[1,35,178,191,201,207,221,241,244,327],"L(8,3)"],
	[[1,35,178,194,197,212,218,222,241,254],"L(3,1)"],
	[[1,35,178,194,197,212,235,239,241,254],"B2"],
	[[1,35,178,194,199,212,237,239,241,254],"L(3,1)"],
	[[1,35,178,194,201,212,241,254,314,325],"L(15,4)"],
	[[1,35,178,194,201,212,241,254,327],"L(8,3)"],
	[[1,35,178,196,197,212,218,222,241,245],"L(3,1)"],
	[[1,35,178,196,197,212,235,239,241,245],"B2"],
	[[1,35,178,196,199,212,237,239,241,245],"L(3,1)"],
	[[1,35,178,196,201,212,241,245,314,325],"L(15,4)"],
	[[1,35,178,196,201,212,241,245,327],"L(8,3)"],
	[[1,35,178,203,224,241],"S^3"],
	[[1,35,178,207,214,221,233,241,244],"S^2xS^1"],
	[[1,35,178,207,214,221,235,241,244,314],"L(5,1)"],
	[[1,35,178,207,216,221,241,244,249,325],"K^2xS^1"],
	[[1,35,178,207,220,221,241,244],"L(3,1)"],
	[[1,35,179,219,261,264,275,301,310],"S^2~S^1"],
	[[1,35,179,219,261,267,273,278,311],"S^2~S^1"],
	[[1,35,179,219,261,267,302,311,314],"S^3/Q16"],
	[[1,35,179,223,271,303],"S^3"],
	[[1,35,180,191,202,214,224,233,241,260],"S^2xS^1"],
	[[1,35,180,191,202,220,224,241,260],"L(3,1)"],
	[[1,35,180,192,202,273,290,297],"S^2~S^1"],
	[[1,35,180,197,202,218,222,224,241,260],"L(3,1)"],
	[[1,35,180,197,202,224,235,239,241,260],"B2"],
	[[1,35,180,199,202,224,237,239,241,260],"L(3,1)"],
	[[1,35,180,201,202,224,241,260,314,325],"L(15,4)"],
	[[1,35,180,201,202,224,241,260,327],"L(8,3)"],
	[[1,35,180,227,259,279,308,309,328],"RP^3"],
	[[1,35,180,231,255,256,279,335],"Sigma^3"],
	[[1,35,180,232,264,267,310,312],"S^2~S^1"],
	[[1,35,180,232,269,272,276,312],"S^2~S^1"],
	[[1,35,180,232,273,283,286,302],"S^2~S^1"],
	[[1,35,180,232,273,297,312],"S^2~S^1"],
	[[1,35,180,232,275,322,326,328,335],"SFS[RP^2:(2,1)(5,1)]"],
	[[1,35,180,232,275,323,330,331,335],"SFS[RP^2:(2,1)(5,1)]"],
	[[1,35,180,232,278,283,286,314],"S^3/Q16"],
	[[1,35,180,232,278,283,286,325,327],"SFS[S^2:(4,1)(5,2)(5,-3)]"],
	[[1,35,181,198,202,236,284,296,312],"RP^3"],
	[[1,35,182,191,201,223,241,253,327],"L(8,3)"],
	[[1,35,182,191,202,214,233,241,255],"S^2xS^1"],
	[[1,35,182,191,202,220,241,255],"L(3,1)"],
	[[1,35,182,196,202,241,314,325,328,330],"L(15,4)"],
	[[1,35,182,196,202,241,327,328,330],"L(8,3)"],
	[[1,35,182,197,202,218,222,241,255],"L(3,1)"],
	[[1,35,182,197,202,235,239,241,255],"B2"],
	[[1,35,182,198,202,243,331,335],"Sigma^3"],
	[[1,35,182,199,202,237,239,241,255],"L(3,1)"],
	[[1,35,182,201,202,241,255,314,325],"L(15,4)"],
	[[1,35,182,201,202,241,255,327],"L(8,3)"],
	[[1,35,182,214,223,233,241,253],"S^2xS^1"],
	[[1,35,182,214,223,235,241,253,314],"L(5,1)"],
	[[1,35,182,216,223,241,249,253,325],"K^2xS^1"],
	[[1,35,182,220,223,241,253],"L(3,1)"],
	[[1,35,182,241,249,252,253,304,325,330],"K^2xS^1"],
	[[1,35,183,216,217,241,247,298,325,326],"K^2xS^1"],
	[[1,35,183,224,232,241,248],"S^2xS^1"],
	[[1,35,183,225,232,297],"S^2~S^1"],
	[[1,35,183,234,236,241,247,248],"S^2xS^1"],
	[[1,35,183,252,299],"S^2~S^1"],
	[[1,35,184,191,201,223,241,244,327],"L(8,3)"],
	[[1,35,184,193,196,240,264,267,274,329],"S^3"],
	[[1,35,184,193,197,240,241,242,318],"L(5,1)"],
	[[1,35,184,193,201,240,241,242,258],"B2"],
	[[1,35,184,193,201,240,241,246,274,276],"B2"],
	[[1,35,184,200,201,221,223,241,246,327],"L(8,3)"],
	[[1,35,184,214,220,240,241,242],"L(3,1)"],
	[[1,35,184,214,223,233,241,244],"S^2xS^1"],
	[[1,35,184,214,223,235,241,244,314],"L(5,1)"],
	[[1,35,184,216,223,241,244,249,325],"K^2xS^1"],
	[[1,35,184,220,223,241,244],"L(3,1)"],
	[[1,35,184,233,240,241,242],"S^2xS^1"],
	[[1,35,184,235,240,241,242,314],"L(5,1)"],
	[[1,35,184,235,240,241,242,325,327],"L(7,1)"],
	[[1,35,184,239,240,241,242,258,316],"L(5,1)"],
	[[1,35,184,239,240,241,246,274,276,316],"L(5,1)"],
	[[1,35,185,193,197,224,232,236,241,318],"L(5,1)"],
	[[1,35,185,193,201,224,232,236,241,258],"B2"],
	[[1,35,185,214,220,224,232,236,241],"L(3,1)"],
	[[1,35,185,224,232,233,236,241],"S^2xS^1"],
	[[1,35,185,224,232,235,236,241,314],"L(5,1)"],
	[[1,35,185,224,232,235,236,241,325,327],"L(7,1)"],
	[[1,35,185,224,232,236,239,241,258,316],"L(5,1)"],
	[[1,35,185,227,232,322,324,326,328,335],"SFS[RP^2:(2,1)(5,1)]"],
	[[1,35,185,227,232,323,324,330,331,335],"SFS[RP^2:(2,1)(5,1)]"],
	[[1,35,185,231,232,258,315,322,326,335],"Sigma^3"],
	[[1,35,186,198,202,284,289,296,332],"RP^3"],
	[[1,35,186,198,202,291,330,331,335],"Sigma^3"],
	[[1,35,186,219,223,307,312,326,328,335],"SFS[RP^2:(2,1)(5,1)]"],
	[[1,35,190,264,267,275,289,310,311],"S^2~S^1"],
	[[1,35,190,264,267,280,283,310,312],"S^2~S^1"],
	[[1,35,190,273,275,289,297,311],"S^2~S^1"],
	[[1,35,190,273,278,289,301],"S^2~S^1"],
	[[1,35,190,275,280,283,322,326,328,335],"SFS[RP^2:(2,1)(5,1)]"],
	[[1,35,190,275,280,283,323,330,331,335],"SFS[RP^2:(2,1)(5,1)]"],
	[[1,35,190,276,278,286,311,323,327],"SFS[S^2:(4,1)(5,2)(5,-3)]"],
	[[1,35,190,278,280,286,314],"S^3/Q16"],
	[[1,35,190,278,280,286,325,327],"SFS[S^2:(4,1)(5,2)(5,-3)]"],
	[[1,35,190,282,303],"S^3"],
	[[1,35,190,286,299,301],"S^2~S^1"],
	[[1,35,190,288,322,324,327,328,329,330],"SFS[S^2:(3,1)(3,1)(9,-7)]"],
	[[2,3,20,178,203,224,241],"S^3"],
	[[2,3,20,178,207,214,221,233,241,244],"S^2xS^1"],
	[[2,3,20,178,207,220,221,241,244],"L(3,1)"],
	[[2,3,20,178,210,214,217,233,241,245],"S^2xS^1"],
	[[2,3,20,178,210,217,220,241,245],"L(3,1)"],
	[[2,3,20,179,223,271,303],"S^3"],
	[[2,3,20,182,214,223,233,241,253],"S^2xS^1"],
	[[2,3,20,182,220,223,241,253],"L(3,1)"],
	[[2,3,20,183,224,232,241,248],"S^2xS^1"],
	[[2,3,20,183,225,232,297],"S^2~S^1"],
	[[2,3,20,183,234,236,241,247,248],"S^2xS^1"],
	[[2,3,20,183,252,299],"S^2~S^1"],
	[[2,3,20,184,214,220,240,241,242],"L(3,1)"],
	[[2,3,20,184,214,223,233,241,244],"S^2xS^1"],
	[[2,3,20,184,220,223,241,244],"L(3,1)"],
	[[2,3,20,184,225,227,241,242,252],"S^2xS^1"],
	[[2,3,20,184,233,240,241,242],"S^2xS^1"],
	[[2,3,20,185,214,220,224,232,236,241],"L(3,1)"],
	[[2,3,20,185,214,220,234,241,247],"L(3,1)"],
	[[2,3,20,185,224,232,233,236,241],"S^2xS^1"],
	[[2,3,20,185,233,234,241,247],"S^2xS^1"],
	[[2,3,20,186,214,220,224,227,240,241],"L(3,1)"],
	[[2,3,20,186,224,225,241,252],"S^2xS^1"],
	[[2,3,20,186,224,227,233,240,241],"S^2xS^1"],
	[[2,3,20,186,248,252,297],"S^2~S^1"],
	[[2,3,20,187,214,215,233,241,256],"S^2xS^1"],
	[[2,3,20,187,214,217,233,241,247],"S^2xS^1"],
	[[2,3,20,187,215,220,241,256],"L(3,1)"],
	[[2,3,20,187,217,220,241,247],"L(3,1)"],
	[[2,3,20,187,262,265,303],"S^3"],
	[[2,3,20,190,282,303],"S^3"],
	[[2,3,20,190,286,299,301],"S^2~S^1"],
	[[2,3,23,54,165,282,303],"S^3"],
	[[2,3,25,61,121,172,183,227],"S^2~S^1"],
	[[2,3,26,54,96,165,204,303],"S^3"],
	[[2,3,26,54,96,168,179,272,303],"S^3"],
	[[2,3,26,57,165,204,271,272,303],"S^3"],
	[[2,3,26,57,165,207,234,269,272,303],"S^3"],
	[[2,3,26,57,168,179,271,303],"S^3"],
	[[2,3,26,57,168,182,220,241,253],"L(3,1)"],
	[[2,3,26,57,168,184,220,241,244],"L(3,1)"],
	[[2,3,29,57,129,165,207,269,272,303],"S^3"],
	[[2,3,30,61,172,183,299],"S^2~S^1"],
	[[2,3,30,61,172,186,224,225,241],"S^2xS^1"],
	[[2,3,30,61,172,186,248,297],"S^2~S^1"],
	[[2,3,31,62,173,187,214,215,233,241],"S^2xS^1"],
	[[2,3,31,62,173,187,215,220,241],"L(3,1)"],
	[[2,5,20,37,53,203,224,241],"S^3"],
	[[2,5,30,45,67,149,154,172,225],"S^2xS^1"],
	[[2,5,30,47,67,154,172,216],"S^2xS^1"],
	[[2,5,30,47,70,81,95,173,205,241],"S^2~S^1"],
	[[2,5,30,48,70,83,154,172,216],"S^2xS^1"],
	[[2,5,31,46,67,154,173,215,241],"S^2~S^1"],
	[[2,5,31,47,69,82,174,204,233],"S^2xS^1"],
	[[2,5,31,47,70,83,173,205,241],"S^2~S^1"],
	[[2,5,31,48,67,173,205,241],"S^2~S^1"],
	[[2,5,32,47,67,174,204,233],"S^2xS^1"],
	[[2,5,32,48,69,82,173,205,241],"S^2~S^1"],
	[[2,5,32,48,69,84,107,174,205,241],"S^2~S^1"],
	[[2,5,32,48,70,83,174,204,233],"S^2xS^1"],
	[[2,6,20,38,54,223,271,303],"S^3"],
	[[2,6,20,38,57,96,223,272,303],"S^3"],
	[[2,6,20,45,61,81,149,225,252],"S^2xS^1"],
	[[2,6,20,46,61,85,97,215,241,256],"S^2~S^1"],
	[[2,6,20,46,61,85,97,217,241,247],"S^2~S^1"],
	[[2,6,20,46,62,81,215,241,256],"S^2~S^1"],
	[[2,6,20,46,62,81,217,241,247],"S^2~S^1"],
	[[2,6,20,47,61,81,216,252],"S^2xS^1"],
	[[2,6,20,47,62,84,96,215,241,256],"S^2~S^1"],
	[[2,6,20,47,62,84,96,217,241,247],"S^2~S^1"],
	[[2,6,23,38,165,179,282,303],"S^3"],
	[[2,6,23,38,168,204,272,282,303],"S^3"],
	[[2,6,26,38,54,57,168,271,303],"S^3"],
	[[2,6,26,38,96,165,179,204,303],"S^3"],
	[[2,6,26,38,96,168,272,303],"S^3"],
	[[2,6,30,45,81,149,172,225],"S^2xS^1"],
	[[2,6,30,46,83,95,173,215,241],"S^2~S^1"],
	[[2,6,30,47,81,172,216],"S^2xS^1"],
	[[2,6,30,48,83,95,154,173,205,241],"S^2~S^1"],
	[[2,6,31,46,81,173,215,241],"S^2~S^1"],
	[[2,6,31,47,83,95,172,216],"S^2xS^1"],
	[[2,6,31,47,84,96,173,215,241],"S^2~S^1"],
	[[2,6,31,48,67,70,95,172,216],"S^2xS^1"],
	[[2,6,31,48,81,154,173,205,241],"S^2~S^1"],
	[[2,7,20,37,53,68,82,224,241],"S^3"],
	[[2,7,20,45,61,94,225,252],"S^2xS^1"],
	[[2,7,20,47,61,94,149,216,252],"S^2xS^1"],
	[[2,7,30,45,94,172,225],"S^2xS^1"],
	[[2,7,30,47,94,149,172,216],"S^2xS^1"],
	[[2,7,32,48,68,69,175,224,241],"S^2~S^1"],
	[[2,7,32,48,69,70,95,174,224,241],"S^2~S^1"],
	[[2,8,31,47,69,70,173,204,205,241],"S^2~S^1"],
	[[2,8,31,47,82,83,174,233],"S^2xS^1"],
	[[2,8,31,48,67,70,82,174,233],"S^2xS^1"],
	[[2,8,32,47,67,69,83,174,233],"S^2xS^1"],
	[[2,8,32,48,68,69,95,175,233],"S^2xS^1"],
	[[2,8,32,48,69,70,174,233],"S^2xS^1"],
	[[2,8,32,48,82,83,173,204,205,241],"S^2~S^1"],
	[[2,9,20,37,53,68,82,95,107,241],"S^3"],
	[[2,9,20,38,57,83,215,223,272,303],"S^3"],
	[[2,9,20,38,57,83,247,272,303,305],"S^3"],
	[[2,9,20,41,57,220,223,241,253],"L(3,1)"],
	[[2,9,20,46,62,81,83,96,241,256],"S^2~S^1"],
	[[2,9,20,47,62,83,84,241,256],"S^2~S^1"],
	[[2,9,26,38,83,165,179,204,215,303],"S^3"],
	[[2,9,26,38,83,168,215,272,303],"S^3"],
	[[2,9,26,41,165,182,204,271,272,303],"S^3"],
	[[2,9,26,41,168,179,182,271,303],"S^3"],
	[[2,9,26,41,168,217,241,295,306],"L(3,1)"],
	[[2,9,26,41,168,220,241,253],"L(3,1)"],
	[[2,9,26,41,173,236,241,295,306,326],"L(3,1)"],
	[[2,9,30,46,95,96,173,241],"S^2~S^1"],
	[[2,9,30,47,81,84,95,173,241],"S^2~S^1"],
	[[2,9,31,46,81,83,96,173,241],"S^2~S^1"],
	[[2,9,31,46,121,131,173,241,285,287],"L(3,1)"],
	[[2,9,31,47,82,83,107,174,241],"S^2~S^1"],
	[[2,9,31,47,83,84,173,241],"S^2~S^1"],
	[[2,9,31,48,67,70,82,107,174,241],"S^2~S^1"],
	[[2,9,31,48,67,70,84,173,241],"S^2~S^1"],
	[[2,9,32,47,67,69,83,107,174,241],"S^2~S^1"],
	[[2,9,32,48,68,69,95,107,175,241],"S^2~S^1"],
	[[2,9,32,48,69,70,82,84,173,241],"S^2~S^1"],
	[[2,9,32,48,69,70,107,174,241],"S^2~S^1"],
	[[2,11,20,37,61,75,97,98,241,246],"S^2~S^1"],
	[[2,11,20,43,59,220,223,241,244],"L(3,1)"],
	[[2,11,20,46,61,97,98,241,247],"S^2~S^1"],
	[[2,11,20,46,62,81,85,98,241,247],"S^2~S^1"],
	[[2,12,20,37,53,68,82,95,106],"S^2xS^1"],
	[[2,13,20,37,53,68,82,94],"S^2xS^1"],
	[[2,13,20,45,61,224,225,241,252],"S^2xS^1"],
	[[2,13,20,45,61,248,252,297],"S^2~S^1"],
	[[2,13,20,47,61,149,216,224,241,252],"S^2xS^1"],
	[[2,13,30,45,172,183,186,299],"S^2~S^1"],
	[[2,13,30,45,172,224,225,241],"S^2xS^1"],
	[[2,13,30,45,172,248,297],"S^2~S^1"],
	[[2,13,30,45,175,257,283,297],"S^2~S^1"],
	[[2,13,30,47,149,172,216,224,241],"S^2xS^1"],
	[[2,13,30,48,150,175,257,290,297],"S^2~S^1"],
	[[2,14,20,37,53,68,81],"S^2xS^1"],
	[[2,14,20,46,62,214,215,233,241,256],"S^2xS^1"],
	[[2,14,20,46,62,214,217,233,241,247],"S^2xS^1"],
	[[2,14,20,46,62,215,220,241,256],"L(3,1)"],
	[[2,14,20,46,62,217,220,241,247],"L(3,1)"],
	[[2,14,20,46,62,262,265,303],"S^3"],
	[[2,14,25,41,143,144,168,220],"S^2xS^1"],
	[[2,14,25,41,144,145,154,167,220],"S^2xS^1"],
	[[2,14,31,46,173,214,215,233,241],"S^2xS^1"],
	[[2,14,31,46,173,215,220,241],"L(3,1)"],
	[[2,14,31,46,173,241,244,285,287],"L(3,1)"],
	[[2,14,31,46,173,256,262,265,303],"S^3"],
	[[2,14,31,48,154,173,205,214,233,241],"S^2xS^1"],
	[[2,14,31,48,154,173,205,220,241],"L(3,1)"],
	[[2,15,20,37,53,67],"S^2xS^1"],
	[[2,15,30,47,154,172,203,216,224,241],"S^2xS^1"],
	[[2,15,32,47,174,203,204,224,233,241],"S^2xS^1"],
	[[2,16,20,37,52],"S^2xS^1"],
	[[2,17,20,36],"S^2xS^1"],
	[[2,18,19],"S^2xS^1"],
	[[3,4,21,22,38,281,329],"S^3"],
	[[3,4,21,22,42,84,264,329],"S^3"],
	[[3,4,21,26,42,193,264,274,329],"S^3"],
	[[3,4,21,29,45,196,267,298,314,329],"S^3"],
	[[3,4,22,25,42,70,84,180,329],"S^3"],
	[[3,4,25,26,42,70,180,193,274,329],"S^3"],
	[[3,4,37,164,178,203,224,241],"S^3"],
	[[3,4,37,164,183,224,232,241,248],"S^2xS^1"],
	[[3,4,37,164,183,225,232,297],"S^2~S^1"],
	[[3,4,37,164,183,234,236,241,247,248],"S^2xS^1"],
	[[3,4,37,164,183,252,299],"S^2~S^1"],
	[[3,4,37,164,184,225,227,241,242,252],"S^2xS^1"],
	[[3,4,37,164,184,233,240,241,242],"S^2xS^1"],
	[[3,4,37,164,185,224,232,233,236,241],"S^2xS^1"],
	[[3,4,37,164,185,233,234,241,247],"S^2xS^1"],
	[[3,4,37,164,186,224,225,241,252],"S^2xS^1"],
	[[3,4,37,164,186,224,227,233,240,241],"S^2xS^1"],
	[[3,4,37,164,186,248,252,297],"S^2~S^1"],
	[[3,4,37,164,190,282,303],"S^3"],
	[[3,4,37,164,190,286,299,301],"S^2~S^1"],
	[[3,4,37,178,191,203,214,224,233,241],"S^2xS^1"],
	[[3,4,37,178,191,203,220,224,241],"L(3,1)"],
	[[3,4,37,178,191,207,221,241,244],"B2"],
	[[3,4,37,178,191,210,217,241,245],"B2"],
	[[3,4,37,178,194,212,241,254],"B2"],
	[[3,4,37,178,196,212,241,245],"B2"],
	[[3,4,37,178,197,203,218,222,224,241],"L(3,1)"],
	[[3,4,37,178,197,203,224,235,239,241],"B2"],
	[[3,4,37,178,199,203,224,237,239,241],"L(3,1)"],
	[[3,4,37,178,200,205,241,255],"B2"],
	[[3,4,37,178,200,207,241,246],"B2"],
	[[3,4,37,178,201,203,224,241,314,325],"L(15,4)"],
	[[3,4,37,178,201,203,224,241,327],"L(8,3)"],
	[[3,4,37,180,202,224,241,260],"B2"],
	[[3,4,37,182,191,223,241,253],"B2"],
	[[3,4,37,182,202,241,255],"B2"],
	[[3,4,37,183,191,220,224,232,241,248],"SFS[S^2:(2,1)(2,1)(3,1)(3,-2)]"],
	[[3,4,37,183,191,220,225,232,297],"SFS[D_:(3,1)(3,1)]"],
	[[3,4,37,183,191,220,252,299],"SFS[D_:(3,1)(3,1)]"],
	[[3,4,37,184,191,214,240,241,242],"B2"],
	[[3,4,37,184,191,223,241,244],"B2"],
	[[3,4,37,184,200,221,223,241,246],"B2"],
	[[3,4,37,185,191,214,224,232,236,241],"B2"],
	[[3,4,37,185,191,214,234,241,247],"B2"],
	[[3,4,37,186,191,214,224,227,240,241],"B2"],
	[[3,4,37,186,191,220,224,225,241,252],"SFS[S^2:(2,1)(2,1)(3,1)(3,-2)]"],
	[[3,4,37,186,191,220,248,252,297],"SFS[D_:(3,1)(3,1)]"],
	[[3,4,37,187,191,215,241,256],"B2"],
	[[3,4,37,187,191,217,241,247],"B2"],
	[[3,4,37,187,196,210,212,241,247],"B2"],
	[[3,4,37,189,192,224,241,261],"B2"],
	[[3,4,37,189,194,241,256],"B2"],
	[[3,4,37,190,198,228,231,282,303],"SFS[K^2:(3,1)]"],
	[[3,4,37,190,201,282,303,327],"SFS[S^2:(3,2)(4,1)(4,-3)]"],
	[[3,4,39,68,182,200,241,253,274],"HS"],
	[[3,4,39,68,184,200,241,244,274],"HS"],
	[[3,4,39,70,95,180,197,224,241,318],"L(5,1)"],
	[[3,4,39,70,95,180,201,224,241,258],"B2"],
	[[3,4,39,72,97,182,199,241,253,274],"HS"],
	[[3,4,39,72,97,184,199,241,244,274],"HS"],
	[[3,4,40,69,164,189,282,303],"S^3"],
	[[3,4,40,69,164,189,286,299,301],"S^2~S^1"],
	[[3,4,40,69,165,167,192,282,303],"S^3"],
	[[3,4,40,69,186,192,224,241,289,311],"B2"],
	[[3,4,40,69,189,201,282,303,327],"SFS[S^2:(3,2)(4,1)(4,-3)]"],
	[[3,4,40,69,190,192,224,241,261],"B2"],
	[[3,4,40,69,190,194,241,256],"B2"],
	[[3,4,40,71,107,178,194,213,241,257],"B2"],
	[[3,4,40,71,107,190,194,241,259],"B2"],
	[[3,4,40,79,106,117,127,150,161],"S^2~S^1"],
	[[3,4,41,67,92,117,127,144,161],"S^2~S^1"],
	[[3,4,41,70,180,201,224,241,260],"B2"],
	[[3,4,41,70,182,196,241,328,330],"B2"],
	[[3,4,41,70,182,201,241,255],"B2"],
	[[3,4,41,70,190,202,282,303,327],"SFS[S^2:(3,2)(4,1)(4,-3)]"],
	[[3,4,41,76,122,190,202,282,303,328],"SFS[S^2:(3,2)(4,1)(4,-3)]"],
	[[3,4,41,79,117,127,145,161],"S^2~S^1"],
	[[3,4,42,67,81,102,127,137,184,196],"S^2~S^1"],
	[[3,4,42,67,81,104,127,137,161],"S^2~S^1"],
	[[3,4,42,67,92,127,138,161],"S^2~S^1"],
	[[3,4,42,71,178,194,213,241,254],"B2"],
	[[3,4,42,71,178,196,213,241,245],"B2"],
	[[3,4,42,71,187,196,210,213,241,247],"B2"],
	[[3,4,42,71,190,194,241,254,257,259],"B2"],
	[[3,4,42,71,190,196,241,245,257,259],"B2"],
	[[3,4,42,79,127,138,144,145,161],"S^2~S^1"],
	[[3,4,43,67,81,94,115,127,129,161],"S^2~S^1"],
	[[3,4,43,67,81,102,127,130,184,196],"S^2~S^1"],
	[[3,4,43,67,81,104,127,130,161],"S^2~S^1"],
	[[3,4,43,68,97,182,200,241,253,314],"HS"],
	[[3,4,43,68,97,184,200,241,244,314],"HS"],
	[[3,4,43,69,108,164,189,303,311],"S^3"],
	[[3,4,43,69,108,165,167,192,303,311],"S^3"],
	[[3,4,43,72,182,191,222,241,253],"B2"],
	[[3,4,43,72,182,196,241,293,304,314],"HS"],
	[[3,4,43,72,182,199,241,253,314],"HS"],
	[[3,4,43,72,184,191,222,241,244],"B2"],
	[[3,4,43,72,184,199,241,244,314],"HS"],
	[[3,4,43,72,184,200,221,222,241,246],"B2"],
	[[3,4,43,74,136,184,193,241,318],"L(5,1)"],
	[[3,4,44,73,190,198,228,232,282,303],"SFS[K^2:(3,1)]"],
	[[3,4,44,77,95,98,123,185,198,335],"S^2~S^1"],
	[[3,4,44,77,106,115,120,161,335],"S^2~S^1"],
	[[3,4,44,77,109,115,161,314,335],"S^3"],
	[[3,4,44,77,119,123,186,198,335],"S^2~S^1"],
	[[3,4,44,77,119,125,161,335],"S^2~S^1"],
	[[3,4,45,67,81,94,108,127,134,161],"S^2~S^1"],
	[[3,4,45,74,184,197,241,242,314],"L(5,1)"],
	[[3,4,45,74,186,197,224,227,241,314],"L(5,1)"],
	[[3,4,45,77,106,109,123,186,198,335],"S^2~S^1"],
	[[3,4,45,77,106,109,125,161,335],"S^2~S^1"],
	[[3,4,45,77,120,123,186,198,314,335],"S^3"],
	[[3,4,45,77,120,125,161,314,335],"S^3"],
	[[3,4,46,67,81,96,127,141,161],"S^2~S^1"],
	[[3,4,46,67,81,97,127,134,161],"S^2~S^1"],
	[[3,4,46,75,164,185,233,234,241,246],"S^2xS^1"],
	[[3,4,46,75,178,200,207,241,247],"B2"],
	[[3,4,46,75,185,191,214,234,241,246],"B2"],
	[[3,4,46,75,187,191,217,241,246],"B2"],
	[[3,4,46,75,187,200,229,230,335],"S^3"],
	[[3,4,47,67,83,117,127,147,161],"S^2~S^1"],
	[[3,4,47,67,84,127,141,161],"S^2~S^1"],
	[[3,4,47,70,122,190,202,282,293,303],"SFS[S^2:(3,2)(4,1)(4,-3)]"],
	[[3,4,48,69,106,117,127,152,161],"S^2~S^1"],
	[[3,4,48,70,83,84,127,141,161],"S^2~S^1"],
	[[3,4,48,70,117,127,147,161],"S^2~S^1"],
	[[3,4,50,69,106,117,127,152,180,192],"S^2~S^1"],
	[[3,4,50,70,117,127,143,156,180,193],"S^2~S^1"],
	[[3,4,50,70,117,127,147,180,192],"S^2~S^1"],
	[[3,4,50,74,110,113,161,281,329],"S^2~S^1"],
	[[3,4,50,74,110,115,185,197,281,329],"S^2~S^1"],
	[[3,4,51,69,116,179,215,303],"S^3"],
	[[3,4,51,73,98,105,186,236,267],"S^2~S^1"],
	[[3,4,51,73,148,179,219,299,301],"S^2~S^1"],
	[[3,4,51,73,148,186,267,299],"S^2~S^1"],
	[[3,4,51,80,179,282,303],"S^3"],
	[[3,4,51,80,179,286,299,301],"S^2~S^1"],
	[[3,4,51,80,186,219,267,286,299],"S^2~S^1"],
	[[3,5,21,23,38,54,282,303],"S^3"],
	[[3,5,21,26,38,54,96,204,303],"S^3"],
	[[3,5,47,63,67,181,204,282,303],"S^2~S^1"],
	[[3,5,47,63,67,184,204,234,303,309],"S^2~S^1"],
	[[3,5,48,63,70,83,181,204,282,303],"S^2~S^1"],
	[[3,6,36,82,103,115,183,220,248],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[3,6,36,82,103,115,185,214,236],"K^2xS^1"],
	[[3,6,36,86,103,147,189,222,323],"S^2xS^1"],
	[[3,6,36,90,103,113,115,185,218],"S^2xS^1"],
	[[3,6,36,90,103,161,197,218],"S^2xS^1"],
	[[3,6,36,91,104,179,271,303],"S^2~S^1"],
	[[3,6,36,91,104,182,214,233,241,253],"B4"],
	[[3,6,36,91,104,182,220,241,253],"SFS[RP^2:(3,1)(3,2)]"],
	[[3,6,36,91,104,184,214,233,241,244],"B4"],
	[[3,6,36,91,104,184,220,241,244],"SFS[RP^2:(3,1)(3,2)]"],
	[[3,6,36,92,103,183,216,325],"K^2xS^1"],
	[[3,6,36,92,103,183,220,249],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[3,6,37,53,68,182,223,241,253],"B2"],
	[[3,6,37,53,68,184,214,240,241,242],"B2"],
	[[3,6,37,53,68,184,223,241,244],"B2"],
	[[3,6,37,53,68,185,214,234,241,247],"B2"],
	[[3,6,37,53,68,187,215,241,256],"B2"],
	[[3,6,37,53,68,187,217,241,247],"B2"],
	[[3,6,37,53,71,96,188,215,241,256],"B2"],
	[[3,6,37,53,71,96,188,217,241,247],"B2"],
	[[3,6,39,53,184,221,241,242,258,276],"HS"],
	[[3,6,39,53,184,221,241,246,274],"HS"],
	[[3,6,39,59,86,184,222,241,246,274],"HS"],
	[[3,6,42,57,92,104,179,271,303],"S^2~S^1"],
	[[3,6,42,57,92,104,182,220,241,253],"SFS[RP^2:(3,1)(3,2)]"],
	[[3,6,42,57,92,104,184,220,241,244],"SFS[RP^2:(3,1)(3,2)]"],
	[[3,6,43,53,68,72,182,222,241,253],"B2"],
	[[3,6,43,53,68,72,184,222,241,244],"B2"],
	[[3,6,43,53,97,184,221,241,246,314],"HS"],
	[[3,6,43,58,91,103,179,271,303],"S^2~S^1"],
	[[3,6,43,58,91,103,182,220,241,253],"SFS[RP^2:(3,1)(3,2)]"],	
	[[3,6,43,58,91,103,184,220,241,244],"SFS[RP^2:(3,1)(3,2)]"],
	[[3,6,45,61,81,149,183,225,299],"B2"],
	[[3,6,45,61,85,94,108,179,271,335],"B2"],
	[[3,6,46,61,85,97,179,271,335],"B2"],
	[[3,6,46,62,81,179,271,335],"B2"],
	[[3,6,47,56,81,121,188,221,323],"S^2~S^1"],
	[[3,6,47,61,81,183,216,299],"B2"],
	[[3,6,47,61,81,186,216,248,297],"B2"],
	[[3,6,47,62,84,96,179,271,335],"B2"],
	[[3,7,36,82,92,103,183,224,241,249],"B4"],
	[[3,7,36,95,114,123,161,198,227],"T^3"],
	[[3,7,36,95,114,125,186,227],"T^3"],
	[[3,7,36,102,107,133,161,196,227,329],"B2"],
	[[3,7,36,102,114,123,125,186,228],"T^3"],
	[[3,7,36,102,114,161,196,226,329],"B2"],
	[[3,7,36,102,114,161,198,228],"T^3"],
	[[3,7,36,103,115,183,224,241,248],"B4"],
	[[3,7,36,103,115,183,225,297],"B2"],
	[[3,7,36,104,107,133,184,227,329],"B2"],
	[[3,7,36,104,114,184,226,329],"B2"],
	[[3,7,37,54,69,189,224,241,261],"B2"],
	[[3,7,37,56,69,95,189,224,241,259],"B2"],
	[[3,7,38,54,82,190,224,241,262],"B2"],
	[[3,7,38,54,93,116,182,271,303],"S^3"],
	[[3,7,38,57,93,96,116,182,272,303],"S^3"],
	[[3,7,40,54,164,189,192,282,303],"S^3"],
	[[3,7,40,54,165,167,282,303],"S^3"],
	[[3,7,40,54,186,224,241,289,311],"B2"],
	[[3,7,40,54,189,192,201,282,303,327],"SFS[S^2:(3,2)(4,1)(4,-3)]"],
	[[3,7,40,54,190,192,194,241,256],"B2"],
	[[3,7,40,54,190,224,241,261],"B2"],
	[[3,7,40,56,95,190,224,241,259],"B2"],
	[[3,7,40,64,103,180,278,286],"S^2~S^1"],
	[[3,7,43,54,108,164,189,192,303,311],"S^3"],
	[[3,7,43,54,108,165,167,303,311],"S^3"],
	[[3,7,43,58,103,114,184,226,329],"B2"],
	[[3,7,44,59,102,115,161,196,226,329],"B2"],
	[[3,7,44,59,102,115,161,198,228],"T^3"],
	[[3,7,44,59,104,115,184,226,329],"B2"],
	[[3,7,45,61,94,183,225,299],"B2"],
	[[3,7,45,62,97,108,149,179,271,335],"B2"],
	[[3,7,46,62,94,149,179,271,335],"B2"],
	[[3,7,47,61,94,149,183,216,299],"B2"],
	[[3,7,47,64,103,112,180,278,323],"S^2~S^1"],
	[[3,7,48,54,106,117,127,152,161,192],"S^2~S^1"],
	[[3,7,50,54,69,70,117,127,147,180],"S^2~S^1"],
	[[3,7,50,54,106,117,127,149,159,179],"S^2~S^1"],
	[[3,7,50,54,106,117,127,152,180],"S^2~S^1"],
	[[3,7,51,54,116,179,192,215,303],"S^3"],
	[[3,8,21,26,38,54,69,83,96,303],"S^3"],
	[[3,8,21,26,42,55,70,264,274,329],"S^3"],
	[[3,8,22,25,42,55,84,180,193,329],"S^3"],
	[[3,8,24,26,41,54,96,179,272,303],"S^3"],
	[[3,8,24,26,41,57,179,271,303],"S^3"],
	[[3,8,25,26,42,55,180,274,329],"S^3"],
	[[3,8,36,82,90,124,161,199,236],"S^2xS^1"],
	[[3,8,36,95,103,113,161,197,236],"K^2xS^1"],
	[[3,8,36,95,103,115,185,236],"K^2xS^1"],
	[[3,8,36,113,122,158,161,210,237],"S^2xS^1"],
	[[3,8,36,113,124,161,197,235],"K^2xS^1"],
	[[3,8,36,113,124,161,199,237],"S^2xS^1"],
	[[3,8,36,115,124,185,235],"K^2xS^1"],
	[[3,8,41,55,180,193,201,224,241,260],"B2"],
	[[3,8,41,55,180,224,239,241,260,316],"L(5,1)"],
	[[3,8,41,55,182,193,196,241,328,330],"B2"],
	[[3,8,41,55,182,193,201,241,255],"B2"],
	[[3,8,41,55,182,239,241,255,316],"L(5,1)"],
	[[3,8,41,61,111,182,239,241,255,328],"L(5,1)"],
	[[3,8,46,62,106,143,149,179,271,335],"B2"],
	[[3,8,47,63,67,69,83,181,282,303],"S^2~S^1"],
	[[3,8,48,55,83,84,127,141,161,193],"S^2~S^1"],
	[[3,8,48,55,117,127,143,156,161,192],"S^2~S^1"],
	[[3,8,48,55,117,127,147,161,193],"S^2~S^1"],
	[[3,8,48,63,69,70,181,282,303],"S^2~S^1"],
	[[3,8,50,55,117,127,143,156,180],"S^2~S^1"],
	[[3,8,50,55,117,127,147,180,192,193],"S^2~S^1"],
	[[3,9,36,84,91,125,182,291,329],"B2"],
	[[3,9,36,95,104,133,186,242,329],"B2"],
	[[3,9,36,123,133,161,198,243,329],"B2"],
	[[3,9,36,124,134,187,217,220,241],"SFS[RP^2:(3,1)(3,2)]"],
	[[3,9,36,125,133,182,291,329,330],"B2"],
	[[3,9,36,125,133,186,243,329],"B2"],
	[[3,9,37,53,68,83,96,187,241,256],"B2"],
	[[3,9,37,53,69,83,107,188,241,259],"B2"],
	[[3,9,37,53,71,83,188,241,256],"B2"],
	[[3,9,37,54,69,95,107,189,241,261],"B2"],
	[[3,9,37,56,69,107,189,241,259],"B2"],
	[[3,9,37,56,71,178,212,241,254],"B2"],
	[[3,9,37,56,71,189,241,256],"B2"],
	[[3,9,38,54,82,95,107,190,241,262],"B2"],
	[[3,9,39,62,102,122,132,184,241,274],"HS"],
	[[3,9,40,54,95,107,190,241,261],"B2"],
	[[3,9,40,56,69,71,190,241,256],"B2"],
	[[3,9,40,56,107,190,241,259],"B2"],
	[[3,9,42,56,190,241,254,257,259],"B2"],
	[[3,9,46,61,123,134,161,198,243,329],"B2"],
	[[3,9,46,61,124,133,187,217,220,241],"SFS[RP^2:(3,1)(3,2)]"],
	[[3,9,46,61,125,134,182,291,329,330],"B2"],
	[[3,9,46,61,125,134,186,243,329],"B2"],
	[[3,9,47,62,83,84,179,215,271,335],"B2"],
	[[3,9,47,62,95,96,173,174,187,241],"S^2~S^1"],
	[[3,11,36,136,143,149,154,158,161,241],"S^2~S^1"],
	[[3,11,36,140,144,159,184,264,292,303],"S^2~S^1"],
	[[3,11,36,140,144,159,187,253,264,303],"S^2~S^1"],
	[[3,11,36,140,147,187,215,220,241],"SFS[RP^2:(3,1)(3,2)]"],
	[[3,11,37,53,68,85,98,187,241,247],"B2"],
	[[3,11,37,58,73,178,212,241,245],"B2"],
	[[3,11,37,58,73,187,210,212,241,247],"B2"],
	[[3,11,39,57,102,127,145,184,274,322],"L(3,1)"],
	[[3,11,46,61,97,98,179,217,271,335],"B2"],
	[[3,11,47,62,141,147,187,215,220,241],"SFS[RP^2:(3,1)(3,2)]"],
	[[3,11,48,55,107,127,145,189,258],"S^2~S^1"],
	[[3,11,48,57,98,102,127,184,305,328],"S^2~S^1"],
	[[3,11,48,57,127,145,184,305,324],"S^2~S^1"],
	[[3,11,48,57,127,145,189,255],"S^2~S^1"],
	[[3,12,28,30,45,61,183,299],"S^2~S^1"],
	[[3,12,36,84,91,129,147,185,305,329],"B2"],
	[[3,12,36,86,90,99,103,161,197],"S^2xS^1"],
	[[3,12,36,86,92,151,189,258,300],"S^2xS^1"],
	[[3,12,36,98,104,129,133,185,329],"B2"],
	[[3,12,36,129,133,147,185,305,329,330],"B2"],
	[[3,12,36,143,149,154,158,161,233],"S^2xS^1"],
	[[3,12,36,147,151,189,258,327],"S^2xS^1"],
	[[3,12,37,53,69,83,106,188,259],"T^3"],
	[[3,12,37,54,69,95,106,189,261],"T^3"],
	[[3,12,37,56,69,106,189,259],"T^3"],
	[[3,12,38,54,82,95,106,187,271,335],"B2"],
	[[3,12,38,54,82,95,106,190,262],"T^3"],
	[[3,12,40,54,95,106,190,261],"T^3"],
	[[3,12,40,56,106,190,259],"T^3"],
	[[3,12,41,54,95,145,150,190,261],"T^3"],
	[[3,12,41,56,145,150,190,259],"T^3"],
	[[3,12,42,62,129,144,185,295,306,329],"HS"],
	[[3,12,45,52,77,95,98,124,186,335],"S^2~S^1"],
	[[3,12,45,52,77,119,124,185,335],"S^2~S^1"],
	[[3,12,45,59,184,241,242,314],"L(5,1)"],
	[[3,12,45,59,184,241,242,325,327],"L(7,1)"],
	[[3,12,45,59,186,224,227,241,314],"L(5,1)"],
	[[3,12,45,59,186,224,227,241,325,327],"L(7,1)"],
	[[3,12,45,61,75,77,119,123,185,335],"S^2~S^1"],
	[[3,12,45,62,144,182,242,302,307,335],"HS"],
	[[3,12,50,59,110,113,161,197,281,329],"S^2~S^1"],
	[[3,12,50,59,110,115,185,281,329],"S^2~S^1"],
	[[3,13,30,31,47,60,185,308,327],"S^3/P24"],
	[[3,13,36,120,123,130,133,161,198,329],"B2"],
	[[3,13,36,120,125,130,133,186,329],"B2"],
	[[3,13,36,149,154,158,161,224,241],"S^2~S^1"],
	[[3,13,37,54,69,94,189,261],"T^3"],
	[[3,13,37,56,69,94,95,189,259],"T^3"],
	[[3,13,38,54,82,94,187,271,335],"B2"],
	[[3,13,38,54,82,94,190,262],"T^3"],
	[[3,13,38,54,85,94,108,186,271,335],"B2"],
	[[3,13,38,57,82,94,96,187,272,335],"B2"],
	[[3,13,40,54,94,186,289,311],"T^3"],
	[[3,13,40,54,94,190,261],"T^3"],
	[[3,13,40,56,94,95,190,259],"T^3"],
	[[3,13,44,60,75,77,119,123,186,335],"S^2~S^1"],
	[[3,14,36,111,113,122,124,161,199],"S^2xS^1"],
	[[3,14,36,111,113,158,161,210],"S^2xS^1"],
	[[3,14,36,111,115,158,185,197,210],"S^2xS^1"],
	[[3,14,36,137,140,159,187,264,303],"S^2~S^1"],
	[[3,14,36,154,158,161,214,233],"S^2xS^1"],
	[[3,14,36,154,158,161,220],"S^2xS^1"],
	[[3,14,36,155,159,187,265,303],"S^2~S^1"],
	[[3,14,38,54,81,187,271,335],"B2"],
	[[3,14,38,54,81,190,262],"T^3"],
	[[3,14,38,57,81,96,187,272,335],"B2"],
	[[3,14,40,54,81,82,186,289,311],"T^3"],
	[[3,14,40,54,81,82,190,261],"T^3"],
	[[3,14,40,56,81,82,95,190,259],"T^3"],
	[[3,14,43,61,72,76,182,241,253,314],"HS"],
	[[3,14,43,61,72,76,184,241,244,314],"HS"],
	[[3,14,45,61,149,183,220,225,299],"SFS[D_:(3,1)(3,1)]"],
	[[3,14,47,61,172,174,183,299],"S^2~S^1"],
	[[3,14,47,61,172,174,186,224,225,241],"S^2xS^1"],
	[[3,14,47,61,172,174,186,248,297],"S^2~S^1"],
	[[3,14,47,61,183,216,220,299],"SFS[D_:(3,1)(3,1)]"],
	[[3,14,47,61,186,216,220,248,297],"SFS[D_:(3,1)(3,1)]"],
	[[3,14,47,61,186,224,225,241,249,325],"K^2xS^1"],
	[[3,14,47,64,155,180,229,286,323],"S^2~S^1"],
	[[3,15,36,158,161,203,224,241],"S^2~S^1"],
	[[3,15,37,53,67,184,233,240,241,242],"B4"],
	[[3,15,37,53,67,185,233,234,241,247],"B4"],
	[[3,15,37,53,67,190,282,303],"S^2~S^1"],
	[[3,15,39,62,102,184,241,245,274],"HS"],
	[[3,15,40,53,67,69,189,282,303],"S^2~S^1"],
	[[3,15,41,56,144,145,190,257,259],"T^3"],
	[[3,15,43,53,67,69,108,189,303,311],"S^2~S^1"],
	[[3,15,43,62,97,102,184,241,245,314],"HS"],
	[[3,15,44,62,106,115,120,161,200,335],"S^2~S^1"],
	[[3,15,44,62,109,115,161,200,314,335],"S^3"],
	[[3,15,44,62,119,125,161,200,335],"S^2~S^1"],
	[[3,15,45,62,106,109,125,161,200,335],"S^2~S^1"],
	[[3,15,45,62,120,125,161,200,314,335],"S^3"],
	[[3,15,46,62,154,178,215,241,256],"B2"],
	[[3,15,46,62,154,178,217,241,247],"B2"],
	[[3,15,48,55,107,127,138,189,254,258],"S^2~S^1"],
	[[3,15,48,57,127,138,184,254,305,324],"S^2~S^1"],
	[[3,15,48,57,127,138,189,254,255],"S^2~S^1"],
	[[3,15,48,58,127,131,184,254,305,324],"S^2~S^1"],
	[[3,15,48,58,127,131,189,254,255],"S^2~S^1"],
	[[3,15,48,62,178,205,241,256],"B2"],
	[[3,16,36,86,90,147,189,323],"S^2xS^1"],
	[[3,16,36,86,92,145,189,300],"S^2xS^1"],
	[[3,16,36,95,102,125,186,227,231],"T^3"],
	[[3,16,36,95,104,123,186,242,246,329],"B2"],
	[[3,16,36,102,104,184,226,231,329],"B2"],
	[[3,16,36,113,115,185,218,222],"S^2xS^1"],
	[[3,16,36,113,115,185,235,239],"K^2xS^1"],
	[[3,16,36,123,125,186,228,231],"T^3"],
	[[3,16,36,123,125,186,243,246,329],"B2"],
	[[3,16,36,145,147,189,314,325],"S^2xS^1"],
	[[3,16,36,145,147,189,327],"S^2xS^1"],
	[[3,16,36,161,164],"S^2xS^1"],
	[[3,16,36,161,191,214,233],"S^2xS^1"],
	[[3,16,36,161,191,219,228,229],"T^3"],
	[[3,16,36,161,191,220],"S^2xS^1"],
	[[3,16,36,161,196,226,231,329],"B2"],
	[[3,16,36,161,197,218,222],"S^2xS^1"],
	[[3,16,36,161,197,235,239],"K^2xS^1"],
	[[3,16,36,161,198,228,231],"T^3"],
	[[3,16,36,161,198,243,246,329],"B2"],
	[[3,16,36,161,199,237,239],"S^2xS^1"],
	[[3,16,36,161,201,314,325],"S^2xS^1"],
	[[3,16,36,161,201,327],"S^2xS^1"],
	[[3,16,37,52,178,203,224,241],"S^2~S^1"],
	[[3,16,37,52,183,224,232,241,248],"B4"],
	[[3,16,37,52,183,225,232,297],"B2"],
	[[3,16,37,52,183,234,236,241,247,248],"B4"],
	[[3,16,37,52,183,252,299],"B2"],
	[[3,16,37,52,184,225,227,241,242,252],"B4"],
	[[3,16,37,52,184,233,240,241,242],"B4"],
	[[3,16,37,52,185,224,232,233,236,241],"B4"],
	[[3,16,37,52,185,233,234,241,247],"B4"],
	[[3,16,37,52,186,224,225,241,252],"B4"],
	[[3,16,37,52,186,224,227,233,240,241],"B4"],
	[[3,16,37,52,186,248,252,297],"B2"],
	[[3,16,37,52,190,282,303],"S^2~S^1"],
	[[3,16,37,52,190,286,299,301],"B2"],
	[[3,16,40,52,69,189,282,303],"S^2~S^1"],
	[[3,16,40,52,69,189,286,299,301],"B2"],
	[[3,16,43,52,69,108,189,303,311],"S^2~S^1"],
	[[3,16,43,58,102,103,184,226,231,329],"B2"],
	[[3,16,46,52,75,185,233,234,241,246],"B4"],
	[[3,17,36,178,203,224,241],"S^2~S^1"],
	[[3,17,36,178,207,214,221,233,241,244],"B4"],
	[[3,17,36,178,207,220,221,241,244],"SFS[RP^2:(3,1)(3,2)]"],
	[[3,17,36,179,223,271,303],"S^2~S^1"],
	[[3,17,36,182,214,223,233,241,253],"B4"],
	[[3,17,36,182,220,223,241,253],"SFS[RP^2:(3,1)(3,2)]"],
	[[3,17,36,183,224,232,241,248],"B4"],
	[[3,17,36,183,225,232,297],"B2"],
	[[3,17,36,183,234,236,241,247,248],"B4"],
	[[3,17,36,183,252,299],"B2"],
	[[3,17,36,184,214,220,240,241,242],"SFS[RP^2:(3,1)(3,2)]"],
	[[3,17,36,184,214,223,233,241,244],"B4"],
	[[3,17,36,184,220,223,241,244],"SFS[RP^2:(3,1)(3,2)]"],
	[[3,17,36,184,233,240,241,242],"B4"],
	[[3,17,36,185,214,220,224,232,236,241],"SFS[RP^2:(3,1)(3,2)]"],
	[[3,17,36,185,224,232,233,236,241],"B4"],
	[[3,17,36,190,282,303],"S^2~S^1"],
	[[3,17,36,190,286,299,301],"B2"],
	[[4,5,20,22,167,227,259,275,309],"S^3"],
	[[4,5,20,22,167,232,280,309,321],"S^3"],
	[[4,5,20,22,177,203,224,241],"S^3"],
	[[4,5,22,23,58,96,127,277,286],"S^3"],
	[[4,5,22,23,58,103,127,130,155],"S^2~S^1"],
	[[4,5,22,23,59,103,117,121,127,155],"S^2~S^1"],
	[[4,5,22,23,61,97,127,133,155],"S^2~S^1"],
	[[4,5,22,23,61,98,117,124,127,155],"S^2~S^1"],
	[[4,5,22,23,64,97,127,133,275,283],"S^2~S^1"],
	[[4,5,22,23,65,104,274,329],"S^3"],
	[[4,5,22,26,58,127,204,280,304],"S^3"],
	[[4,5,22,26,64,127,136,151,205,274],"S^2~S^1"],
	[[4,5,22,27,58,211,234,238,241,277],"B2"],
	[[4,5,22,27,58,211,241,254,275],"B2"],
	[[4,5,22,27,61,127,133,155,205,283],"S^2~S^1"],
	[[4,5,22,27,64,127,129,151,205,274],"S^2~S^1"],
	[[4,5,22,27,64,127,133,205,275],"S^2~S^1"],
	[[4,5,22,29,58,136,211,238,241,277],"B2"],
	[[4,5,22,30,58,127,130,208,278],"S^3"],
	[[4,5,22,30,59,117,121,127,208,278],"S^3"],
	[[4,5,23,26,65,84,104,193,329],"S^3"],
	[[4,5,36,41,92,117,127,144,158],"S^2~S^1"],
	[[4,5,36,42,81,104,127,137,158],"S^2~S^1"],
	[[4,5,36,42,92,127,138,158],"S^2~S^1"],
	[[4,5,36,43,81,94,115,127,129,158],"S^2~S^1"],
	[[4,5,36,43,81,104,127,130,158],"S^2~S^1"],
	[[4,5,36,45,81,94,108,127,134,158],"S^2~S^1"],
	[[4,5,36,46,81,96,127,141,158],"S^2~S^1"],
	[[4,5,36,46,81,97,127,134,158],"S^2~S^1"],
	[[4,5,36,47,83,117,127,147,158],"S^2~S^1"],
	[[4,5,36,47,84,127,141,158],"S^2~S^1"],
	[[4,5,36,49,83,117,127,147,193,204],"S^2~S^1"],
	[[4,5,36,49,84,127,136,152,193,205],"S^2~S^1"],
	[[4,5,36,49,84,127,141,193,204],"S^2~S^1"],
	[[4,5,38,51,66,281,329],"S^3"],
	[[4,5,39,42,66,84,105,193,329],"S^3"],
	[[4,5,42,51,66,84,264,329],"S^3"],
	[[4,5,53,164,178,184,233,240,241,242],"S^2xS^1"],
	[[4,5,53,164,178,185,233,234,241,247],"S^2xS^1"],
	[[4,5,53,164,178,190,282,303],"S^3"],
	[[4,5,53,164,203,224,241],"S^3"],
	[[4,5,53,164,203,225,248,297],"S^2~S^1"],
	[[4,5,53,178,182,191,223,241,253],"B2"],
	[[4,5,53,178,182,202,241,255],"B2"],
	[[4,5,53,178,184,191,214,240,241,242],"B2"],
	[[4,5,53,178,184,191,223,241,244],"B2"],
	[[4,5,53,178,185,191,214,234,241,247],"B2"],
	[[4,5,53,178,187,191,215,241,256],"B2"],
	[[4,5,53,178,187,191,217,241,247],"B2"],
	[[4,5,53,178,189,194,241,256],"B2"],
	[[4,5,53,178,190,201,282,303,327],"SFS[S^2:(3,2)(4,1)(4,-3)]"],
	[[4,5,53,191,203,214,224,233,241],"S^2xS^1"],
	[[4,5,53,191,203,220,224,241],"L(3,1)"],
	[[4,5,53,191,203,220,225,248,297],"SFS[D_:(3,1)(3,1)]"],
	[[4,5,53,191,207,221,241,244],"B2"],
	[[4,5,53,191,210,217,241,245],"B2"],
	[[4,5,53,194,212,241,254],"B2"],
	[[4,5,53,196,212,241,245],"B2"],
	[[4,5,53,197,203,218,222,224,241],"L(3,1)"],
	[[4,5,53,197,203,224,235,239,241],"B2"],
	[[4,5,53,199,203,224,237,239,241],"L(3,1)"],
	[[4,5,53,200,205,241,255],"B2"],
	[[4,5,53,200,207,241,246],"B2"],
	[[4,5,53,201,203,224,241,314,325],"L(15,4)"],
	[[4,5,53,201,203,224,241,327],"L(8,3)"],
	[[4,5,54,84,96,193,204,313,329],"S^3"],
	[[4,5,54,84,96,196,204,234,310,329],"S^3"],
	[[4,5,54,93,105,192,204,290,297],"S^2~S^1"],
	[[4,5,55,84,107,197,205,241,318],"L(5,1)"],
	[[4,5,55,84,107,201,205,241,258],"B2"],
	[[4,5,56,83,164,181,189,204,282,303],"S^3"],
	[[4,5,56,83,181,190,194,204,241,256],"B2"],
	[[4,5,57,84,201,205,241,255],"B2"],
	[[4,5,57,84,201,207,241,246],"B2"],
	[[4,5,57,89,127,138,196,207],"S^2~S^1"],
	[[4,5,57,91,127,138,158],"S^2~S^1"],
	[[4,5,58,81,103,127,130,158],"S^2~S^1"],
	[[4,5,58,85,194,211,241,254],"B2"],
	[[4,5,58,85,196,211,241,245],"B2"],
	[[4,5,58,89,127,131,196,207],"S^2~S^1"],
	[[4,5,58,91,127,131,158],"S^2~S^1"],
	[[4,5,59,81,103,117,121,127,158],"S^2~S^1"],
	[[4,5,59,83,117,127,197,208,286,287],"S^3"],
	[[4,5,59,86,191,207,222,241,244],"B2"],
	[[4,5,59,86,199,207,241,244,314],"HS"],
	[[4,5,61,81,97,127,133,158],"S^2~S^1"],
	[[4,5,61,81,98,117,124,127,158],"S^2~S^1"],
	[[4,5,62,84,127,140,158],"S^2~S^1"],
	[[4,5,62,85,127,133,158],"S^2~S^1"],
	[[4,5,64,84,127,136,151,193,205],"S^2~S^1"],
	[[4,5,64,84,127,140,194,205],"S^2~S^1"],
	[[4,5,64,84,127,141,151,152,193,204],"S^2~S^1"],
	[[4,5,64,85,127,133,194,205],"S^2~S^1"],
	[[4,5,65,84,134,193,205,234,241,283],"L(5,1)"],
	[[4,5,66,82,116,164,241,253,260],"S^3"],
	[[4,5,66,82,116,201,241,253,260,327],"L(8,3)"],
	[[4,5,66,84,107,116,197,241,274,318],"L(5,1)"],
	[[4,5,66,84,107,116,201,241,258,274],"B2"],
	[[4,5,66,84,135,197,241,242,274,318],"L(5,1)"],
	[[4,5,66,84,135,201,241,242,258,274],"B2"],
	[[4,5,66,84,135,201,241,246,276],"B2"],
	[[4,5,66,89,127,135,138,196,276],"S^2~S^1"],
	[[4,5,66,93,164,224,241,260],"S^3"],
	[[4,5,66,93,191,214,224,233,241,260],"S^2xS^1"],
	[[4,5,66,93,191,220,224,241,260],"L(3,1)"],
	[[4,5,66,93,192,273,290,297],"S^2~S^1"],
	[[4,5,66,93,197,218,222,224,241,260],"L(3,1)"],
	[[4,5,66,93,197,224,235,239,241,260],"B2"],
	[[4,5,66,93,199,224,237,239,241,260],"L(3,1)"],
	[[4,5,66,93,201,224,241,260,314,325],"L(15,4)"],
	[[4,5,66,93,201,224,241,260,327],"L(8,3)"],
	[[4,6,21,22,54,179,281,329],"S^3"],
	[[4,6,21,22,58,97,179,280,329],"S^3"],
	[[4,6,22,24,54,69,273,290,297],"S^2~S^1"],
	[[4,6,38,51,54,69,116,215,303],"S^3"],
	[[4,6,38,51,54,80,282,303],"S^3"],
	[[4,6,52,90,103,197,218],"S^2xS^1"],
	[[4,6,58,80,93,97,202,232,283,329],"S^3"],
	[[4,6,59,73,90,102,197,218],"S^2xS^1"],
	[[4,6,59,80,83,95,148,197,287],"S^2xS^1"],
	[[4,7,20,22,68,82,177,224,241],"S^3"],
	[[4,7,22,25,39,42,55,70,84,329],"S^3"],
	[[4,7,36,44,73,103,115,196,226,329],"B2"],
	[[4,7,36,44,73,103,115,198,228],"T^3"],
	[[4,7,37,39,55,202,224,241,260],"B2"],
	[[4,7,37,41,55,95,202,224,241,258],"B2"],
	[[4,7,37,45,61,94,164,225,252],"S^2xS^1"],
	[[4,7,39,41,55,70,201,224,241,260],"B2"],
	[[4,7,39,50,55,70,117,127,147,192],"S^2~S^1"],
	[[4,7,43,48,52,103,108,192,303,312],"S^2~S^1"],
	[[4,7,52,102,107,133,196,227,329],"B2"],
	[[4,7,52,103,115,161,183,224,241,248],"B4"],
	[[4,7,52,103,115,161,183,225,297],"B2"],
	[[4,7,52,104,107,133,161,184,227,329],"B2"],
	[[4,7,53,68,82,164,224,241],"S^3"],
	[[4,7,53,68,82,164,225,248,297],"S^2~S^1"],
	[[4,7,53,68,82,201,224,241,327],"L(8,3)"],
	[[4,7,53,68,93,116,164,241,253],"S^3"],
	[[4,7,53,68,93,116,201,241,253,327],"L(8,3)"],
	[[4,7,55,70,83,84,96,192,313,329],"S^3"],
	[[4,7,55,70,95,197,224,241,318],"L(5,1)"],
	[[4,7,55,70,95,201,224,241,258],"B2"],
	[[4,7,59,74,94,143,193,318],"S^2xS^1"],
	[[4,7,59,74,94,147,156,192,318],"S^2xS^1"],
	[[4,7,59,77,102,115,147,192,310,335],"S^3"],
	[[4,7,61,74,94,192,287,289,314],"S^2xS^1"],
	[[4,7,61,74,94,197,227,314],"S^2xS^1"],
	[[4,7,65,74,94,143,147,193,287],"S^2xS^1"],
	[[4,7,65,74,94,156,192,287],"S^2xS^1"],
	[[4,7,65,74,94,156,197,227,289],"S^2xS^1"],
	[[4,8,22,26,58,69,83,127,280,304],"S^3"],
	[[4,8,22,32,52,82,124,236,280],"S^2xS^1"],
	[[4,8,36,47,69,117,127,147,158,204],"S^2~S^1"],
	[[4,8,36,49,69,83,84,127,141,193],"S^2~S^1"],
	[[4,8,36,49,69,117,127,143,156,192],"S^2~S^1"],
	[[4,8,36,49,69,117,127,147,193],"S^2~S^1"],
	[[4,8,39,50,54,69,117,127,147,193],"S^2~S^1"],
	[[4,8,40,41,65,79,193,327],"S^2xS^1"],
	[[4,8,40,47,65,79,122,193,293],"S^2xS^1"],
	[[4,8,52,82,90,124,199,236],"S^2xS^1"],
	[[4,8,52,95,103,113,197,236],"K^2xS^1"],
	[[4,8,52,95,103,115,161,185,236],"K^2xS^1"],
	[[4,8,52,113,122,158,204,293,295],"S^2xS^1"],
	[[4,8,52,113,122,158,210,237],"S^2xS^1"],
	[[4,8,52,113,124,197,235],"K^2xS^1"],
	[[4,8,52,113,124,199,237],"S^2xS^1"],
	[[4,8,52,115,124,161,185,235],"K^2xS^1"],
	[[4,8,54,69,83,84,96,193,313,329],"S^3"],
	[[4,8,56,69,164,181,189,282,303],"S^3"],
	[[4,8,56,69,165,167,181,192,282,303],"S^3"],
	[[4,8,56,69,181,190,192,224,241,261],"B2"],
	[[4,8,56,69,181,190,194,241,256],"B2"],
	[[4,8,59,73,111,121,193,318],"S^2xS^1"],
	[[4,8,59,73,115,121,156,193,310],"S^2xS^1"],
	[[4,8,59,74,106,193,318],"S^2xS^1"],
	[[4,8,61,75,113,123,197,235],"K^2xS^1"],
	[[4,8,61,75,113,123,199,237],"S^2xS^1"],
	[[4,8,61,75,115,123,161,185,235],"K^2xS^1"],
	[[4,8,65,73,111,121,147,193,287],"S^2xS^1"],
	[[4,8,65,74,106,143,156,192,287],"S^2xS^1"],
	[[4,8,65,74,106,147,193,287],"S^2xS^1"],
	[[4,9,20,22,68,82,95,107,177,241],"S^3"],
	[[4,9,23,26,65,70,104,193,205,329],"S^3"],
	[[4,9,36,46,75,124,134,198,243,329],"B2"],
	[[4,9,36,47,70,127,136,152,158,204],"S^2~S^1"],
	[[4,9,36,47,70,127,141,158,205],"S^2~S^1"],
	[[4,9,36,49,70,127,136,152,193],"S^2~S^1"],
	[[4,9,36,49,70,127,141,193,204,205],"S^2~S^1"],
	[[4,9,37,39,55,95,107,202,241,260],"B2"],
	[[4,9,37,41,55,107,202,241,258],"B2"],
	[[4,9,37,41,57,191,223,241,253],"B2"],
	[[4,9,37,41,57,202,241,255],"B2"],
	[[4,9,38,51,54,69,83,96,116,303],"S^3"],
	[[4,9,39,41,57,68,200,241,253,274],"HS"],
	[[4,9,39,42,66,70,105,193,205,329],"S^3"],
	[[4,9,39,50,54,70,127,136,152,193],"S^2~S^1"],
	[[4,9,42,51,55,70,116,264,274,329],"S^3"],
	[[4,9,42,51,66,70,205,264,329],"S^3"],
	[[4,9,52,95,104,133,161,186,242,329],"B2"],
	[[4,9,52,123,133,198,243,329],"B2"],
	[[4,9,52,125,133,161,186,243,329],"B2"],
	[[4,9,53,68,82,95,107,164,241],"S^3"],
	[[4,9,53,70,84,200,241,255],"B2"],
	[[4,9,54,70,96,193,204,205,313,329],"S^3"],
	[[4,9,54,70,104,129,152,196,241,310],"L(5,1)"],
	[[4,9,55,70,107,197,241,318],"L(5,1)"],
	[[4,9,55,70,107,201,241,258],"B2"],
	[[4,9,55,70,111,130,196,241,318],"L(5,1)"],
	[[4,9,55,70,115,130,156,196,241,310],"L(5,1)"],
	[[4,9,57,70,196,241,328,330],"B2"],
	[[4,9,57,70,201,241,255],"B2"],
	[[4,9,58,70,127,194,323,331],"S^2~S^1"],
	[[4,9,58,75,121,127,194,324,331],"S^2~S^1"],
	[[4,9,58,75,121,127,202,247,292,331],"S^2~S^1"],
	[[4,9,58,79,121,127,156,194,292,313],"L(3,1)"],
	[[4,9,61,70,130,196,241,283,289,310],"B2"],
	[[4,9,61,70,130,196,241,325,328],"B2"],
	[[4,9,61,75,124,133,198,243,329],"B2"],
	[[4,9,61,76,122,130,196,241,314],"HS"],
	[[4,9,61,76,122,134,156,196,241,313],"HS"],
	[[4,9,62,70,127,140,158,205],"S^2~S^1"],
	[[4,9,64,70,127,136,151,193],"S^2~S^1"],
	[[4,9,64,70,127,140,194],"S^2~S^1"],
	[[4,9,65,70,134,193,234,241,283],"L(5,1)"],
	[[4,9,65,70,134,196,241,283,310,313],"L(5,1)"],
	[[4,9,65,76,122,130,156,196,241,289],"HS"],
	[[4,11,37,43,59,191,223,241,244],"B2"],
	[[4,11,37,43,59,200,221,223,241,246],"B2"],
	[[4,11,37,46,61,97,98,164,241,247],"S^2~S^1"],
	[[4,11,39,43,59,68,200,241,244,274],"HS"],
	[[4,11,52,136,143,149,154,158,241],"S^2~S^1"],
	[[4,11,53,68,86,97,199,241,246],"B2"],
	[[4,11,53,72,86,191,221,241,244],"B2"],
	[[4,11,53,72,86,200,241,246],"B2"],
	[[4,11,59,67,81,102,127,130,196],"S^2~S^1"],
	[[4,11,59,67,81,104,127,130,161,184],"S^2~S^1"],
	[[4,11,59,68,97,200,241,244,314],"HS"],
	[[4,11,59,72,191,222,241,244],"B2"],
	[[4,11,59,72,196,277,278,292,293,335],"S^3"],
	[[4,11,59,72,199,241,244,273,278,302],"SFS[A:(2,1)(2,1)]/[1,-11|1,-10]"],
	[[4,11,59,72,199,241,244,314],"HS"],
	[[4,11,59,72,200,215,221,241,298,300],"B2"],
	[[4,11,59,72,200,221,222,241,246],"B2"],
	[[4,11,59,74,136,193,241,318],"L(5,1)"],
	[[4,11,59,80,142,192,215,227,241,289],"L(3,1)"],
	[[4,11,59,80,142,197,215,241,287],"L(3,1)"],
	[[4,11,61,75,97,98,164,241,246],"S^2~S^1"],
	[[4,11,62,72,144,199,253,302,307,335],"S^3"],
	[[4,11,65,74,136,143,156,192,241,287],"L(5,1)"],
	[[4,11,65,74,136,147,193,241,287],"L(5,1)"],
	[[4,11,66,80,136,143,149,155,158,241],"S^2~S^1"],
	[[4,12,20,22,68,82,95,106,177],"S^2xS^1"],
	[[4,12,28,30,52,149,154,158,225],"S^2xS^1"],
	[[4,12,37,39,55,95,106,202,260],"K^2xS^1"],
	[[4,12,37,41,55,106,202,258],"K^2xS^1"],
	[[4,12,52,86,90,99,103,197],"S^2xS^1"],
	[[4,12,52,98,104,129,133,161,185,329],"B2"],
	[[4,12,52,143,149,154,158,233],"S^2xS^1"],
	[[4,12,52,146,150,202,259,282,303],"S^2~S^1"],
	[[4,12,53,68,82,95,106,164],"S^2xS^1"],
	[[4,12,53,68,83,95,148,153,164],"S^2xS^1"],
	[[4,12,54,69,94,147,156,197,318],"S^2xS^1"],
	[[4,12,54,69,94,147,156,201,258],"K^2xS^1"],
	[[4,12,54,73,104,152,196,241,242,312],"L(5,1)"],
	[[4,12,55,69,147,152,197,318],"S^2xS^1"],
	[[4,12,55,69,147,152,201,258],"K^2xS^1"],
	[[4,12,55,70,106,197,318],"S^2xS^1"],
	[[4,12,55,70,106,201,258],"K^2xS^1"],
	[[4,12,61,75,98,99,197,314],"S^2xS^1"],
	[[4,12,65,75,98,99,156,197,289],"S^2xS^1"],
	[[4,12,66,80,143,149,155,158,233],"S^2xS^1"],
	[[4,13,20,22,68,82,94,177],"S^2xS^1"],
	[[4,13,29,31,52,154,158,215,241],"S^2~S^1"],
	[[4,13,37,39,55,94,202,260],"K^2xS^1"],
	[[4,13,37,41,55,94,95,202,258],"K^2xS^1"],
	[[4,13,37,45,61,164,224,225,241,252],"S^2xS^1"],
	[[4,13,37,45,61,164,248,252,297],"S^2~S^1"],
	[[4,13,39,41,55,70,94,201,260],"K^2xS^1"],
	[[4,13,44,45,61,77,119,123,198,335],"S^2~S^1"],
	[[4,13,52,130,133,150,198,289,296,329],"B2"],
	[[4,13,52,149,154,158,224,241],"S^2~S^1"],
	[[4,13,52,149,154,158,225,248,297],"B2"],
	[[4,13,52,150,155,198,276,290,308],"T^3"],
	[[4,13,52,150,155,198,279,294],"T^3"],
	[[4,13,52,150,155,198,281,296,329],"B2"],
	[[4,13,53,68,82,94,164],"S^2xS^1"],
	[[4,13,53,68,82,94,201,327],"S^2xS^1"],
	[[4,13,53,69,82,153,157,164],"S^2xS^1"],
	[[4,13,53,69,82,153,157,201,327],"S^2xS^1"],
	[[4,13,53,69,83,106,148,157,164],"S^2xS^1"],
	[[4,13,54,70,95,152,156,197,318],"S^2xS^1"],
	[[4,13,54,70,95,152,156,201,258],"K^2xS^1"],
	[[4,13,55,70,94,95,197,318],"S^2xS^1"],
	[[4,13,55,70,94,95,201,258],"K^2xS^1"],
	[[4,13,58,74,127,130,197,208,209],"S^3"],
	[[4,13,59,74,143,193,224,241,318],"L(5,1)"],
	[[4,13,59,74,147,156,192,224,241,318],"L(5,1)"],
	[[4,13,61,74,184,186,197,241,242,314],"L(5,1)"],
	[[4,13,61,74,192,224,241,287,289,314],"L(5,1)"],
	[[4,13,61,74,197,224,227,241,314],"L(5,1)"],
	[[4,13,61,77,106,109,125,161,186,335],"S^2~S^1"],
	[[4,13,61,77,150,198,289,296,314,335],"S^3"],
	[[4,13,61,79,129,134,164,260,275,303],"S^3"],
	[[4,13,61,79,152,202,261,301,302],"S^2~S^1"],
	[[4,13,65,74,143,147,193,224,241,287],"L(5,1)"],
	[[4,13,65,74,156,192,224,241,287],"L(5,1)"],
	[[4,13,65,74,156,197,224,227,241,289],"L(5,1)"],
	[[4,13,65,77,150,156,198,296,335],"S^3"],
	[[4,13,66,74,127,130,142,197,209,277],"S^3"],
	[[4,13,66,80,149,155,158,224,241],"S^2~S^1"],
	[[4,14,20,22,68,81,177],"S^2xS^1"],
	[[4,14,28,32,52,143,158,204,225],"S^2xS^1"],
	[[4,14,30,32,52,158,204,233],"S^2xS^1"],
	[[4,14,37,46,62,191,215,241,256],"B2"],
	[[4,14,37,46,62,191,217,241,247],"B2"],
	[[4,14,52,111,113,122,124,199],"S^2xS^1"],
	[[4,14,52,111,113,158,210],"S^2xS^1"],
	[[4,14,52,154,158,214,233],"S^2xS^1"],
	[[4,14,52,154,158,220],"S^2xS^1"],
	[[4,14,53,68,81,164],"S^2xS^1"],
	[[4,14,53,68,81,201,290,291,308],"T^3"],
	[[4,14,53,68,81,201,327],"S^2xS^1"],
	[[4,14,54,68,157,160,164],"S^2xS^1"],
	[[4,14,54,68,157,160,201,327],"S^2xS^1"],
	[[4,14,54,69,94,153,160,164],"S^2xS^1"],
	[[4,14,54,69,94,153,160,201,327],"S^2xS^1"],
	[[4,14,57,75,127,138,196,207,210],"S^2~S^1"],
	[[4,14,57,75,127,138,196,253,294,305],"S^2~S^1"],
	[[4,14,58,75,127,131,196,207,210],"S^2~S^1"],
	[[4,14,58,75,127,131,196,253,294,305],"S^2~S^1"],
	[[4,14,59,73,121,122,191,222],"K^2xS^1"],
	[[4,14,59,73,121,122,199,314],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[4,14,62,72,137,199,302,307,335],"S^3"],
	[[4,14,62,75,191,217,241,246],"B2"],
	[[4,14,62,75,199,300,302,305,307,335],"S^3"],
	[[4,14,66,74,153,155,158,287],"S^2xS^1"],
	[[4,14,66,75,127,135,138,196,210,276],"S^2~S^1"],
	[[4,14,66,80,155,158,214,233],"S^2xS^1"],
	[[4,14,66,80,155,158,220],"S^2xS^1"],
	[[4,15,20,22,67,177],"S^2xS^1"],
	[[4,15,29,33,52,149,192,215,241],"S^2~S^1"],
	[[4,15,31,32,54,68,192,224,241],"S^2~S^1"],
	[[4,15,31,33,52,192,224,241],"S^2~S^1"],
	[[4,15,52,158,178,184,233,240,241,242],"B4"],
	[[4,15,52,158,178,185,233,234,241,247],"B4"],
	[[4,15,52,158,178,190,282,303],"S^2~S^1"],
	[[4,15,52,158,203,224,241],"S^2~S^1"],
	[[4,15,52,158,203,225,248,297],"B2"],
	[[4,15,53,67,164],"S^2xS^1"],
	[[4,15,53,67,191,214,233],"S^2xS^1"],
	[[4,15,53,67,191,220],"S^2xS^1"],
	[[4,15,53,67,197,218,222],"S^2xS^1"],
	[[4,15,53,67,197,235,239],"K^2xS^1"],
	[[4,15,53,67,199,237,239],"S^2xS^1"],
	[[4,15,53,67,201,290,291,308],"T^3"],
	[[4,15,53,67,201,314,325],"S^2xS^1"],
	[[4,15,53,67,201,327],"S^2xS^1"],
	[[4,15,59,67,86,197,218,221],"S^2xS^1"],
	[[4,15,59,67,86,197,257,287,300],"S^2xS^1"],
	[[4,16,52,164],"S^2xS^1"],
	[[4,16,52,191,214,233],"S^2xS^1"],
	[[4,16,52,191,219,228,229],"T^3"],
	[[4,16,52,191,220],"S^2xS^1"],
	[[4,16,52,192,231,279,281,329],"B2"],
	[[4,16,52,192,231,290,291,310],"T^3"],
	[[4,16,52,192,231,294,296],"T^3"],
	[[4,16,52,196,226,231,329],"B2"],
	[[4,16,52,197,218,222],"S^2xS^1"],
	[[4,16,52,197,235,239],"K^2xS^1"],
	[[4,16,52,198,228,231],"T^3"],
	[[4,16,52,201,273,278,302,325],"T^3"],
	[[4,16,52,201,276,279,291,294],"T^3"],
	[[4,16,52,201,276,281,291,296,329],"B2"],
	[[4,16,52,201,277,278,303,304,325],"B2"],
	[[4,16,52,201,290,291,308],"T^3"],
	[[4,16,52,201,314,325],"S^2xS^1"],
	[[4,16,52,201,327],"S^2xS^1"],
	[[5,6,21,23,179,282,303],"S^3"],
	[[5,6,21,26,96,179,204,303],"S^3"],
	[[5,6,23,24,40,41,56,57,71,335],"S^3"],
	[[5,6,23,24,55,65,70,125,291],"S^3"],
	[[5,6,23,27,72,191,199,241,255,283],"B2"],
	[[5,6,23,27,72,222,241,255,283,314],"HS"],
	[[5,6,23,29,74,215,241,287],"L(3,1)"],
	[[5,6,36,42,52,104,127,137,154],"S^2~S^1"],
	[[5,6,36,43,52,94,115,127,129,154],"S^2~S^1"],
	[[5,6,36,43,52,104,127,130,154],"S^2~S^1"],
	[[5,6,36,45,52,94,108,127,134,154],"S^2~S^1"],
	[[5,6,36,46,52,96,127,141,154],"S^2~S^1"],
	[[5,6,36,46,52,97,127,134,154],"S^2~S^1"],
	[[5,6,36,48,52,96,127,141,205,215],"S^2~S^1"],
	[[5,6,36,48,52,97,127,134,205,215],"S^2~S^1"],
	[[5,6,37,39,205,223,241,255,274],"HS"],
	[[5,6,37,39,207,223,241,242,258,276],"HS"],
	[[5,6,37,39,207,223,241,246,274],"HS"],
	[[5,6,37,43,97,205,223,241,255,314],"HS"],
	[[5,6,37,43,97,207,223,241,246,314],"HS"],
	[[5,6,37,44,104,119,125,212,223,335],"S^2~S^1"],
	[[5,6,39,43,72,205,222,241,255,274],"HS"],
	[[5,6,39,43,72,207,222,241,246,274],"HS"],
	[[5,6,39,50,75,156,207,217,335],"S^2~S^1"],
	[[5,6,39,50,77,154,156,335],"S^2~S^1"],
	[[5,6,52,58,103,127,130,154],"S^2~S^1"],
	[[5,6,52,59,103,117,121,127,154],"S^2~S^1"],
	[[5,6,52,61,97,127,133,154],"S^2~S^1"],
	[[5,6,52,61,98,117,124,127,154],"S^2~S^1"],
	[[5,6,54,55,69,205,215,313,329],"S^3"],
	[[5,6,54,58,69,108,205,215,312,329],"S^3"],
	[[5,6,55,65,80,104,116,205,329],"S^3"],
	[[5,6,58,59,73,102,127,130,154],"S^2~S^1"],
	[[5,6,58,66,80,96,127,277,286],"S^3"],
	[[5,6,58,66,80,103,127,130,155],"S^2~S^1"],
	[[5,6,59,66,80,103,117,121,127,155],"S^2~S^1"],
	[[5,6,61,66,80,97,127,133,155],"S^2~S^1"],
	[[5,6,61,66,80,98,117,124,127,155],"S^2~S^1"],
	[[5,6,64,66,80,97,127,133,275,283],"S^2~S^1"],
	[[5,6,65,66,80,104,274,329],"S^3"],
	[[5,6,68,164,191,203,224,241],"S^3"],
	[[5,6,68,164,191,203,225,248,297],"S^2~S^1"],
	[[5,6,68,167,169,214,233,241,254],"S^2xS^1"],
	[[5,6,68,167,169,220,241,254],"L(3,1)"],
	[[5,6,68,168,170,203,224,241,249],"S^2xS^1"],
	[[5,6,68,172,174,203,216,224,241],"S^2xS^1"],
	[[5,6,68,173,175,205,214,233,241],"S^2xS^1"],
	[[5,6,68,173,175,205,220,241],"L(3,1)"],
	[[5,6,68,178,182,191,202,241,255],"B2"],
	[[5,6,68,178,182,223,241,253],"B2"],
	[[5,6,68,178,184,214,240,241,242],"B2"],
	[[5,6,68,178,184,223,241,244],"B2"],
	[[5,6,68,178,185,214,234,241,247],"B2"],
	[[5,6,68,178,187,215,241,256],"B2"],
	[[5,6,68,178,187,217,241,247],"B2"],
	[[5,6,68,178,189,191,194,241,256],"B2"],
	[[5,6,68,187,189,212,215,241,254],"B2"],
	[[5,6,68,191,194,212,241,254],"B2"],
	[[5,6,68,191,196,212,241,245],"B2"],
	[[5,6,68,191,200,205,241,255],"B2"],
	[[5,6,68,191,200,207,241,246],"B2"],
	[[5,6,68,191,201,203,224,241,327],"L(8,3)"],
	[[5,6,68,200,202,205,223,241,253],"B2"],
	[[5,6,68,203,214,224,233,241],"S^2xS^1"],
	[[5,6,68,203,214,224,235,241,314],"L(5,1)"],
	[[5,6,68,203,216,224,241,249,325],"K^2xS^1"],
	[[5,6,68,203,220,224,241],"L(3,1)"],
	[[5,6,68,203,220,225,248,297],"SFS[D_:(3,1)(3,1)]"],
	[[5,6,68,207,221,241,244],"B2"],
	[[5,6,68,210,217,241,245],"B2"],
	[[5,6,68,211,220,273,281,299,329],"SFS[S^2:(3,1)(3,1)(5,-3)]"],
	[[5,6,70,95,173,174,205,220,241],"L(3,1)"],
	[[5,6,72,97,191,199,205,241,255],"B2"],
	[[5,6,72,97,191,199,207,241,246],"B2"],
	[[5,6,72,97,199,202,205,223,241,253],"B2"],
	[[5,6,72,97,205,222,241,255,314],"HS"],
	[[5,6,72,97,207,215,241,298,300,314],"HS"],
	[[5,6,72,97,207,222,241,246,314],"HS"],
	[[5,6,73,98,205,217,241,326,328],"B2"],
	[[5,6,73,98,210,220,273,281,299,329],"SFS[S^2:(3,1)(3,1)(5,-3)]"],
	[[5,6,73,98,211,217,241,245],"B2"],
	[[5,6,73,102,117,121,127,154],"S^2~S^1"],
	[[5,6,73,103,146,210,220,332],"S^3/P24"],
	[[5,6,74,99,211,221,273,281,299,329],"SFS[S^2:(3,1)(3,1)(5,-3)]"],
	[[5,6,75,97,123,124,127,133,154],"S^2~S^1"],
	[[5,6,75,98,117,123,127,154],"S^2~S^1"],
	[[5,6,78,98,146,210,220,328],"S^3/P24"],
	[[5,6,79,104,167,169,214,233,241,253],"S^2xS^1"],
	[[5,6,79,104,167,169,220,241,253],"L(3,1)"],
	[[5,6,79,104,178,182,223,241,254],"B2"],
	[[5,6,79,104,191,194,212,241,253],"B2"],
	[[5,6,80,95,117,127,148,208,286],"S^3"],
	[[5,6,80,96,127,142,208,286],"S^3"],
	[[5,6,80,98,146,163,165,289,332],"S^3"],
	[[5,6,80,103,133,135,205,241,289,312],"B2"],
	[[5,6,80,104,115,116,205,283,314,329],"S^3"],
	[[5,6,80,105,167,169,233,241,254,257],"S^2xS^1"],
	[[5,6,80,105,173,175,205,233,241,257],"S^2xS^1"],
	[[5,6,80,105,203,224,233,241,257],"S^2xS^1"],
	[[5,6,80,105,203,224,235,241,257,314],"L(5,1)"],
	[[5,6,80,105,205,234,241,283,298,314],"L(5,1)"],
	[[5,7,54,66,69,93,273,290,297],"S^2~S^1"],
	[[5,7,69,84,96,192,193,204,313,329],"S^3"],
	[[5,7,69,93,105,204,290,297],"S^2~S^1"],
	[[5,7,79,93,105,115,204,297,311],"S^2~S^1"],
	[[5,8,23,26,55,65,70,84,104,329],"S^3"],
	[[5,8,36,49,55,70,84,127,141,204],"S^2~S^1"],
	[[5,8,39,42,55,66,70,84,105,329],"S^3"],
	[[5,8,40,47,56,63,67,204,282,303],"S^2~S^1"],
	[[5,8,41,47,67,113,204,295,327],"S^2xS^1"],
	[[5,8,54,55,70,84,96,204,313,329],"S^3"],
	[[5,8,67,113,122,204,293,295],"S^2xS^1"],
	[[5,8,67,113,122,210,237],"S^2xS^1"],
	[[5,8,67,113,124,158,197,235],"K^2xS^1"],
	[[5,8,67,113,124,158,199,237],"S^2xS^1"],
	[[5,8,67,115,124,158,161,185,235],"K^2xS^1"],
	[[5,8,70,92,104,105,116,205,327,329],"S^3/P24"],
	[[5,8,70,92,115,205,234,241,298,327],"L(7,1)"],
	[[5,8,73,82,117,127,210,236],"S^3"],
	[[5,8,73,92,106,152,204,299,328],"T^3"],
	[[5,9,21,26,83,179,204,215,303],"S^3"],
	[[5,9,23,29,74,83,96,241,287],"L(3,1)"],
	[[5,9,36,46,52,83,127,141,154,215],"S^2~S^1"],
	[[5,9,36,48,52,83,127,136,152,204],"S^2~S^1"],
	[[5,9,36,48,52,83,127,141,205],"S^2~S^1"],
	[[5,9,36,49,55,69,83,127,141,205],"S^2~S^1"],
	[[5,9,53,56,71,178,189,241,256],"B2"],
	[[5,9,53,56,71,212,241,254],"B2"],
	[[5,9,54,55,69,83,96,205,313,329],"S^3"],
	[[5,9,58,66,69,83,116,127,280,304],"S^3"],
	[[5,9,58,66,80,83,127,215,277,286],"S^3"],
	[[5,9,68,83,96,178,187,241,256],"B2"],
	[[5,9,68,83,96,187,189,212,241,254],"B2"],
	[[5,9,69,83,107,208,241,273,280,302],"B2"],
	[[5,9,69,83,107,208,241,299,301],"B2"],
	[[5,9,71,83,184,189,212,241,304,321],"B2"],
	[[5,9,74,83,129,205,283,287,304,329],"HS"],
	[[5,9,75,83,130,207,241,304,305,314],"HS"],
	[[5,9,75,91,124,130,207,241,314],"HS"],
	[[5,9,75,91,124,134,156,207,241,313],"HS"],
	[[5,9,80,83,127,142,208,215,286],"S^3"],
	[[5,11,22,23,73,96,127,196,277,286],"S^3"],
	[[5,11,22,23,73,103,127,130,155,196],"S^2~S^1"],
	[[5,11,22,26,73,127,196,204,280,304],"S^3"],
	[[5,11,22,30,73,127,130,196,208,278],"S^3"],
	[[5,11,39,50,75,85,98,156,207,335],"S^2~S^1"],
	[[5,11,52,66,93,136,143,150,155,241],"S^2~S^1"],
	[[5,11,53,58,73,212,241,245],"B2"],
	[[5,11,67,136,143,149,154,241],"S^2~S^1"],
	[[5,11,67,136,145,149,167,168,241],"S^2~S^1"],
	[[5,11,67,138,143,154,168,169,241],"S^2~S^1"],
	[[5,11,67,138,145,167,169,241],"S^2~S^1"],
	[[5,11,68,85,98,178,187,241,247],"B2"],
	[[5,11,68,85,98,210,241,245],"B2"],
	[[5,11,73,81,103,127,130,158,196],"S^2~S^1"],
	[[5,11,73,84,127,207,322,323],"S^2~S^1"],
	[[5,11,73,85,205,241,326,328],"B2"],
	[[5,11,73,85,211,241,245],"B2"],
	[[5,11,73,89,127,131,207],"S^2~S^1"],
	[[5,11,73,91,127,131,158,196],"S^2~S^1"],
	[[5,11,73,92,136,152,204,241,299,328],"B2"],
	[[5,11,73,92,141,205,241,299,328],"B2"],
	[[5,11,79,81,104,137,144,167,169,241],"S^2~S^1"],
	[[5,11,79,84,127,147,207,294,322],"S^2~S^1"],
	[[5,11,79,84,127,147,210,253,305,322],"S^2~S^1"],
	[[5,11,79,92,136,144,149,167,168,241],"S^2~S^1"],
	[[5,11,79,92,138,144,167,169,241],"S^2~S^1"],
	[[5,11,80,93,136,143,150,154,241],"S^2~S^1"],
	[[5,11,80,93,136,145,150,167,168,241],"S^2~S^1"],
	[[5,12,28,30,45,47,67,154,216],"S^2xS^1"],
	[[5,12,28,30,67,149,154,225],"S^2xS^1"],
	[[5,12,52,66,93,143,150,155,233],"S^2xS^1"],
	[[5,12,67,143,149,154,233],"S^2xS^1"],
	[[5,12,67,143,149,154,235,314],"S^2xS^1"],
	[[5,12,67,144,150,208,273,281,286,329],"S^2~S^1"],
	[[5,12,67,145,149,167,168,233],"S^2xS^1"],
	[[5,12,69,83,106,208,273,280,302],"T^3"],
	[[5,12,69,83,106,208,277,280,303,304],"B2"],
	[[5,12,69,83,106,208,282,286,303],"B2"],
	[[5,12,69,83,106,208,299,301],"T^3"],
	[[5,12,70,83,147,152,208,282,286,303],"B2"],
	[[5,12,70,83,147,152,208,299,301],"T^3"],
	[[5,12,74,83,117,127,208,286,287],"S^3"],
	[[5,12,74,84,127,141,147,208,286,287],"S^3"],
	[[5,12,79,92,144,149,167,168,233],"S^2xS^1"],
	[[5,12,80,93,143,150,154,233],"S^2xS^1"],
	[[5,12,80,93,143,150,154,235,314],"S^2xS^1"],
	[[5,12,80,93,145,150,167,168,233],"S^2xS^1"],
	[[5,13,29,31,46,47,70,83,205,241],"S^2~S^1"],
	[[5,13,29,31,46,48,67,205,241],"S^2~S^1"],
	[[5,13,29,31,67,154,215,241],"S^2~S^1"],
	[[5,13,45,47,67,154,216,248,297],"B2"],
	[[5,13,52,66,93,150,155,224,241],"S^2~S^1"],
	[[5,13,67,149,154,224,241],"S^2~S^1"],
	[[5,13,67,149,154,225,248,297],"B2"],
	[[5,13,68,82,94,211,273,281,299,329],"S^2~S^1"],
	[[5,13,80,93,150,154,224,241],"S^2~S^1"],
	[[5,14,28,32,67,143,167,170,311],"S^2xS^1"],
	[[5,14,28,32,67,143,204,225],"S^2xS^1"],
	[[5,14,30,31,69,82,204,233],"S^2xS^1"],
	[[5,14,30,31,69,82,204,235,314],"S^2xS^1"],
	[[5,14,30,31,70,83,173,174,205,241],"S^2~S^1"],
	[[5,14,30,32,67,204,233],"S^2xS^1"],
	[[5,14,30,32,67,204,235,314],"S^2xS^1"],
	[[5,14,67,111,113,210],"S^2xS^1"],
	[[5,14,67,143,145,167,170,249],"S^2xS^1"],
	[[5,14,67,143,145,204,225,249,311],"S^2xS^1"],
	[[5,14,67,154,164,191],"S^2xS^1"],
	[[5,14,67,154,168,170,249],"S^2xS^1"],
	[[5,14,67,154,172,174,216],"S^2xS^1"],
	[[5,14,67,154,191,201,290,291,308],"T^3"],
	[[5,14,67,154,191,201,327],"S^2xS^1"],
	[[5,14,67,154,214,233],"S^2xS^1"],
	[[5,14,67,154,214,235,314],"S^2xS^1"],
	[[5,14,67,154,216,249,325],"K^2xS^1"],
	[[5,14,67,154,220],"S^2xS^1"],
	[[5,14,68,81,167,169,241,254],"S^2~S^1"],
	[[5,14,68,81,173,175,205,241],"S^2~S^1"],
	[[5,14,68,81,203,224,241],"S^2~S^1"],
	[[5,14,68,81,203,225,248,297],"B2"],
	[[5,14,68,81,211,273,281,299,329],"S^2~S^1"],
	[[5,14,70,81,95,173,174,205,241],"S^2~S^1"],
	[[5,14,73,81,98,210,273,281,299,329],"S^2~S^1"],
	[[5,14,73,81,103,146,210,332],"S^2~S^1"],
	[[5,14,78,81,98,146,210,328],"S^2~S^1"],
	[[5,14,79,81,104,167,169,241,253],"S^2~S^1"],
	[[5,14,79,92,137,138,167,169,241,253],"S^2~S^1"],
	[[5,14,79,92,143,144,167,170,249],"S^2xS^1"],
	[[5,15,67,167,169,241,254],"S^2~S^1"],
	[[5,15,67,178,184,233,240,241,242],"B4"],
	[[5,15,67,178,184,235,240,241,242,314],"SFS[RP^2:(5,2)(5,3)]"],
	[[5,15,67,178,190,282,303],"S^2~S^1"],
	[[5,15,67,203,224,241],"S^2~S^1"],
	[[5,15,67,203,225,248,297],"B2"],
	[[5,15,67,204,234,254,297,298,311,329],"SFS[K^2:(2,1)]"],
	[[5,15,67,204,235,241,242,254,304,311],"SFS[(RP^2)#3:(1,1)]"],
	[[5,15,67,204,238,273,281,301,329],"S^2~S^1"],
	[[5,15,67,211,273,281,299,329],"S^2~S^1"],
	[[5,15,67,211,277,278,298,299,329],"S^2~S^1"],
	[[5,15,67,211,278,281,286,289,329],"S^2~S^1"],
	[[6,7,20,24,53,177,224,241,260],"S^3"],
	[[6,7,21,22,69,179,192,281,329],"S^3"],
	[[6,7,21,25,69,83,179,273,281,329],"S^3"],
	[[6,7,22,24,164,180,182,241,255],"S^3"],
	[[6,7,22,24,164,224,241,260],"S^3"],
	[[6,7,22,24,175,177,202,224,241],"S^3"],
	[[6,7,22,24,180,182,201,241,255,327],"L(8,3)"],
	[[6,7,22,24,192,273,290,297],"S^2~S^1"],
	[[6,7,22,24,201,224,241,260,327],"L(8,3)"],
	[[6,7,22,24,202,232,273,297,312],"S^2~S^1"],
	[[6,7,24,25,41,42,71,78,91,335],"S^3"],
	[[6,7,24,25,41,42,84,327,329],"S^3/P24"],
	[[6,7,24,25,41,42,90,132,326,329],"S^3/P24"],
	[[6,7,24,25,56,57,71,79,92,335],"S^3"],
	[[6,7,24,25,83,290,297],"S^2~S^1"],
	[[6,7,36,43,52,67,115,127,129,149],"S^2~S^1"],
	[[6,7,36,45,52,67,108,127,134,149],"S^2~S^1"],
	[[6,7,40,41,56,57,71,80,93,335],"S^3"],
	[[6,7,40,43,85,192,194,215,289,303],"S^3"],
	[[6,7,82,164,183,186,191,225,299],"S^2~S^1"],
	[[6,7,82,164,191,224,241],"S^3"],
	[[6,7,82,164,191,225,248,297],"S^2~S^1"],
	[[6,7,82,164,191,232,248,252,299],"S^2~S^1"],
	[[6,7,82,168,170,224,241,249],"S^2xS^1"],
	[[6,7,82,172,174,216,224,241],"S^2xS^1"],
	[[6,7,82,179,190,224,241,262],"B2"],
	[[6,7,82,180,186,220,273,299,312],"SFS[D_:(3,1)(3,1)]"],
	[[6,7,82,183,186,220,225,299],"SFS[D_:(3,1)(3,1)]"],
	[[6,7,82,191,201,224,241,327],"L(8,3)"],
	[[6,7,82,203,211,220,273,281,299,329],"SFS[S^2:(3,1)(3,1)(5,-3)]"],
	[[6,7,82,214,224,233,241],"S^2xS^1"],
	[[6,7,82,214,224,235,241,314],"L(5,1)"],
	[[6,7,82,216,224,241,249,325],"K^2xS^1"],
	[[6,7,82,220,224,241],"L(3,1)"],
	[[6,7,82,220,225,248,297],"SFS[D_:(3,1)(3,1)]"],
	[[6,7,82,220,232,248,252,299],"SFS[D_:(3,1)(3,1)]"],
	[[6,7,82,220,232,261,273,275,299],"SFS[D_:(3,1)(3,1)]"],
	[[6,7,91,116,163,169,179,271,303],"S^3"],
	[[6,7,92,115,167,168,204,297,311],"S^2~S^1"],
	[[6,7,92,115,168,170,224,241,248],"S^2xS^1"],
	[[6,7,92,115,168,170,225,297],"S^2~S^1"],
	[[6,7,92,115,168,170,232,252,299],"S^2~S^1"],
	[[6,7,92,115,216,224,241,248,325],"K^2xS^1"],
	[[6,7,92,115,220,225,249,297],"SFS[D_:(3,1)(3,1)]"],
	[[6,7,92,115,220,232,249,252,299],"SFS[D_:(3,1)(3,1)]"],
	[[6,7,93,116,164,191,241,253],"S^3"],
	[[6,7,93,116,164,202,223,241,255],"S^3"],
	[[6,7,93,116,168,170,241,249,253],"S^2xS^1"],
	[[6,7,93,116,172,174,216,241,253],"S^2xS^1"],
	[[6,7,93,116,179,182,271,303],"S^3"],
	[[6,7,93,116,191,201,241,253,327],"L(8,3)"],
	[[6,7,93,116,201,202,223,241,255,327],"L(8,3)"],
	[[6,7,93,116,214,233,241,253],"S^2xS^1"],
	[[6,7,93,116,214,235,241,253,314],"L(5,1)"],
	[[6,7,93,116,216,241,249,253,325],"K^2xS^1"],
	[[6,7,93,116,217,241,295,306],"L(3,1)"],
	[[6,7,93,116,220,241,253],"L(3,1)"],
	[[6,8,21,26,69,83,96,179,303],"S^3"],
	[[6,8,24,33,53,54,124,239,295],"S^2xS^1"],
	[[6,8,54,55,83,204,205,215,313,329],"S^3"],
	[[6,9,20,24,53,95,107,177,241,260],"S^3"],
	[[6,9,22,24,95,107,164,241,260],"S^3"],
	[[6,9,22,24,95,107,175,177,202,241],"S^3"],
	[[6,9,36,41,57,91,104,220,241,253],"SFS[RP^2:(3,1)(3,2)]"],
	[[6,9,37,39,70,84,223,241,255,274],"HS"],
	[[6,9,37,41,53,57,68,223,241,253],"B2"],
	[[6,9,39,49,90,124,132,217,241,280],"HS"],
	[[6,9,41,42,68,79,92,220,241,254],"SFS[RP^2:(3,1)(3,2)]"],
	[[6,9,41,42,92,104,179,182,271,303],"S^2~S^1"],
	[[6,9,41,42,92,104,217,241,295,306],"SFS[RP^2:(3,1)(3,2)]"],
	[[6,9,41,42,92,104,220,241,253],"SFS[RP^2:(3,1)(3,2)]"],
	[[6,9,41,48,85,102,127,215,292,328],"S^2~S^1"],
	[[6,9,54,55,69,70,84,215,313,329],"S^3"],
	[[6,9,55,65,70,80,84,104,116,329],"S^3"],
	[[6,9,68,70,84,173,175,220,241],"L(3,1)"],
	[[6,9,68,70,84,191,200,241,255],"B2"],
	[[6,9,68,70,84,200,202,223,241,253],"B2"],
	[[6,9,82,95,107,164,191,241],"S^3"],
	[[6,9,82,95,107,179,190,241,262],"B2"],
	[[6,9,82,95,107,220,241],"L(3,1)"],
	[[6,9,84,95,173,174,179,190,241,262],"B2"],
	[[6,9,84,95,173,174,220,241],"L(3,1)"],
	[[6,9,85,102,123,127,215,292,294],"S^2~S^1"],
	[[6,9,85,102,123,127,221,245],"S^2~S^1"],
	[[6,9,90,103,124,132,217,241,314],"HS"],
	[[6,11,21,22,73,97,179,196,280,329],"S^3"],
	[[6,11,36,43,59,91,104,220,241,244],"SFS[RP^2:(3,1)(3,2)]"],
	[[6,11,37,39,72,86,223,241,246,274],"HS"],
	[[6,11,37,43,53,59,68,223,241,244],"B2"],
	[[6,11,39,43,53,59,221,241,246,274],"HS"],
	[[6,11,39,43,86,222,241,246,274],"HS"],
	[[6,11,39,50,72,75,86,156,217,335],"S^2~S^1"],
	[[6,11,53,59,68,72,222,241,244],"B2"],
	[[6,11,53,59,97,221,241,246,314],"HS"],
	[[6,11,58,59,91,103,179,184,271,303],"S^2~S^1"],
	[[6,11,58,59,91,103,215,241,285,287],"SFS[RP^2:(3,1)(3,2)]"],
	[[6,11,58,59,91,103,220,241,244],"SFS[RP^2:(3,1)(3,2)]"],
	[[6,11,67,80,105,136,144,150,241],"S^2~S^1"],
	[[6,11,68,72,86,191,200,241,246],"B2"],
	[[6,11,68,72,86,221,241,244],"B2"],
	[[6,11,79,80,92,105,136,145,150,241],"S^2~S^1"],
	[[6,11,81,136,143,149,241],"S^2~S^1"],
	[[6,11,81,138,143,168,169,241],"S^2~S^1"],
	[[6,11,81,138,145,154,167,169,241],"S^2~S^1"],
	[[6,11,86,97,191,199,241,246],"B2"],
	[[6,11,86,97,199,200,221,241,244],"B2"],
	[[6,11,86,97,215,241,298,300,314],"HS"],
	[[6,11,86,97,217,300,302,305,307,335],"S^3"],
	[[6,11,86,97,222,241,246,314],"HS"],
	[[6,11,86,97,223,256,300,302,307,335],"S^3"],
	[[6,11,91,105,138,144,163,168,182,241],"S^2~S^1"],
	[[6,11,92,104,137,143,168,169,241],"S^2~S^1"],
	[[6,11,92,104,137,145,154,167,169,241],"S^2~S^1"],
	[[6,11,93,105,136,144,149,241],"S^2~S^1"],
	[[6,11,93,105,138,144,168,169,241],"S^2~S^1"],
	[[6,12,20,24,53,95,106,177,260],"S^2xS^1"],
	[[6,12,22,24,95,106,164,260],"S^2xS^1"],
	[[6,12,22,24,95,106,175,177,202],"S^2xS^1"],
	[[6,12,28,30,45,47,81,216],"S^2xS^1"],
	[[6,12,28,30,81,149,225],"S^2xS^1"],
	[[6,12,52,59,74,90,103,218],"S^2xS^1"],
	[[6,12,59,61,81,149,227,314],"S^2xS^1"],
	[[6,12,59,65,81,149,156,227,289],"S^2xS^1"],
	[[6,12,67,80,105,144,150,233],"S^2xS^1"],
	[[6,12,67,80,105,144,150,235,314],"S^2xS^1"],
	[[6,12,73,74,90,102,218],"S^2xS^1"],
	[[6,12,74,80,83,95,148,287],"S^2xS^1"],
	[[6,12,79,80,92,105,145,150,233],"S^2xS^1"],
	[[6,12,81,143,149,233],"S^2xS^1"],
	[[6,12,81,143,149,235,314],"S^2xS^1"],
	[[6,12,82,95,106,164,191],"S^2xS^1"],
	[[6,12,82,95,106,179,187,271,335],"B2"],
	[[6,12,82,95,106,179,190,262],"T^3"],
	[[6,12,82,95,106,220],"S^2xS^1"],
	[[6,12,82,95,106,223,262,265,335],"B2"],
	[[6,12,83,95,148,153,164,191],"S^2xS^1"],
	[[6,12,83,95,148,153,220],"S^2xS^1"],
	[[6,12,93,105,144,149,233],"S^2xS^1"],
	[[6,12,93,105,144,149,235,314],"S^2xS^1"],
	[[6,13,20,24,53,94,177,260],"S^2xS^1"],
	[[6,13,22,23,150,154,164,260],"S^2xS^1"],
	[[6,13,22,23,150,154,201,260,327],"S^2xS^1"],
	[[6,13,22,24,94,164,260],"S^2xS^1"],
	[[6,13,22,24,94,175,177,202],"S^2xS^1"],
	[[6,13,22,24,94,201,260,327],"S^2xS^1"],
	[[6,13,29,30,83,95,215,241],"S^2~S^1"],
	[[6,13,29,31,81,215,241],"S^2~S^1"],
	[[6,13,40,43,85,94,223,261,289,335],"B2"],
	[[6,13,45,46,85,97,179,186,271,335],"B2"],
	[[6,13,45,47,81,183,186,216,299],"B2"],
	[[6,13,45,47,81,216,248,297],"B2"],
	[[6,13,81,149,180,186,273,299,312],"B2"],
	[[6,13,81,149,183,186,225,299],"B2"],
	[[6,13,81,149,203,211,273,281,299,329],"S^2~S^1"],
	[[6,13,81,149,224,241],"S^2~S^1"],
	[[6,13,81,149,225,248,297],"B2"],
	[[6,13,81,149,232,248,252,299],"B2"],
	[[6,13,81,149,232,261,273,275,299],"B2"],
	[[6,13,82,94,164,191],"S^2xS^1"],
	[[6,13,82,94,168,170,249],"S^2xS^1"],
	[[6,13,82,94,172,174,216],"S^2xS^1"],
	[[6,13,82,94,179,187,271,335],"B2"],
	[[6,13,82,94,179,190,262],"T^3"],
	[[6,13,82,94,191,201,327],"S^2xS^1"],
	[[6,13,82,94,214,233],"S^2xS^1"],
	[[6,13,82,94,214,235,314],"S^2xS^1"],
	[[6,13,82,94,216,249,325],"K^2xS^1"],
	[[6,13,82,94,220],"S^2xS^1"],
	[[6,13,82,94,223,262,265,335],"B2"],
	[[6,13,85,94,108,179,186,271,335],"B2"],
	[[6,13,92,94,115,168,170,248],"S^2xS^1"],
	[[6,13,92,94,115,216,248,325],"K^2xS^1"],
	[[6,14,81,164,191],"S^2xS^1"],
	[[6,14,81,168,170,249],"S^2xS^1"],
	[[6,14,81,179,187,271,335],"B2"],
	[[6,14,81,179,190,262],"T^3"],
	[[6,14,81,191,201,290,291,308],"T^3"],
	[[6,14,81,191,201,327],"S^2xS^1"],
	[[6,14,81,214,233],"S^2xS^1"],
	[[6,14,81,214,235,314],"S^2xS^1"],
	[[6,14,81,216,249,325],"K^2xS^1"],
	[[6,14,81,220],"S^2xS^1"],
	[[7,8,22,25,39,42,84,193,329],"S^3"],
	[[7,8,25,26,39,42,274,329],"S^3"],
	[[7,8,25,26,71,78,91,92,104,335],"S^3"],
	[[7,8,25,26,71,78,103,298,303,326],"Sigma^3"],
	[[7,8,25,26,84,92,104,327,329],"S^3/P24"],
	[[7,8,25,26,90,92,104,132,326,329],"S^3/P24"],
	[[7,8,25,26,97,127,302,307],"S^3"],
	[[7,8,26,33,97,124,127,192,200,305],"S^3"],
	[[7,8,36,39,65,117,127,143,155],"S^2~S^1"],
	[[7,8,36,49,54,117,127,143,156],"S^2~S^1"],
	[[7,8,36,49,54,117,127,147,192,193],"S^2~S^1"],
	[[7,8,37,39,70,193,202,224,241,260],"B2"],
	[[7,8,39,41,180,182,193,201,241,255],"B2"],
	[[7,8,39,41,180,182,239,241,255,316],"L(5,1)"],
	[[7,8,39,41,193,201,224,241,260],"B2"],
	[[7,8,39,41,224,236,241,280,295,316],"L(5,1)"],
	[[7,8,39,41,224,239,241,260,316],"L(5,1)"],
	[[7,8,39,48,117,127,147,161,180,193],"S^2~S^1"],
	[[7,8,39,49,124,224,236,241,280],"L(5,1)"],
	[[7,8,39,49,124,224,239,241,260,295],"L(5,1)"],
	[[7,8,39,50,117,127,143,156],"S^2~S^1"],
	[[7,8,39,50,117,127,147,192,193],"S^2~S^1"],
	[[7,8,41,42,71,78,91,93,105,335],"S^3"],
	[[7,8,41,42,84,93,105,327,329],"S^3/P24"],
	[[7,8,41,42,90,93,105,132,326,329],"S^3/P24"],
	[[7,8,41,46,102,143,145,292,322,335],"S^2~S^1"],
	[[7,8,52,59,77,103,119,232,240,335],"S^2~S^1"],
	[[7,8,52,59,77,103,121,143,335],"S^2~S^1"],
	[[7,8,52,61,77,98,124,143,335],"S^2~S^1"],
	[[7,8,56,57,71,79,92,93,105,335],"S^3"],
	[[7,8,70,75,98,143,330,331,335],"S^2~S^1"],
	[[7,8,70,75,104,143,147,305,331,335],"S^2~S^1"],
	[[7,8,70,77,102,232,240,310,312,335],"S^2~S^1"],
	[[7,8,73,77,102,119,232,240,335],"S^2~S^1"],
	[[7,8,73,77,102,121,143,335],"S^2~S^1"],
	[[7,8,75,77,98,123,143,335],"S^2~S^1"],
	[[7,8,83,84,96,192,193,313,329],"S^3"],
	[[7,8,83,93,105,290,297],"S^2~S^1"],
	[[7,8,95,164,191,214,224,241],"S^3"],
	[[7,8,95,179,190,214,224,241,262],"B2"],
	[[7,8,95,193,197,224,241,318],"L(5,1)"],
	[[7,8,95,193,201,224,241,258],"B2"],
	[[7,8,95,214,220,224,241],"L(3,1)"],
	[[7,8,95,224,233,241],"S^2xS^1"],
	[[7,8,95,224,235,241,314],"L(5,1)"],
	[[7,8,95,224,235,241,325,327],"L(7,1)"],
	[[7,8,95,224,239,241,258,316],"L(5,1)"],
	[[7,8,102,119,145,232,240,322,326,335],"S^2~S^1"],
	[[7,8,102,121,143,145,322,326,335],"S^2~S^1"],
	[[7,8,103,124,224,236,241,314],"L(5,1)"],
	[[7,8,103,124,224,236,241,325,327],"L(7,1)"],
	[[7,8,103,124,232,234,241,247,314],"L(5,1)"],
	[[7,8,103,124,232,234,241,247,325,327],"L(7,1)"],
	[[7,8,105,126,224,241,248],"S^2xS^1"],
	[[7,8,105,126,225,297],"S^2~S^1"],
	[[7,8,105,126,232,252,299],"S^2~S^1"],
	[[7,11,39,48,107,127,145,180,189,258],"S^2~S^1"],
	[[7,11,67,79,93,104,116,138,145,241],"S^2~S^1"],
	[[7,11,81,93,116,137,144,241],"S^2~S^1"],
	[[7,11,85,93,98,116,241,295,306],"L(3,1)"],
	[[7,11,92,93,104,116,138,144,241],"S^2~S^1"],
	[[7,11,94,136,143,241],"S^2~S^1"],
	[[7,11,105,116,137,143,241],"S^2~S^1"],
	[[7,12,28,30,94,225],"S^2xS^1"],
	[[7,12,54,55,147,152,192,197,318],"S^2xS^1"],
	[[7,12,54,55,147,152,192,201,258],"K^2xS^1"],
	[[7,12,59,61,94,227,314],"S^2xS^1"],
	[[7,12,59,61,94,227,325,327],"S^2xS^1"],
	[[7,12,59,65,94,156,192,197,287],"S^2xS^1"],
	[[7,12,59,65,94,156,227,289],"S^2xS^1"],
	[[7,12,94,143,164,191,214],"S^2xS^1"],
	[[7,12,94,143,179,187,214,271,335],"B2"],
	[[7,12,94,143,179,190,214,262],"T^3"],
	[[7,12,94,143,193,197,318],"S^2xS^1"],
	[[7,12,94,143,193,201,258],"K^2xS^1"],
	[[7,12,94,143,214,220],"S^2xS^1"],
	[[7,12,94,143,214,223,262,265,335],"B2"],
	[[7,12,94,143,233],"S^2xS^1"],
	[[7,12,94,143,235,314],"S^2xS^1"],
	[[7,12,94,143,235,325,327],"S^2xS^1"],
	[[7,12,94,143,239,258,316],"S^2xS^1"],
	[[7,12,94,145,156,161,189,192,258],"K^2xS^1"],
	[[7,12,94,147,156,192,197,318],"S^2xS^1"],
	[[7,12,94,147,156,192,201,258],"K^2xS^1"],
	[[7,12,94,147,156,227,287,289,318],"S^2xS^1"],
	[[7,12,95,106,224,241],"S^2~S^1"],
	[[7,13,94,180,186,273,299,312],"B2"],
	[[7,13,94,180,186,276,327,330,331,335],"S^2~S^1"],
	[[7,13,94,183,186,225,299],"B2"],
	[[7,13,94,203,211,273,281,299,329],"S^2~S^1"],
	[[7,13,94,224,241],"S^2~S^1"],
	[[7,13,94,225,248,297],"B2"],
	[[7,13,94,228,273,275,299,310,323],"B2"],
	[[7,13,94,228,322,323,326,327,335],"S^2~S^1"],
	[[8,9,20,26,53,82,177,241,253,260],"S^3"],
	[[8,9,22,26,53,55,200,241,255,274],"B2"],
	[[8,9,22,26,82,164,241,253,260],"S^3"],
	[[8,9,22,26,82,175,177,202,241,253],"S^3"],
	[[8,9,23,26,55,65,104,205,329],"S^3"],
	[[8,9,24,26,164,191,241,253],"S^3"],
	[[8,9,24,26,164,202,223,241,255],"S^3"],
	[[8,9,24,26,173,175,200,202,223,241],"S^3"],
	[[8,9,24,26,175,177,223,241,255,260],"S^3"],
	[[8,9,24,26,179,182,271,303],"S^3"],
	[[8,9,24,26,198,202,223,243,331,335],"Sigma^3"],
	[[8,9,24,26,217,241,295,306],"L(3,1)"],
	[[8,9,24,26,220,241,253],"L(3,1)"],
	[[8,9,26,32,113,241,253,295],"L(3,1)"],
	[[8,9,36,39,65,94,127,136,155],"S^2~S^1"],
	[[8,9,36,40,52,79,127,136,150],"S^2~S^1"],
	[[8,9,36,40,65,127,136,151],"S^2~S^1"],
	[[8,9,36,40,65,127,140,193,194],"S^2~S^1"],
	[[8,9,36,47,55,127,141,158,193,205],"S^2~S^1"],
	[[8,9,36,48,52,69,127,136,152],"S^2~S^1"],
	[[8,9,36,48,52,69,127,141,204,205],"S^2~S^1"],
	[[8,9,36,49,54,94,127,136,156],"S^2~S^1"],
	[[8,9,36,49,55,127,136,152],"S^2~S^1"],
	[[8,9,36,49,55,127,141,204,205],"S^2~S^1"],
	[[8,9,39,42,55,66,105,205,329],"S^3"],
	[[8,9,39,42,105,116,274,329],"S^3"],
	[[8,9,39,50,54,55,127,136,152],"S^2~S^1"],
	[[8,9,39,50,94,127,136,156],"S^2~S^1"],
	[[8,9,40,46,133,155,234,241,275],"L(5,1)"],
	[[8,9,40,47,62,65,127,141,193,194],"S^2~S^1"],
	[[8,9,40,50,127,136,151,155,156],"S^2~S^1"],
	[[8,9,40,50,127,140,155,156,193,194],"S^2~S^1"],
	[[8,9,47,49,55,62,127,140,204,205],"S^2~S^1"],
	[[8,9,53,55,84,193,200,241,255],"B2"],
	[[8,9,53,61,85,111,127,239,245],"S^3"],
	[[8,9,53,61,85,116,127,157,239,306],"S^3"],
	[[8,9,54,55,96,204,205,313,329],"S^3"],
	[[8,9,54,62,97,116,127,160,239,306],"S^3"],
	[[8,9,55,57,193,196,241,328,330],"B2"],
	[[8,9,55,57,193,201,241,255],"B2"],
	[[8,9,55,57,234,241,283,285,316],"L(5,1)"],
	[[8,9,55,57,234,241,310,313,328,330],"B2"],
	[[8,9,55,57,239,241,255,316],"L(5,1)"],
	[[8,9,55,58,127,193,194,323,331],"S^2~S^1"],
	[[8,9,55,62,127,140,158,193,205],"S^2~S^1"],
	[[8,9,55,64,127,136,151],"S^2~S^1"],
	[[8,9,55,64,127,140,193,194],"S^2~S^1"],
	[[8,9,55,65,134,234,241,283],"L(5,1)"],
	[[8,9,55,65,134,239,241,255,285],"L(5,1)"],
	[[8,9,57,61,111,234,241,283,285,328],"L(5,1)"],
	[[8,9,57,61,111,239,241,255,328],"L(5,1)"],
	[[8,9,58,66,116,127,204,280,304],"S^3"],
	[[8,9,58,66,116,127,212,239,280,306],"S^3"],
	[[8,9,62,66,116,127,140,158,193,274],"S^2~S^1"],
	[[8,9,71,78,91,92,104,105,116,335],"S^3"],
	[[8,9,84,92,104,105,116,327,329],"S^3/P24"],
	[[8,9,84,92,115,234,241,298,327],"L(7,1)"],
	[[8,9,90,92,104,105,116,132,326,329],"S^3/P24"],
	[[8,9,90,92,115,132,234,241,298,326],"L(7,1)"],
	[[8,9,107,164,191,214,241],"S^3"],
	[[8,9,107,179,190,214,241,262],"B2"],
	[[8,9,107,193,197,241,318],"L(5,1)"],
	[[8,9,107,193,201,241,258],"B2"],
	[[8,9,107,214,220,241],"L(3,1)"],
	[[8,9,107,233,241],"S^2xS^1"],
	[[8,9,107,235,241,273,278,302],"B2"],
	[[8,9,107,235,241,314],"L(5,1)"],
	[[8,9,107,235,241,325,327],"L(7,1)"],
	[[8,9,107,239,241,258,316],"L(5,1)"],
	[[8,9,108,127,193,194,312,313,323,331],"S^2~S^1"],
	[[8,9,111,130,193,196,241,318],"L(5,1)"],
	[[8,9,111,130,234,241,285,289,310,330],"L(5,1)"],
	[[8,9,111,130,234,241,310,313,318],"L(5,1)"],
	[[8,9,111,133,155,234,241,275,285,322],"L(5,1)"],
	[[8,9,115,130,156,193,196,241,310],"L(5,1)"],
	[[8,9,115,130,156,234,241,313],"L(5,1)"],
	[[8,9,115,134,234,241,314],"L(5,1)"],
	[[8,9,115,134,234,241,325,327],"L(7,1)"],
	[[8,11,106,136,241],"S^2~S^1"],
	[[8,12,106,164,191,214],"S^2xS^1"],
	[[8,12,106,179,190,214,262],"T^3"],
	[[8,12,106,193,197,318],"S^2xS^1"],
	[[8,12,106,193,201,258],"K^2xS^1"],
	[[8,12,106,214,220],"S^2xS^1"],
	[[8,12,106,233],"S^2xS^1"],
	[[8,12,106,235,314],"S^2xS^1"],
	[[8,12,106,235,325,327],"S^2xS^1"],
	[[9,10,118,182,184,274,277,292,317,335],"(S^2xS^1)#12"],
	[[9,10,118,182,184,274,281,292,313,335],"(S^2~S^1)#12"],
	[[9,10,118,182,184,277,281,292,307,335],"(S^2~S^1)#12"],
	[[9,10,118,182,184,292,307,313,317,335],"(S^2xS^1)#12"],
	[[9,10,118,182,187,253,274,277,317,335],"(S^2xS^1)#12"],
	[[9,10,118,182,187,253,274,281,313,335],"(S^2~S^1)#12"],
	[[9,10,118,182,187,253,277,281,307,335],"(S^2~S^1)#12"],
	[[9,10,118,182,187,253,307,313,317,335],"(S^2xS^1)#12"],
	[[9,10,118,184,187,244,274,277,317,335],"(S^2xS^1)#12"],
	[[9,10,118,184,187,244,274,281,313,335],"(S^2~S^1)#12"],
	[[9,10,118,184,187,244,277,281,307,335],"(S^2~S^1)#12"],
	[[9,10,118,184,187,244,307,313,317,335],"(S^2xS^1)#12"],
	[[9,10,118,194,196,265,268,285,322,329],"(S^2xS^1)#12"],
	[[9,10,118,194,196,265,271,282,322,329],"(S^2~S^1)#12"],
	[[9,10,118,194,196,282,285,306,322,329],"(S^2~S^1)#12"],
	[[9,10,118,194,200,254,265,268,285,329],"(S^2xS^1)#12"],
	[[9,10,118,194,200,254,265,271,282,329],"(S^2~S^1)#12"],
	[[9,10,118,194,200,254,268,271,306,329],"(S^2xS^1)#12"],
	[[9,10,118,194,200,254,282,285,306,329],"(S^2~S^1)#12"],
	[[9,10,118,196,200,245,265,268,285,329],"(S^2xS^1)#12"],
	[[9,10,118,196,200,245,265,271,282,329],"(S^2~S^1)#12"],
	[[9,10,118,196,200,245,282,285,306,329],"(S^2~S^1)#12"],
	[[9,10,118,205,207,265,271,282,329,331],"(S^2~S^1)#12"],
	[[9,10,118,205,207,268,271,306,329,331],"(S^2xS^1)#12"],
	[[9,10,118,205,207,282,285,306,329,331],"(S^2~S^1)#12"],
	[[9,10,118,205,212,255,265,268,285,329],"(S^2xS^1)#12"],
	[[9,10,118,205,212,255,265,271,282,329],"(S^2~S^1)#12"],
	[[9,10,118,205,212,255,268,271,306,329],"(S^2xS^1)#12"],
	[[9,10,118,205,212,255,282,285,306,329],"(S^2~S^1)#12"],
	[[9,10,118,207,212,246,265,271,282,329],"(S^2~S^1)#12"],
	[[9,10,118,207,212,246,268,271,306,329],"(S^2xS^1)#12"],
	[[9,10,118,207,212,246,282,285,306,329],"(S^2~S^1)#12"],
	[[9,10,118,215,217,274,281,305,313,335],"(S^2~S^1)#12"],
	[[9,10,118,215,217,277,281,305,307,335],"(S^2~S^1)#12"],
	[[9,10,118,215,223,256,274,281,313,335],"(S^2~S^1)#12"],
	[[9,10,118,215,223,256,277,281,307,335],"(S^2~S^1)#12"],
	[[9,10,118,215,223,256,307,313,317,335],"(S^2xS^1)#12"],
	[[9,10,118,217,223,247,274,281,313,335],"(S^2~S^1)#12"],
	[[9,10,118,217,223,247,277,281,307,335],"(S^2~S^1)#12"],
	[[9,10,118,241],"S^2xS^1"],
	[[9,10,118,244,253,274,281,292,313,335],"(S^2~S^1)#12"],
	[[9,10,118,244,253,277,281,292,307,335],"(S^2~S^1)#12"],
	[[9,10,118,245,254,265,268,285,322,329],"(S^2xS^1)#12"],
	[[9,10,118,245,254,265,271,282,322,329],"(S^2~S^1)#12"],
	[[9,10,118,245,254,282,285,306,322,329],"(S^2~S^1)#12"],
	[[9,10,118,246,255,265,271,282,329,331],"(S^2~S^1)#12"],
	[[9,10,118,246,255,282,285,306,329,331],"(S^2~S^1)#12"],
	[[9,10,118,247,256,274,281,305,313,335],"(S^2~S^1)#12"],
	[[9,10,118,247,256,277,281,305,307,335],"(S^2~S^1)#12"],
	[[9,11,117,182,184,274,281,292,313,335],"(S^2~S^1)#12"],
	[[9,11,117,182,184,277,281,292,307,335],"(S^2~S^1)#12"],
	[[9,11,117,182,187,253,274,281,313,335],"(S^2~S^1)#12"],
	[[9,11,117,182,187,253,277,281,307,335],"(S^2~S^1)#12"],
	[[9,11,117,184,187,244,274,281,313,335],"(S^2~S^1)#12"],
	[[9,11,117,184,187,244,277,281,307,335],"(S^2~S^1)#12"],
	[[9,11,117,194,196,282,285,306,322,329],"(S^2~S^1)#12"],
	[[9,11,117,194,200,254,265,271,282,329],"(S^2~S^1)#12"],
	[[9,11,117,194,200,254,282,285,306,329],"(S^2~S^1)#12"],
	[[9,11,117,196,200,245,282,285,306,329],"(S^2~S^1)#12"],
	[[9,11,117,205,207,282,285,306,329,331],"(S^2~S^1)#12"],
	[[9,11,117,215,217,274,281,305,313,335],"(S^2~S^1)#12"],
	[[9,11,117,241],"S^2~S^1"],
	[[19,22,36,66,81,94,106,117,127,158],"S^2~S^1"],
	[[19,22,39,86,105,147,193,300],"S^2xS^1"],
	[[19,22,39,92,105,193,327],"S^2xS^1"],
	[[19,22,45,92,153,199,276,300],"S^2xS^1"],
	[[19,23,36,52,80,94,106,117,127,154],"S^2~S^1"],
	[[19,23,36,66,94,106,117,127,155],"S^2~S^1"],
	[[19,23,37,80,104,212,223,335],"S^2~S^1"],
	[[19,23,40,69,79,105,204,290,297],"B2"],
	[[19,23,40,105,115,204,297,311],"B2"],
	[[19,23,43,72,80,104,212,222,335],"S^2~S^1"],
	[[19,23,48,69,80,94,152,154],"T^3"],
	[[19,23,48,70,80,94,106,147,154],"T^3"],
	[[19,23,50,54,66,282,303],"S^2~S^1"],
	[[19,23,50,105,204,273,282,303],"S^2~S^1"],
	[[19,24,36,52,67,93,106,117,127,149],"S^2~S^1"],
	[[19,24,36,52,80,106,117,127,150],"S^2~S^1"],
	[[19,24,36,66,106,117,127,150,154,155],"S^2~S^1"],
	[[19,24,41,55,65,70,80,327],"S^2xS^1"],
	[[19,24,41,55,65,76,80,122,328],"S^2xS^1"],
	[[19,24,41,70,80,115,283,325],"K^2xS^1"],
	[[19,24,41,70,80,115,289,310,328],"K^2xS^1"],
	[[19,24,41,82,92,126,220,248],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[19,24,41,115,126,216,325],"K^2xS^1"],
	[[19,24,41,115,126,220,249],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[19,24,45,69,80,152,283,290],"T^3"],
	[[19,24,47,55,65,70,80,122,293],"S^2xS^1"],
	[[19,24,47,67,80,83,106,147,150],"T^3"],
	[[19,24,47,69,70,83,93,149,152],"T^3"],
	[[19,24,47,76,80,115,289,293,310],"S^2xS^1"],
	[[19,24,47,83,93,106,147,149],"T^3"],
	[[19,24,48,67,69,93,149,152],"T^3"],
	[[19,24,48,67,70,93,106,147,149],"T^3"],
	[[19,24,48,69,80,150,152],"T^3"],
	[[19,24,48,70,80,106,147,150],"T^3"],
	[[19,24,50,69,80,273,282,290,297,303],"S^2~S^1"],
	[[19,24,50,69,80,280,283,290,312],"T^3"],
	[[19,24,50,116,215,273,290,297,303],"S^2~S^1"],
	[[19,25,36,52,67,93,117,127,144],"S^2~S^1"],
	[[19,25,36,52,75,98,105,123,143],"B2"],
	[[19,25,36,52,77,126,225,297,335],"S^2~S^1"],
	[[19,25,36,52,80,117,127,144,149,150],"S^2~S^1"],
	[[19,25,36,52,80,123,221,227,257],"S^2~S^1"],
	[[19,25,36,61,98,105,124,143],"B2"],
	[[19,25,47,83,93,144,147],"T^3"],
	[[19,25,48,67,69,93,106,144,152],"T^3"],
	[[19,25,48,67,70,93,144,147],"T^3"],
	[[19,25,48,70,80,144,147,149,150],"T^3"],
	[[19,26,36,52,67,81,94,116,127,136],"S^2~S^1"],
	[[19,26,36,52,67,93,127,137,143,144],"S^2~S^1"],
	[[19,26,50,54,66,96,204,303],"S^2~S^1"],
	[[19,26,50,96,105,273,303],"S^2~S^1"],
	[[19,26,50,135,234,273,297,303],"S^2~S^1"],
	[[19,27,36,52,67,81,94,116,127,129],"S^2~S^1"],
	[[19,27,50,108,116,273,297,303],"S^2~S^1"],
	[[19,28,36,52,77,119,126,335],"S^2~S^1"],
	[[19,28,39,104,119,126,274,329],"S^2~S^1"],
	[[19,28,45,74,80,147,216,287],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[19,28,47,74,80,147,149,225,287],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[19,28,50,119,126,281,329],"S^2~S^1"],
	[[19,35,50,184,234,240,273,297,303],"S^2~S^1"],
	[[19,35,50,190,273,282,293,294,300,303],"S^2~S^1"],
	[[20,22,55,93,116,169,202,274,329],"S^2~S^1"],
	[[20,22,61,68,75,86,98,177,199],"S^2xS^1"],
	[[20,22,66,93,169,202,205,329],"S^2~S^1"],
	[[20,23,54,167,225,252,289,311],"S^2xS^1"],
	[[20,23,54,170,204,252,289],"S^2xS^1"],
	[[20,23,59,98,170,204,252,288],"S^2xS^1"],
	[[20,23,64,103,167,232,283,314],"S^2xS^1"],
	[[20,23,64,103,167,232,283,325,327],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[20,23,66,68,74,153,177,287],"S^2xS^1"],
	[[20,23,66,68,80,177,214,233],"S^2xS^1"],
	[[20,23,66,68,80,177,220],"S^2xS^1"],
	[[20,23,66,74,80,99,168,240,276],"S^2xS^1"],
	[[20,23,66,74,80,99,177,221],"S^2xS^1"],
	[[20,23,66,95,126,177,248,257],"S^2xS^1"],
	[[20,23,66,105,177,214,220,257],"S^2xS^1"],
	[[20,23,66,105,177,233,257],"S^2xS^1"],
	[[20,24,55,169,180,182,223,274,329],"S^2~S^1"],
	[[20,24,57,84,93,116,169,223,327,329],"SFS[D_:(3,1)(3,1)]"],
	[[20,24,66,113,154,157,177,295],"S^2xS^1"],
	[[20,25,41,42,57,169,220,223,241,253],"L(3,1)"],
	[[20,25,53,93,95,105,177,233,260],"S^2xS^1"],
	[[20,25,53,93,126,177,248,260],"S^2xS^1"],
	[[20,25,58,83,85,169,241,256,299],"S^2~S^1"],
	[[20,25,58,118,169,241,252,299],"S^2~S^1"],
	[[20,25,61,83,85,137,169,241,256,304],"S^2~S^1"],
	[[20,25,61,118,137,169,216,223,241,330],"S^2~S^1"],
	[[20,25,66,95,105,177,203,214,220],"S^2xS^1"],
	[[20,25,66,95,105,177,203,233],"S^2xS^1"],
	[[20,25,66,126,177,203,248],"S^2xS^1"],
	[[20,26,55,84,93,116,169,193,202,329],"S^2~S^1"],
	[[20,26,57,167,207,232,234,303,323,327],"HS"],
	[[20,26,57,173,184,186,240,327,329,330],"SFS[D_:(3,1)(3,1)]"],
	[[20,26,57,173,234,261,303,310,322,327],"HS"],
	[[20,26,57,177,234,254,275,303,327,330],"HS"],
	[[20,27,58,174,185,187,247,326,327,329],"SFS[D_:(3,1)(3,1)]"],
	[[20,27,58,176,185,187,247,258,326,329],"S^2~S^1"],
	[[20,27,61,118,121,172,227,232,241],"S^2~S^1"],
	[[20,27,62,138,174,185,187,247,327,329],"SFS[D_:(3,1)(3,1)]"],
	[[20,27,62,138,176,185,187,247,258,329],"S^2~S^1"],
	[[20,28,53,93,148,177,225,260],"S^2xS^1"],
	[[20,28,66,143,157,177,204,225],"S^2xS^1"],
	[[20,28,66,148,177,203,225],"S^2xS^1"],
	[[20,29,57,129,167,207,232,303,323,327],"HS"],
	[[20,29,57,129,173,261,303,310,322,327],"HS"],
	[[20,29,57,129,177,254,275,303,327,330],"HS"],
	[[20,29,58,142,153,174,256,326,327,329],"SFS[D_:(3,1)(3,1)]"],
	[[20,29,58,142,153,176,256,258,326,329],"S^2~S^1"],
	[[20,29,62,138,142,153,174,256,327,329],"SFS[D_:(3,1)(3,1)]"],
	[[20,29,62,138,142,153,176,256,258,329],"S^2~S^1"],
	[[20,29,62,153,160,176,256,313,329],"S^2~S^1"],
	[[20,30,46,47,62,174,215,220,241,256],"L(3,1)"],
	[[20,30,46,47,62,174,217,220,241,247],"L(3,1)"],
	[[20,30,53,92,156,177,258,273],"S^2xS^1"],
	[[20,30,53,93,143,148,177,233,260],"S^2xS^1"],
	[[20,30,58,83,85,121,169,227,241,256],"S^2~S^1"],
	[[20,30,65,156,177,257,273],"S^2xS^1"],
	[[20,30,66,143,148,177,203,214,220],"S^2xS^1"],
	[[20,30,66,143,148,177,203,233],"S^2xS^1"],
	[[20,30,66,157,177,204,214,220],"S^2xS^1"],
	[[20,30,66,157,177,204,233],"S^2xS^1"],
	[[20,31,37,47,53,68,83,95,174],"S^2xS^1"],
	[[20,31,37,47,53,70,83,175],"S^2xS^1"],
	[[20,31,37,47,55,70,82,83,176],"S^2xS^1"],
	[[20,31,37,48,53,67,175],"S^2xS^1"],
	[[20,31,53,83,122,177,260,293],"S^2xS^1"],
	[[20,31,53,89,177,260,294],"S^2xS^1"],
	[[20,31,59,144,175,197,202,257,287],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[20,31,62,165,189,212,261,329],"S^2~S^1"],
	[[20,31,62,176,187,189,261,313,329],"S^2~S^1"],
	[[20,32,37,48,53,69,82,175],"S^2xS^1"],
	[[20,32,37,49,52,176],"S^2xS^1"],
	[[20,32,39,49,52,68,177],"S^2xS^1"],
	[[20,32,52,68,103,177,280,314],"S^2xS^1"],
	[[20,32,54,68,77,165,178,335],"S^2~S^1"],
	[[20,32,54,68,77,167,203,261,335],"S^2~S^1"],
	[[20,32,54,95,117,127,145,168,214,261],"B2"],
	[[20,32,54,95,117,127,145,176,258,261],"L(5,1)"],
	[[20,32,54,102,165,178,244,245,335],"S^2~S^1"],
	[[20,32,54,102,173,241,244,261,322],"S^3"],
	[[20,32,59,68,73,102,177,280,314],"S^2xS^1"],
	[[20,32,61,121,123,177,220,221,280],"S^2xS^1"],
	[[20,32,61,154,167,220,232,275,280],"S^2xS^1"],
	[[20,32,66,154,157,177,214,233],"S^2xS^1"],
	[[20,32,66,154,157,177,220],"S^2xS^1"],
	[[20,33,36,50,177],"S^2xS^1"],
	[[20,33,37,49,54,68,176],"S^2xS^1"],
	[[20,33,39,49,54,177],"S^2xS^1"],
	[[20,33,52,74,110,113,165,178],"B2"],
	[[20,33,52,74,110,113,177,281,329],"B4"],
	[[20,33,52,77,165,178,335],"S^2~S^1"],
	[[20,33,52,77,167,203,261,335],"S^2~S^1"],
	[[20,33,54,103,177,276,311,323,327],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[20,33,54,103,177,280,314],"S^2xS^1"],
	[[20,33,54,103,177,280,325,327],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[20,33,61,155,177,283,310,327,328],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[20,33,61,155,177,289,314],"S^2xS^1"],
	[[20,33,61,155,177,289,325,327],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[20,33,64,168,221,240,312,323,327],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[20,33,64,174,214,221,240,312,323],"K^2xS^1"],
	[[20,33,64,177,275,283,289,314],"S^2xS^1"],
	[[20,33,64,177,275,283,289,325,327],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[20,33,64,177,275,310,314,325,328],"S^2xS^1"],
	[[20,33,64,177,275,310,327,328],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[20,33,64,177,276,312,314,323,325],"S^2xS^1"],
	[[20,33,64,177,276,312,323,327],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[20,33,64,177,280,311,312,314],"S^2xS^1"],
	[[20,33,64,177,280,311,312,325,327],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[20,33,65,155,156,177],"S^2xS^1"],
	[[20,35,52,73,94,102,106,121,163],"B2"],
	[[20,35,52,74,94,102,111,163],"B2"],
	[[20,35,52,74,110,113,163],"B2"],
	[[20,35,52,77,163,335],"S^2~S^1"],
	[[20,35,54,105,165,204,293,294,300],"B2"],
	[[20,35,57,84,91,163,314,325,329],"S^2~S^1"],
	[[20,35,57,84,91,163,327,329],"SFS[D_:(3,1)(3,1)]"],
	[[20,35,57,84,93,169,182,314,325,329],"S^2~S^1"],
	[[20,35,57,84,93,169,182,327,329],"SFS[D_:(3,1)(3,1)]"],
	[[20,35,57,133,163,314,325,329,330],"S^2~S^1"],
	[[20,35,57,133,163,327,329,330],"SFS[D_:(3,1)(3,1)]"],
	[[20,35,57,135,173,186,314,325,329,330],"S^2~S^1"],
	[[20,35,57,135,173,186,327,329,330],"SFS[D_:(3,1)(3,1)]"],
	[[20,35,58,97,103,163,314,329],"S^2~S^1"],
	[[20,35,58,97,103,163,325,327,329],"SFS[D_:(3,1)(3,1)]"],
	[[20,35,58,142,174,187,326,327,329],"SFS[D_:(3,1)(3,1)]"],
	[[20,35,58,142,176,187,258,326,329],"S^2~S^1"],
	[[20,35,59,94,103,106,121,163],"B2"],
	[[20,35,59,148,173,184,304,314,329],"S^2~S^1"],
	[[20,35,59,148,173,184,304,325,327,329],"SFS[D_:(3,1)(3,1)]"],
	[[20,35,61,130,135,173,186,314,329],"S^2~S^1"],
	[[20,35,61,130,135,173,186,325,327,329],"SFS[D_:(3,1)(3,1)]"],
	[[20,35,62,138,142,174,187,327,329],"SFS[D_:(3,1)(3,1)]"],
	[[20,35,62,138,142,176,187,258,329],"S^2~S^1"],
	[[20,35,62,160,165,212,329],"S^2~S^1"],
	[[20,35,62,160,176,187,313,329],"S^2~S^1"],
	[[20,35,64,163,314,322,325,328,329,330],"S^2~S^1"],
	[[20,35,64,163,322,327,328,329,330],"SFS[D_:(3,1)(3,1)]"],
	[[20,35,66,165,281,329],"S^2~S^1"],
	[[20,35,66,169,180,205,329],"S^2~S^1"],
	[[22,23,53,66,95,126,164,248,257],"S^2xS^1"],
	[[22,23,53,66,105,164,233,257],"S^2xS^1"],
	[[22,23,58,66,93,97,202,232,283,329],"S^2~S^1"],
	[[22,23,65,66,92,105,193,327],"S^2xS^1"],
	[[22,23,89,103,158,276,289,312,328],"B2"],
	[[22,23,90,103,154,155,164,222],"S^2xS^1"],
	[[22,23,93,105,164,203,233,257,260],"S^2xS^1"],
	[[22,24,69,73,86,199,220,280,294],"S^2xS^1"],
	[[22,24,72,79,85,115,164,222,275,335],"L(3,1)"],
	[[22,24,82,180,185,202,214,232,236],"K^2xS^1"],
	[[22,24,90,94,103,150,155,164,222],"S^2xS^1"],
	[[22,24,90,113,180,185,202,218,232],"S^2xS^1"],
	[[22,24,93,116,166,169,274,329],"S^2~S^1"],
	[[22,24,93,116,169,177,200,223,329],"S^2~S^1"],
	[[22,25,53,66,95,105,164,203,233],"S^2xS^1"],
	[[22,25,53,66,126,164,203,248],"S^2xS^1"],
	[[22,25,53,66,126,191,203,220,248],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[22,25,70,77,92,161,180,314,325,335],"L(8,3)"],
	[[22,25,70,77,92,161,180,327,335],"S^3"],
	[[22,25,83,194,232,241,256,275,299],"S^2~S^1"],
	[[22,25,90,123,195,225,276,299],"S^2xS^1"],
	[[22,25,91,124,180,185,200,257,283,329],"S^3"],
	[[22,25,92,123,161,186,224,241,276],"S^3"],
	[[22,25,93,95,105,164,233,260],"S^2xS^1"],
	[[22,25,93,95,105,175,177,202,233],"S^2xS^1"],
	[[22,25,93,126,164,248,260],"S^2xS^1"],
	[[22,25,93,126,172,177,202,257,283],"S^2xS^1"],
	[[22,25,93,126,175,177,202,248],"S^2xS^1"],
	[[22,25,93,126,191,220,248,260],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[22,26,91,130,155,197,233,285,303],"L(3,1)"],
	[[22,27,58,66,93,202,205,232,329],"S^2~S^1"],
	[[22,27,85,192,232,241,256,273,299],"S^2~S^1"],
	[[22,28,53,66,143,157,164,204,225],"S^2xS^1"],
	[[22,28,53,66,148,164,203,225],"S^2xS^1"],
	[[22,28,53,66,148,191,203,220,225],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[22,28,86,180,187,199,225,303,304,331],"L(5,1)"],
	[[22,28,90,145,195,275,297,299],"S^2xS^1"],
	[[22,28,91,128,130,155,197,242,285,303],"L(3,1)"],
	[[22,28,93,148,164,225,260],"S^2xS^1"],
	[[22,28,93,148,175,177,202,225],"S^2xS^1"],
	[[22,28,93,148,191,220,225,260],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[22,29,69,73,83,106,196,280,303,304],"B2"],
	[[22,29,84,134,152,201,255,280,326,335],"B2"],
	[[22,29,89,131,134,152,196,280,331,335],"B2"],
	[[22,29,90,132,134,152,164,255,280,335],"L(3,1)"],
	[[22,30,53,65,156,164,257,273],"S^2xS^1"],
	[[22,30,53,66,143,148,164,203,233],"S^2xS^1"],
	[[22,30,53,66,143,148,191,203,214],"K^2xS^1"],
	[[22,30,53,66,157,164,204,233],"S^2xS^1"],
	[[22,30,53,66,157,191,204,214],"K^2xS^1"],
	[[22,30,54,66,83,95,157,192,203],"T^3"],
	[[22,30,75,77,92,161,185,276,297,335],"S^2~S^1"],
	[[22,30,83,121,180,184,192,241,242,312],"S^2~S^1"],
	[[22,30,92,156,164,258,273],"S^2xS^1"],
	[[22,30,93,143,148,164,233,260],"S^2xS^1"],
	[[22,30,93,143,148,175,177,202,233],"S^2xS^1"],
	[[22,30,93,143,148,191,214,260],"K^2xS^1"],
	[[22,30,93,144,148,155,158,203,257],"T^3"],
	[[22,30,93,157,164,203,204,233,260],"S^2xS^1"],
	[[22,30,93,157,191,203,204,214,260],"K^2xS^1"],
	[[22,31,37,47,70,83,202,260],"K^2xS^1"],
	[[22,31,37,48,67,202,260],"K^2xS^1"],
	[[22,31,39,47,83,95,193,233],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[22,31,39,47,83,95,201,258],"K^2xS^1"],
	[[22,31,39,48,67,70,95,193,233],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[22,31,39,48,67,70,95,201,258],"K^2xS^1"],
	[[22,31,39,49,83,95,158,204,233],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[22,31,41,47,83,201,260],"K^2xS^1"],
	[[22,31,41,48,67,70,201,260],"K^2xS^1"],
	[[22,31,53,59,144,197,257,260,287],"SFS[K^2:(2,1)(2,1)]"],
	[[22,31,53,66,149,153,191,203],"T^3"],
	[[22,31,53,66,160,192,203],"T^3"],
	[[22,31,67,80,150,153,191,260],"T^3"],
	[[22,31,89,164,260,294],"S^2xS^1"],
	[[22,31,89,175,177,202,294],"S^2xS^1"],
	[[22,31,89,191,214,233,260,294],"S^2xS^1"],
	[[22,31,89,191,220,260,294],"S^2xS^1"],
	[[22,31,89,192,228,281,329],"S^2~S^1"],
	[[22,31,89,197,218,222,260,294],"S^2xS^1"],
	[[22,31,89,197,235,236,280,294,295],"K^2xS^1"],
	[[22,31,89,197,235,239,260,294],"K^2xS^1"],
	[[22,31,89,199,236,237,280,294,295],"S^2xS^1"],
	[[22,31,89,199,237,239,260,294],"S^2xS^1"],
	[[22,31,89,201,260,294,314,325],"S^2xS^1"],
	[[22,31,89,201,260,294,327],"S^2xS^1"],
	[[22,31,93,144,148,192,273],"T^3"],
	[[22,31,93,149,153,191,260],"T^3"],
	[[22,31,93,160,192,260],"T^3"],
	[[22,32,37,48,69,82,202,260],"K^2xS^1"],
	[[22,32,39,49,86,98,123,124,199],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[22,32,39,49,91,102,196,241,244],"HS"],
	[[22,32,41,48,69,70,82,201,260],"K^2xS^1"],
	[[22,32,53,61,121,123,191,221,280],"K^2xS^1"],
	[[22,32,91,102,103,196,241,244,280,314],"HS"],
	[[22,33,36,50,53,164],"S^2xS^1"],
	[[22,33,36,50,53,191,214,233],"S^2xS^1"],
	[[22,33,36,50,53,191,220],"S^2xS^1"],
	[[22,33,36,50,53,197,218,222],"S^2xS^1"],
	[[22,33,36,50,53,197,235,239],"K^2xS^1"],
	[[22,33,36,50,53,199,237,239],"S^2xS^1"],
	[[22,33,36,50,53,201,314,325],"S^2xS^1"],
	[[22,33,36,50,53,201,327],"S^2xS^1"],
	[[22,33,36,50,59,86,197,218,221],"S^2xS^1"],
	[[22,33,39,49,53,54,164],"S^2xS^1"],
	[[22,33,39,49,53,54,201,327],"S^2xS^1"],
	[[22,33,53,54,103,164,280,314],"S^2xS^1"],
	[[22,33,53,54,103,201,280,325],"K^2xS^1"],
	[[22,33,53,61,155,164,289,314],"S^2xS^1"],
	[[22,33,53,64,164,275,283,289,314],"S^2xS^1"],
	[[22,33,53,64,164,280,311,312,314],"S^2xS^1"],
	[[22,33,53,64,180,189,192],"T^3"],
	[[22,33,53,64,201,276,312,323],"K^2xS^1"],
	[[22,33,53,64,201,280,311,312,325],"K^2xS^1"],
	[[22,33,53,65,155,156,164],"S^2xS^1"],
	[[22,33,53,65,155,156,201,327],"S^2xS^1"],
	[[22,33,53,66,158,160],"T^3"],
	[[22,33,89,158,203,228,281,329],"S^2~S^1"],
	[[22,33,92,155,156,164,257,258],"S^2xS^1"],
	[[22,33,93,143,144,157,158,204,273],"T^3"],
	[[22,33,93,144,148,158,203,273],"T^3"],
	[[22,35,91,155,157,190,200,329],"S^2~S^1"],
	[[22,35,93,166,169,182,223,274,329],"S^2~S^1"],
	[[22,35,93,169,177,182,200,329],"S^2~S^1"],
	[[22,35,93,185,197,232,280,295,314],"S^2xS^1"],
	[[22,35,93,189,192,260,261],"T^3"],
	[[23,24,37,40,79,104,212,223,335],"S^2~S^1"],
	[[23,24,40,43,72,79,104,212,222,335],"S^2~S^1"],
	[[23,24,98,106,123,150,207,217,335],"S^2~S^1"],
	[[23,25,70,80,105,166,169,193,205,329],"S^2~S^1"],
	[[23,25,70,80,105,168,169,205,327,329],"SFS[D_:(3,1)(3,1)]"],
	[[23,26,37,42,80,105,212,223,335],"S^2~S^1"],
	[[23,26,57,65,104,193,201,212,335],"B2"],
	[[23,26,57,65,104,212,233,258,335],"HS"],
	[[23,26,103,131,158,197,233,285,303],"L(3,1)"],
	[[23,27,72,80,104,115,116,212,222,335],"S^2~S^1"],
	[[23,28,39,47,143,155,158,193,225],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[23,28,39,49,143,155,204,225],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[23,28,54,65,143,156,204,225],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[23,28,57,65,104,128,212,242,258,335],"HS"],
	[[23,28,65,66,104,119,126,274,329],"S^2~S^1"],
	[[23,28,103,128,131,158,197,242,285,303],"L(3,1)"],
	[[23,29,40,46,94,207,323,326,327,335],"S^2~S^1"],
	[[23,29,70,74,83,84,96,205,241,287],"L(3,1)"],
	[[23,29,94,107,130,164,197,212,285,335],"SFS[D_:(3,1)(3,1)]"],
	[[23,29,94,107,130,212,218,222,285,335],"S^2~S^1"],
	[[23,29,94,108,140,155,207,276,327,335],"S^2~S^1"],
	[[23,30,39,47,155,158,193,233],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[23,30,39,47,155,158,201,258],"K^2xS^1"],
	[[23,30,39,49,155,204,214,220],"S^2xS^1"],
	[[23,30,39,49,155,204,233],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[23,30,40,45,94,167,172,225],"S^2xS^1"],
	[[23,30,40,47,94,149,167,172,216],"S^2xS^1"],
	[[23,30,54,59,143,192,197,204,287],"K^2xS^1"],
	[[23,30,54,61,204,235,289],"K^2xS^1"],
	[[23,30,54,65,156,204,214,220],"S^2xS^1"],
	[[23,30,54,65,156,204,233],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[23,30,54,66,157,257,273],"T^3"],
	[[23,30,59,61,98,204,235,288],"K^2xS^1"],
	[[23,30,69,80,94,153,204,214],"T^3"],
	[[23,30,96,130,164,197,212,285,335],"SFS[D_:(3,1)(3,1)]"],
	[[23,30,96,130,167,169,194,197,285,335],"SFS[D_:(3,1)(3,1)]"],
	[[23,30,96,130,212,218,222,285,335],"S^2~S^1"],
	[[23,30,105,143,148,203,257],"T^3"],
	[[23,30,105,157,204,257],"T^3"],
	[[23,31,105,143,148,155,158,192,273],"T^3"],
	[[23,32,52,66,69,94,153,155],"T^3"],
	[[23,32,54,57,118,123,127,207,249,289],"S^2~S^1"],
	[[23,32,54,65,113,115,210,310,318],"S^2xS^1"],
	[[23,32,102,178,190,244,245,335],"S^2~S^1"],
	[[23,33,52,54,77,178,190,335],"S^2~S^1"],
	[[23,33,54,66,155,157],"T^3"],
	[[23,33,105,143,148,155,203,273],"T^3"],
	[[23,33,105,149,153,158,192,214,257],"K^2xS^1"],
	[[23,35,54,66,190,281,329],"S^2~S^1"],
	[[23,35,58,66,97,190,280,329],"S^2~S^1"],
	[[23,35,70,80,95,172,175,178,216],"S^2xS^1"],
	[[23,35,95,126,178,248,257],"S^2xS^1"],
	[[23,35,105,164,178,191,214,257],"S^2xS^1"],
	[[23,35,105,178,193,197,257,318],"S^2xS^1"],
	[[23,35,105,178,193,201,257,258],"K^2xS^1"],
	[[23,35,105,178,214,220,257],"S^2xS^1"],
	[[23,35,105,178,233,257],"S^2xS^1"],
	[[23,35,105,178,235,257,314],"S^2xS^1"],
	[[23,35,105,178,235,257,325,327],"S^2xS^1"],
	[[23,35,105,178,239,257,258,316],"S^2xS^1"],
	[[23,35,105,190,204,273,281,329],"S^2~S^1"],
	[[24,25,40,47,123,149,150,299,323],"T^3"],
	[[24,27,57,58,107,182,185,217,326,329],"(S^2xS^1)#12"],
	[[24,27,57,62,107,138,182,185,217,329],"(S^2xS^1)#12"],
	[[24,27,58,62,107,131,182,185,217,329],"(S^2~S^1)#12"],
	[[24,27,84,85,107,182,185,217,329,330],"(S^2~S^1)#12"],
	[[24,27,84,91,107,140,182,185,217,329],"(S^2xS^1)#12"],
	[[24,27,85,91,107,133,182,185,217,329],"(S^2~S^1)#12"],
	[[24,27,107,131,138,182,185,217,326,329],"(S^2~S^1)#12"],
	[[24,27,107,133,140,182,185,217,329,330],"(S^2xS^1)#12"],
	[[24,28,40,45,86,90,150,216,323],"S^2xS^1"],
	[[24,28,40,45,145,150,216,327],"S^2xS^1"],
	[[24,28,40,47,145,149,150,225,327],"S^2xS^1"],
	[[24,28,58,59,108,216,292,325,326,329],"B2"],
	[[24,28,58,59,108,220,249,292,326,329],"HS"],
	[[24,28,59,62,108,138,216,292,325,329],"B2"],
	[[24,28,59,62,108,138,220,249,292,329],"HS"],
	[[24,28,82,84,128,182,186,220,329,330],"HS"],
	[[24,28,82,91,128,133,182,186,220,329],"HS"],
	[[24,28,108,138,144,216,298,325,329],"B2"],
	[[24,28,108,138,144,220,249,298,329],"HS"],
	[[24,29,107,132,150,218,285,335],"S^2~S^1"],
	[[24,29,113,150,217,285,335],"S^2~S^1"],
	[[24,30,94,96,132,150,218,285,335],"S^2~S^1"],
	[[24,32,39,49,94,150,155,214,233],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[24,32,39,49,94,150,155,220],"S^2xS^1"],
	[[24,32,54,61,94,150,214,235,289],"K^2xS^1"],
	[[24,32,54,61,94,150,220,289,314],"S^2xS^1"],
	[[24,32,54,65,94,150,156,214,233],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[24,32,54,65,94,150,156,220],"S^2xS^1"],
	[[24,32,67,69,89,164,191,294],"S^2xS^1"],
	[[24,32,67,69,89,168,170,249,294],"S^2xS^1"],
	[[24,32,67,69,89,172,174,216,294],"S^2xS^1"],
	[[24,32,67,69,89,191,201,294,327],"S^2xS^1"],
	[[24,32,67,69,89,214,233,294],"S^2xS^1"],
	[[24,32,67,69,89,214,235,294,314],"S^2xS^1"],
	[[24,32,67,69,89,216,249,294,325],"K^2xS^1"],
	[[24,32,67,69,89,220,294],"S^2xS^1"],
	[[24,32,69,73,86,90,164,191,294],"S^2xS^1"],
	[[24,32,69,73,86,90,220,294],"S^2xS^1"],
	[[24,32,69,73,145,220,294,323,327],"S^2xS^1"],
	[[24,32,69,76,122,123,164,191,294],"S^2xS^1"],
	[[24,32,69,76,122,123,220,294],"S^2xS^1"],
	[[24,32,69,80,150,153],"T^3"],
	[[24,33,36,50,53,82,214,233,260],"S^2xS^1"],
	[[24,33,36,50,53,82,220,260],"S^2xS^1"],
	[[24,33,52,54,69,102,221,294],"T^3"],
	[[24,33,107,133,220,253,257,289,298,303],"S^3"],
	[[25,27,83,85,232,241,256,299],"S^2~S^1"],
	[[25,27,83,92,141,232,241,256,326],"S^2~S^1"],
	[[25,27,118,225,241,297],"S^2~S^1"],
	[[25,28,73,76,122,225,299],"S^2xS^1"],
	[[25,28,86,90,123,225,299],"S^2xS^1"],
	[[25,28,119,224,241,248],"S^2~S^1"],
	[[25,29,36,42,62,124,134,149,217,329],"B2"],
	[[25,29,70,76,149,222,255,298,329,330],"L(3,1)"],
	[[25,29,118,138,149,216,298,325,329],"B2"],
	[[25,29,118,138,149,220,249,298,329],"HS"],
	[[25,29,122,149,196,197,218,298,329],"L(5,1)"],
	[[25,29,122,149,196,200,218,257,285,329],"L(5,1)"],
	[[25,29,122,149,218,254,257,285,322,329],"L(5,1)"],
	[[25,29,124,138,140,149,217,298,325,329],"B2"],
	[[25,32,67,70,92,149,152,220,327],"S^2xS^1"],
	[[25,32,123,195,199,225,276,280,299],"S^2xS^1"],
	[[25,33,124,165,172,185,203,248],"S^2xS^1"],
	[[25,33,124,172,174,203,210,248,310],"S^2xS^1"],
	[[25,33,124,203,210,248,249,310,325],"K^2xS^1"],
	[[25,35,126,178,203,248],"S^2xS^1"],
	[[25,35,126,190,225,282,297,303],"S^2~S^1"],
	[[26,27,42,43,194,196,234,298,322,335],"(S^2xS^1)#12"],
	[[26,27,42,43,194,200,234,254,298,335],"(S^2~S^1)#12"],
	[[26,27,42,43,196,200,234,245,298,335],"(S^2xS^1)#12"],
	[[26,27,42,43,205,207,234,298,331,335],"(S^2~S^1)#12"],
	[[26,27,42,43,205,212,234,255,298,335],"(S^2~S^1)#12"],
	[[26,27,42,43,207,212,234,246,298,335],"(S^2xS^1)#12"],
	[[26,27,42,43,234,245,254,298,322,335],"(S^2~S^1)#12"],
	[[26,27,42,43,234,246,255,298,331,335],"(S^2xS^1)#12"],
	[[26,27,42,46,137,194,196,234,322,335],"(S^2~S^1)#12"],
	[[26,27,42,46,137,194,200,234,254,335],"(S^2~S^1)#12"],
	[[26,27,42,46,137,196,200,234,245,335],"(S^2~S^1)#12"],
	[[26,27,42,46,137,205,207,234,331,335],"(S^2~S^1)#12"],
	[[26,27,42,46,137,205,212,234,255,335],"(S^2~S^1)#12"],
	[[26,27,42,46,137,207,212,234,246,335],"(S^2~S^1)#12"],
	[[26,27,42,46,137,234,245,254,322,335],"(S^2~S^1)#12"],
	[[26,27,42,46,137,234,246,255,331,335],"(S^2~S^1)#12"],
	[[26,27,43,46,130,194,196,234,322,335],"(S^2~S^1)#12"],
	[[26,27,43,46,130,194,200,234,254,335],"(S^2xS^1)#12"],
	[[26,27,43,46,130,196,200,234,245,335],"(S^2~S^1)#12"],
	[[26,27,43,46,130,205,207,234,331,335],"(S^2xS^1)#12"],
	[[26,27,43,46,130,205,212,234,255,335],"(S^2xS^1)#12"],
	[[26,27,43,46,130,207,212,234,246,335],"(S^2~S^1)#12"],
	[[26,27,43,46,130,234,245,254,322,335],"(S^2xS^1)#12"],
	[[26,27,43,46,130,234,246,255,331,335],"(S^2~S^1)#12"],
	[[26,27,57,58,182,184,234,292,326,329],"(S^2~S^1)#12"],
	[[26,27,57,58,182,187,234,253,326,329],"(S^2xS^1)#12"],
	[[26,27,57,58,184,187,234,244,326,329],"(S^2~S^1)#12"],
	[[26,27,57,58,194,196,234,303,322,326],"(S^2~S^1)#12"],
	[[26,27,57,58,194,200,234,254,303,326],"(S^2~S^1)#12"],
	[[26,27,57,58,196,200,234,245,303,326],"(S^2xS^1)#12"],
	[[26,27,57,58,234,245,254,303,322,326],"(S^2~S^1)#12"],
	[[26,27,57,62,138,182,184,234,292,329],"(S^2~S^1)#12"],
	[[26,27,57,62,138,182,187,234,253,329],"(S^2xS^1)#12"],
	[[26,27,57,62,138,184,187,234,244,329],"(S^2~S^1)#12"],
	[[26,27,57,62,138,194,196,234,303,322],"(S^2~S^1)#12"],
	[[26,27,57,62,138,194,200,234,254,303],"(S^2~S^1)#12"],
	[[26,27,57,62,138,196,200,234,245,303],"(S^2~S^1)#12"],
	[[26,27,57,62,138,234,245,254,303,322],"(S^2xS^1)#12"],
	[[26,27,58,62,131,182,184,234,292,329],"(S^2~S^1)#12"],
	[[26,27,58,62,131,182,187,234,253,329],"(S^2~S^1)#12"],
	[[26,27,58,62,131,184,187,234,244,329],"(S^2~S^1)#12"],
	[[26,27,58,62,131,194,196,234,303,322],"(S^2xS^1)#12"],
	[[26,27,58,62,131,194,200,234,254,303],"(S^2~S^1)#12"],
	[[26,27,58,62,131,196,200,234,245,303],"(S^2~S^1)#12"],
	[[26,27,58,62,131,234,245,254,303,322],"(S^2~S^1)#12"],
	[[26,27,72,77,132,234,241,274,281,313],"(S^2~S^1)#12"],
	[[26,27,84,85,182,184,234,292,329,330],"(S^2~S^1)#12"],
	[[26,27,84,85,182,187,234,253,329,330],"(S^2~S^1)#12"],
	[[26,27,84,85,184,187,234,244,329,330],"(S^2~S^1)#12"],
	[[26,27,84,85,234,246,255,303,330,331],"(S^2~S^1)#12"],
	[[26,27,84,91,140,182,184,234,292,329],"(S^2~S^1)#12"],
	[[26,27,84,91,140,182,187,234,253,329],"(S^2xS^1)#12"],
	[[26,27,84,91,140,184,187,234,244,329],"(S^2~S^1)#12"],
	[[26,27,84,91,140,205,207,234,303,331],"(S^2xS^1)#12"],
	[[26,27,84,91,140,207,212,234,246,303],"(S^2~S^1)#12"],
	[[26,27,84,91,140,234,246,255,303,331],"(S^2~S^1)#12"],
	[[26,27,85,91,133,182,184,234,292,329],"(S^2~S^1)#12"],
	[[26,27,85,91,133,182,187,234,253,329],"(S^2~S^1)#12"],
	[[26,27,85,91,133,184,187,234,244,329],"(S^2~S^1)#12"],
	[[26,27,85,91,133,205,207,234,303,331],"(S^2~S^1)#12"],
	[[26,27,85,91,133,207,212,234,246,303],"(S^2~S^1)#12"],
	[[26,27,85,91,133,234,246,255,303,331],"(S^2xS^1)#12"],
	[[26,27,130,137,194,196,234,298,322,335],"(S^2~S^1)#12"],
	[[26,27,130,137,194,200,234,254,298,335],"(S^2~S^1)#12"],
	[[26,27,130,137,196,200,234,245,298,335],"(S^2~S^1)#12"],
	[[26,27,130,137,205,207,234,298,331,335],"(S^2~S^1)#12"],
	[[26,27,130,137,207,212,234,246,298,335],"(S^2~S^1)#12"],
	[[26,27,130,137,234,245,254,298,322,335],"(S^2~S^1)#12"],
	[[26,27,130,137,234,246,255,298,331,335],"(S^2~S^1)#12"],
	[[26,27,133,140,182,184,234,292,329,330],"(S^2~S^1)#12"],
	[[26,27,133,140,182,187,234,253,329,330],"(S^2xS^1)#12"],
	[[26,27,133,140,184,187,234,244,329,330],"(S^2~S^1)#12"],
	[[26,27,133,140,205,207,234,303,330,331],"(S^2~S^1)#12"],
	[[26,27,133,140,207,212,234,246,303,330],"(S^2xS^1)#12"],
	[[26,27,133,140,234,246,255,303,330,331],"(S^2~S^1)#12"],
	[[26,29,42,43,136,194,196,298,322,335],"(S^2~S^1)#12"],
	[[26,29,42,43,136,196,200,245,298,335],"(S^2~S^1)#12"],
	[[26,29,42,43,136,205,207,298,331,335],"(S^2xS^1)#12"],
	[[26,29,42,43,136,207,212,246,298,335],"(S^2~S^1)#12"],
	[[26,29,42,43,136,245,254,298,322,335],"(S^2xS^1)#12"],
	[[26,29,42,43,136,246,255,298,331,335],"(S^2~S^1)#12"],
	[[26,29,42,46,136,137,194,196,322,335],"(S^2~S^1)#12"],
	[[26,29,42,46,136,137,196,200,245,335],"(S^2~S^1)#12"],
	[[26,29,42,46,136,137,205,207,331,335],"(S^2~S^1)#12"],
	[[26,29,42,46,136,137,207,212,246,335],"(S^2~S^1)#12"],
	[[26,29,42,46,136,137,245,254,322,335],"(S^2~S^1)#12"],
	[[26,29,42,46,136,137,246,255,331,335],"(S^2~S^1)#12"],
	[[26,29,57,58,136,182,184,292,326,329],"(S^2~S^1)#12"],
	[[26,29,57,58,136,182,187,253,326,329],"(S^2~S^1)#12"],
	[[26,29,57,58,136,184,187,244,326,329],"(S^2xS^1)#12"],
	[[26,29,57,62,136,138,182,184,292,329],"(S^2~S^1)#12"],
	[[26,29,57,62,136,138,182,187,253,329],"(S^2~S^1)#12"],
	[[26,29,57,62,136,138,184,187,244,329],"(S^2xS^1)#12"],
	[[26,29,57,62,136,138,196,200,245,303],"(S^2~S^1)#12"],
	[[26,29,57,62,136,138,245,254,303,322],"(S^2~S^1)#12"],
	[[26,29,58,62,131,136,182,184,292,329],"(S^2xS^1)#12"],
	[[26,29,58,62,131,136,182,187,253,329],"(S^2~S^1)#12"],
	[[26,29,58,62,131,136,184,187,244,329],"(S^2~S^1)#12"],
	[[26,29,58,62,131,136,196,200,245,303],"(S^2~S^1)#12"],
	[[26,29,58,62,131,136,245,254,303,322],"(S^2xS^1)#12"],
	[[26,29,72,77,132,136,241,274,281,313],"(S^2~S^1)#12"],
	[[26,29,84,91,136,140,182,184,292,329],"(S^2~S^1)#12"],
	[[26,29,84,91,136,140,182,187,253,329],"(S^2~S^1)#12"],
	[[26,29,84,91,136,140,184,187,244,329],"(S^2xS^1)#12"],
	[[26,29,84,91,136,140,205,207,303,331],"(S^2~S^1)#12"],
	[[26,29,84,91,136,140,207,212,246,303],"(S^2~S^1)#12"],
	[[26,29,84,91,136,140,246,255,303,331],"(S^2xS^1)#12"],
	[[26,29,85,91,133,136,182,184,292,329],"(S^2xS^1)#12"],
	[[26,29,85,91,133,136,182,187,253,329],"(S^2~S^1)#12"],
	[[26,29,85,91,133,136,184,187,244,329],"(S^2~S^1)#12"],
	[[26,29,85,91,133,136,205,207,303,331],"(S^2xS^1)#12"],
	[[26,29,85,91,133,136,207,212,246,303],"(S^2~S^1)#12"],
	[[26,29,85,91,133,136,246,255,303,331],"(S^2~S^1)#12"],
	[[26,29,130,136,137,246,255,298,331,335],"(S^2~S^1)#12"],
	[[26,33,53,57,91,233,258,303,313],"SFS[D_:(3,1)(3,1)]"],
	[[26,33,53,57,91,239,303,313,316],"S^2~S^1"],
	[[26,33,57,64,233,254,255,258,303,313],"SFS[D_:(3,1)(3,1)]"],
	[[26,33,57,64,239,254,255,303,313,316],"S^2~S^1"],
	[[26,33,133,197,200,233,285,289,303],"L(3,1)"],
	[[26,33,133,214,220,257,289,298,303],"S^3"],
	[[26,33,133,233,257,289,298,303],"L(3,1)"],
	[[27,28,118,119,241],"S^2~S^1"],
	[[27,29,42,43,129,196,200,245,298,335],"(S^2~S^1)#12"],
	[[27,29,42,43,129,205,207,298,331,335],"(S^2~S^1)#12"],
	[[27,29,42,43,129,207,212,246,298,335],"(S^2~S^1)#12"],
	[[27,29,42,43,129,245,254,298,322,335],"(S^2~S^1)#12"],
	[[27,29,42,46,129,137,196,200,245,335],"(S^2xS^1)#12"],
	[[27,29,42,46,129,137,205,207,331,335],"(S^2~S^1)#12"],
	[[27,29,42,46,129,137,207,212,246,335],"(S^2xS^1)#12"],
	[[27,29,42,46,129,137,245,254,322,335],"(S^2~S^1)#12"],
	[[27,29,57,62,129,138,184,187,244,329],"(S^2~S^1)#12"],
	[[27,29,57,62,129,138,196,200,245,303],"(S^2~S^1)#12"],
	[[27,29,57,62,129,138,245,254,303,322],"(S^2xS^1)#12"],
	[[27,29,72,77,129,132,241,274,281,313],"(S^2~S^1)#12"],
	[[27,29,84,91,129,140,184,187,244,329],"(S^2~S^1)#12"],
	[[27,29,84,91,129,140,205,207,303,331],"(S^2xS^1)#12"],
	[[27,29,84,91,129,140,207,212,246,303],"(S^2~S^1)#12"],
	[[27,29,106,108,119,241],"S^2~S^1"],
	[[28,32,108,113,138,144,249,295,298,329],"HS"],
	[[28,33,36,50,59,216,221,276,325],"K^2xS^1"],
	[[28,33,36,50,59,220,221,249,276],"SFS[S^2:(2,1)(2,1)(3,1)(3,-4)]"],
	[[28,33,53,57,91,128,242,258,303,313],"SFS[D_:(3,1)(3,1)]"],
	[[28,33,128,133,197,200,242,285,289,303],"L(3,1)"],
	[[28,33,128,133,242,257,289,298,303],"L(3,1)"],
	[[28,35,119,126,190,282,303],"S^2~S^1"],
	[[36,42,52,67,92,127,138],"S^2~S^1"],
	[[36,49,65,178,190,203,224,241],"S^2~S^1"],
	[[36,49,65,282,303],"S^2~S^1"],
	[[42,43,72,77,132,169,173,224,298,303],"(S^2~S^1)#12"],
	[[42,43,84,91,140,184,187,241,244,298],"(S^2~S^1)#12"],
	[[42,43,84,91,140,217,223,241,247,298],"(S^2~S^1)#12"],
	[[42,50,134,273,303,304],"S^2~S^1"],
	[[72,77,127,132],"S^2~S^1"]];

	if type = 1 then
		for k in [5..22] do
			if i in SCCyclic3MfldTopTypes(k) then
				for l in [1..SCNrCyclic3Mflds(k)] do
					if dc[k][l][2] = i then 
						j:=l;
						i:=k;
						break;
					fi;
				od;
			fi;
			if j <> 0 then break; fi;				
		od;
	fi;

	if type =2 then
		list:=[];
		for k in [5..22] do
			topTypes:=SCCyclic3MfldTopTypes(k);
			if i in SCCyclic3MfldTopTypes(k) then
				for l in [1..SCNrCyclic3Mflds(k)] do
					if dc[k][l][2] = i then 
						Add(list,[k,l]);
					fi;
				od;
			fi;
		od;
		return list;
	fi;

	list:=[];
	for k in dc[i][j][1] do
		Add(list,all[i][k]);
	od;
	
	c:=SCFromDifferenceCycles(list);
	SetSCDifferenceCycles(c,list);
	SCRename(c,Concatenation("Cyclic 3-mfld (",String(i),",",String(j),"): ",
    dc[i][j][2]));
	SetSCTopologicalType(c,dc[i][j][2]);
	SetSCAutomorphismGroupTransitivity(c,1);
	SetSCIsManifold(c,true);
	
	return c;
end;
	

################################################################################
##<#GAPDoc Label="SCCyclic3Mfld">
## <ManSection>
## <Func Name="SCCyclic3Mfld" Arg="i,j"/>
## <Returns> simplicial complex of type <C>SCSimplicialComplex</C> upon 
## success, <K>fail</K> otherwise.</Returns>
## <Description>	
## Returns the <Arg>j</Arg>th combinatorial 3-manifold with <Arg>i</Arg> 
## vertices in the classification of combinatorial 3-manifolds with transitive 
## cyclic symmetry.
## 
## See <Cite Key="Spreer11CyclicCombMflds"/> for more about the classification 
## of combinatorial 3-manifolds with transitive cyclic symmetry up to 
## <M>22</M> vertices.
## <Example><![CDATA[
## gap> SCCyclic3Mfld(15,34);
## ]]></Example>
## </Description>
## </ManSection>
##<#/GAPDoc>
################################################################################
InstallGlobalFunction(SCCyclic3Mfld,
function(i,j)
	return SCIntFunc.SCCyclic3MfldEx(i,j,0);
end);

################################################################################
##<#GAPDoc Label="SCCyclic3MfldByType">
## <ManSection>
## <Func Name="SCCyclic3MfldByType" Arg="type"/>
## <Returns> simplicial complex of type <C>SCSimplicialComplex</C> upon 
## success, <K>fail</K> otherwise.</Returns>
## <Description>	
## Returns the smallest combinatorial 3-manifolds in the classification of 
## combinatorial 3-manifolds with transitive cyclic symmetry of topological 
## type <Arg>type</Arg>.
## 
## See <Cite Key="Spreer11CyclicCombMflds"/> for more about the classification 
## of combinatorial 3-manifolds with transitive cyclic symmetry up to 
## <M>22</M> vertices.
## <Example><![CDATA[
## gap> SCCyclic3MfldByType("T^3");
## ]]></Example>
## </Description>
## </ManSection>
##<#/GAPDoc>
################################################################################
InstallGlobalFunction(SCCyclic3MfldByType,
function(type)
	return SCIntFunc.SCCyclic3MfldEx(type,0,1);
end);




################################################################################
##<#GAPDoc Label="SCCyclic3MfldListOfGivenType">
## <ManSection>
## <Func Name="SCCyclic3MfldListOfGivenType" Arg="type"/>
## <Returns> simplicial complex of type <C>SCSimplicialComplex</C> upon 
## success, <K>fail</K> otherwise.</Returns>
## <Description>	
## Returns a list of indices 
## <M>\{ (i_1, j_1) , (i_1, j_1) , \ldots (i_n, j_n) \}</M> of all 
## combinatorial 3-manifolds in the classification of combinatorial 
## 3-manifolds with transitive cyclic symmetry of topological type 
## <Arg>type</Arg>. Complexes can be obtained by calling 
## <Ref Func="SCCyclic3Mfld" /> using these indices.
## 
## See <Cite Key="Spreer11CyclicCombMflds"/> for more about the 
## classification of combinatorial 3-manifolds with transitive cyclic 
## symmetry up to <M>22</M> vertices.
## <Example><![CDATA[
## gap> SCCyclic3MfldListOfGivenType("Sigma(2,3,7)");
## ]]></Example>
## </Description>
## </ManSection>
##<#/GAPDoc>
################################################################################
InstallGlobalFunction(SCCyclic3MfldListOfGivenType,
function(type)
	return SCIntFunc.SCCyclic3MfldEx(type,0,2);
end);

