# (c) Copyright 2014, 2017-2021. CodeWeavers, Inc.

"""Defines classes to represent and manipulate the CXFixes information."""

import re

# for localization
from cxutils import cxgettext as _


#####
#
# Helper functions for validating fields
#
#####

def _validate_regular_expression(pattern, location, allow_empty=True):
    if pattern:
        try:
            if re.match(pattern, '') and not allow_empty:
                raise AttributeError("the %s regular expression for %s is overly broad" % (pattern, location))
        except re.error:
            raise AttributeError("%s is an invalid regular expression for %s" % (pattern, location)) #pylint: disable=W0707


#####
#
# Helper functions for dumping the profiles
#
#####

def _dump_value(out, indent, value):
    if hasattr(value, 'dump'):
        out.write(" %s\n" % type(value))
        value.dump(out, indent + "| ")
    else:
        out.write(" %s\n" % repr(value))

def _dump_fields(out, indent, obj, fields):
    for field in fields:
        if field not in obj.__dict__:
            continue
        value = obj.__dict__[field]
        if isinstance(value, dict):
            for key in sorted(value):
                out.write("%s%s{%s} =" % (indent, field, key))
                _dump_value(out, indent, value[key])
        elif isinstance(value, list):
            for (i, item) in enumerate(value):
                out.write("%s%s[%d] =" % (indent, field, i))
                _dump_value(out, indent, item)
        else:
            out.write("%s%s =" % (indent, field))
            _dump_value(out, indent, value)


#####
#
# CXDistribution & co
#
#####

class CXDistGlob:
    """Contains the information needed to detect a given distribution."""

    ### Property defaults

    # These class variables provide defaults in case the corresponding
    # instance variable is not set. They should all be immutable objects,

    # This distribution can only be a match for the current system if it
    # contains a file matching this glob.
    # This field is mandatory and thus defaults to None.
    file_glob = None

    # If this list is empty, then finding a file matching the above glob is
    # enough. Otherwise the content of that file must also match all the
    # regular expression patterns in this list in order for the distribution
    # to be a match. This field is optional.
    patterns = tuple()

    def __init__(self, file_glob, patterns=None):
        self.file_glob = file_glob
        if patterns:
            self.patterns = patterns

    ### Instance validation and dumping

    def validate(self, distid):
        """Checks that all the mandatory fields have been set for this
        distribution glob.
        """
        if not self.file_glob:
            raise AttributeError("CXDistGlob.file_glob is not set in %s" % distid)
        for pattern in self.patterns:
            _validate_regular_expression(pattern, "the CXDistGlob.patterns field of %s" % distid)
        return True

    def dump(self, out, indent=""):
        _dump_fields(out, indent, self, ('file_glob', 'patterns'))


class CXDistribution:
    """Describes a distribution: its name, how to install packages, etc."""

    ### Property defaults

    # These class variables provide defaults in case the corresponding
    # instance variable is not set. They should all be immutable objects,

    # This distribution's identifier.
    # This field is mandatory and thus defaults to None.
    distid = None

    # A more human-readable name for this distribution.
    # This field is mandatory and thus defaults to None.
    name = None

    # The priority is used to pick a distribution in case more than one seems
    # to match the current platform. This field is optional.
    priority = 0

    # The command to run to cause the distribution to refresh its known
    # packages list. This field is optional.
    updatecmd = None

    # The command to run before installing the needed packages to make sure the
    # the packaging system lock is available.
    # This field is optional and defaults to None.
    lockcmd = None

    # The command to run to install the needed packages. This assumes it can
    # simply be given a space-separated list of packages to install.
    # This field is mandatory and thus defaults to None.
    packagecmd = None

    # A list of CXDistGlobs to be used to detect the current platform.
    # This field is mandatory and thus defaults to None.
    globs = None

    # A distribution id to be used as a fallback if there is no fix
    # for this one. This field is optional.
    fallback = None

    def __init__(self, distid, name, priority):
        self.distid = distid
        self.name = name
        self.priority = priority

    def add_glob(self, glob):
        if self.globs is None:
            self.globs = [glob]
        else:
            self.globs.append(glob)

    ### Instance validation and dumping

    def validate(self):
        """Checks that all the mandatory fields have been set for this
        distribution.

        Note that this only checks internal consistency, not consistency with
        other distributions.
        """
        if not self.distid:
            raise AttributeError("CXDistribution.distid is not set")
        if not self.name:
            raise AttributeError("CXDistribution.name is not set")
        # Note that in the future distributions may define other methods of
        # fixing errors than 'packagecmd', and may not support 'packagecmd'.
        # So don't require 'packagecmd' to be set so old and new-style
        # distributions can be mixed in a single XML file.
        for glob in self.globs:
            glob.validate(self.distid)
        return True

    def dump(self, out, indent=""):
        _dump_fields(out, indent, self, ('distid', 'name', 'priority', 'updatecmd', 'lockcmd', 'packagecmd', 'globs', 'fallback'))


class CXProduct:
    """Identifies a CrossOver product so fixes can be assigned priority
    levels for it."""

    ### Property defaults

    # These class variables provide defaults in case the corresponding
    # instance variable is not set. They should all be immutable objects,

    # This CrossOver product's identifier (for the cxfixes purposes).
    # This field is mandatory and thus defaults to None.
    cxid = None

    # A CrossOver product id (cxid) to be used as a fallback if a fix
    # specifies no priority level for this one. This field is optional.
    fallback = None

    # A regular expression identifying the CrossOver productids to match.
    # This field is optional.
    productid = None

    # A regular expression identifying the CrossOver versions to match.
    # This field is optional.
    version = None

    # The priority is used identify the current CrossOver product in case
    # more than one matches. This field is optional.
    priority = 0

    def __init__(self, cxid, fallback, productid, version, priority):
        self.cxid = cxid
        self.fallback = fallback
        self.productid = productid
        self.version = version
        self.priority = priority

    ### Instance validation and dumping

    def validate(self):
        """Checks that all the mandatory fields have been set for this
        cxproduct.

        Note that this only checks internal consistency, not consistency with
        other cxproducts.
        """
        if not self.cxid:
            raise AttributeError("CXProduct.cxid is not set")
        if self.productid:
            _validate_regular_expression(self.productid, "the CXProduct.productid field of %s" % self.cxid)
        if self.version:
            _validate_regular_expression(self.version, "the CXProduct.version field of %s" % self.cxid)
        return True

    def dump(self, out, indent=""):
        _dump_fields(out, indent, self, ('cxid', 'fallback', 'productid', 'version', 'priority'))


#####
#
# CXFix & co
#
#####

class CXDistFix:
    """Describes how to fix a specific issue on the specified platform."""

    ### Property defaults

    # These class variables provide defaults in case the corresponding
    # instance variable is not set. They should all be immutable objects,

    # The identifier of the distribution this fix applies to.
    # This field is mandatory and thus defaults to None.
    distid = None

    # The bitness this fix applies to. This is either '32', '64', or None if
    # the fix is bitness-independent. This field is optional.
    bitness = None

    # A set of packages to install before installing the remainder of the
    # packages.
    # This field is optional.
    prepackages = frozenset()

    # A set of packages to install to fix this issue. If this set is empty
    # it means there is no fix for this distribution+bitness combination.
    # This field is optional.
    packages = frozenset()

    def __init__(self, distid, bitness):
        self.distid = distid
        self.bitness = bitness

    ### Instance validation and dumping

    BITNESSES = frozenset((None, '32', '64'))

    # A regular expression to validate package names
    _PACKAGE_RE = re.compile(r'^[a-zA-Z0-9+_.:-]+$')

    def display_name(self, errid):
        if errid:
            name = errid + "/" + self.distid
        else:
            name = self.distid
        if self.bitness:
            name += ":" + self.bitness
        return name

    def validate(self, errid):
        """Checks that all the mandatory fields have been set for this
        distribution fix.

        Note that this only checks internal consistency, not consistency with
        the known distributions list.
        """
        if not self.distid:
            raise AttributeError("CXDistFix.distid is not set in issue %s" % errid)
        if self.bitness not in CXDistFix.BITNESSES:
            raise AttributeError("Invalid bitness value %s for %s" % (self.bitness, self.display_name(errid)))
        if self.bitness and self.bitness != '64' and errid.endswith(".amd64"):
            raise AttributeError("The %s error cannot happen on the 32-bit %s distribution" % (errid, self.distid))
        for package in self.packages:
            if not CXDistFix._PACKAGE_RE.search(package):
                raise AttributeError("invalid package name '%s' in %s" % (package, self.display_name(errid)))
        return True

    def dump(self, out, indent=""):
        _dump_fields(out, indent, self, ('distid', 'bitness', 'packages'))


class CXFix:
    """Groups the known fixes for a given error."""

    ### Property defaults

    # These class variables provide defaults in case the corresponding
    # instance variable is not set. They should all be immutable objects,

    # The identifier of the error this provides fixes for.
    # This field is mandatory and thus defaults to None.
    errid = None

    # If no fix can be found for the current distribution in this object,
    # try to apply the fixes for the fallback error id.
    # This field is optional and defaults to None.
    fallback = None

    # If the error corresponds to a missing library, this is the filename
    # of the missing library.
    # This field is optional and defaults to None.
    lib = None

    # This is the title of the error.
    # This can either be built from the lib field. In other cases it is
    # mandatory and defaults to None.
    title = None

    # This is a description of the error for the user.
    # This field is mandatory and thus defaults to None.
    description = None

    # A set of issues that prevent the detection of this one. When fixing a
    # 'maskedby' issue, fix this one too if it has the appropriate level.
    # This field is optional.
    maskedby = frozenset()

    # A mapping of the CXProduct ids to the corresponding fix priority level.
    # This field is optional.
    cxlevels = {}

    # A set of distribution-specific fixes, indexed by the distribution id and
    # bitness tuple. If bitness is irrelevant then None is used.
    # This field is optional.
    distfixes = {}

    def __init__(self, errid, fallback, lib, title, description):
        self.errid = errid
        if fallback:
            self.fallback = fallback
        if lib:
            self.lib = lib
        if title:
            self.title = title
        if description:
            self.description = description

    def add_distfix(self, distfix):
        if 'distfixes' not in self.__dict__:
            self.distfixes = {}
        self.distfixes[(distfix.distid, distfix.bitness)] = distfix

    ### Instance validation and dumping

    def validate(self):
        """Checks that all the mandatory fields have been set for this fix.

        Note that this only checks internal consistency, not consistency with
        other fixes.
        """
        if not self.errid:
            raise AttributeError("CXFix.errid is not set")
        if not self.distfixes and not self.fallback:
            raise AttributeError("No actual fixes are present for CXFix %s" % self.errid)
        is64bit = self.errid.endswith(".amd64")
        if self.fallback and self.fallback.endswith(".amd64") != is64bit:
            raise AttributeError("32-/64-bit mismatch in fallback: %s vs. %s" % (self.errid, self.fallback))
        for maskedby in self.maskedby:
            if maskedby.endswith(".amd64") != is64bit:
                raise AttributeError("32-/64-bit mismatch in <tiedto> or <maskedby>: %s vs. %s" % (self.errid, maskedby))
        for distfix in self.distfixes.values():
            distfix.validate(self.errid)
        return True

    def dump(self, out, indent=""):
        _dump_fields(out, indent, self, ('errid', 'fallback', 'lib', 'title', 'description', 'maskedby', 'distfixes'))


#####
#
# CXFixes data
#
#####

class CXFixes:
    """All the data from a CXFixes file."""

    def __init__(self):
        self.release = 0
        self.distributions = {}
        self.cxproducts = {}
        self.fixes = {}
        # Set to None to ensure it is not used before validate() fills it in
        self.maskedby = None

    @classmethod
    def _fallback_check(cls, objlabel, objmap, obj, keyfield, checked, errors):
        """Checks for missing and looping fallbacks."""
        seed = obj.__dict__[keyfield]
        if seed not in checked and obj.fallback:
            seen = set((seed,))
            while obj.fallback:
                key = obj.__dict__[keyfield]
                has_loop = obj.fallback in seen
                seen.add(key)
                if has_loop:
                    errors.append("%s introduces a loop in the fallbacks of the %s %s" % (key, seed, objlabel))
                    break
                if obj.fallback not in objmap:
                    errors.append("%s falls back to the nonexistent %s %s" % (key, obj.fallback, objlabel))
                    break
                obj = objmap[obj.fallback]
            checked.update(seen)

    def validate(self):
        errors = []

        checked = set()
        for distribution in self.distributions.values():
            CXFixes._fallback_check('distribution', self.distributions, distribution, 'distid', checked, errors)

        checked = set()
        for cxproduct in self.cxproducts.values():
            CXFixes._fallback_check('cxproduct', self.cxproducts, cxproduct, 'cxid', checked, errors)

        self.maskedby = {}
        checked = set()
        for fix in self.fixes.values():
            CXFixes._fallback_check('fix', self.fixes, fix, 'errid', checked, errors)

            for errid in fix.maskedby:
                if errid not in self.fixes:
                    errors.append("The %s fix is masked by a nonexistent error: %s" % (fix.errid, errid))
                elif errid not in self.maskedby:
                    self.maskedby[errid] = set((fix.errid,))
                else:
                    self.maskedby[errid].add(fix.errid)

            for cxid in fix.cxlevels.keys():
                if cxid not in self.cxproducts:
                    errors.append("The %s error has properties for the nonexistent %s product" % (fix.errid, cxid))

            for (distid, _bitness) in fix.distfixes.keys():
                if distid not in self.distributions:
                    errors.append("The %s error has a fix for a nonexistent distribution: %s" % (fix.errid, distid))

        return errors

    def _fill_in(self, prepared, dst, srcid):
        if srcid not in self.fixes:
            return 0
        src = self.fixes[srcid]
        if srcid not in prepared:
            self._prepare_fix(prepared, src)
        copied = 0
        if dst.title is None and src.title is not None:
            dst.title = src.title
            copied += 1
        if dst.description is None and src.description is not None:
            dst.description = src.description
            copied += 1
        return copied

    def _prepare_fix(self, prepared, fix):
        prepared.add(fix.errid)

        bitness = 64 if fix.errid.endswith(".amd64") else 32
        missing = 0
        if fix.title is None:
            lib = fix.lib
            if fix.errid.startswith("missinglib"):
                if lib is None:
                    lib = fix.errid[7:] if bitness == 32 else fix.errid[7:-6]
                fix.title = _("Missing %(bitness)s-bit %(library)s library") % {'bitness': bitness, 'library': lib}
            elif fix.errid.startswith("brokenlib"):
                if lib is None:
                    lib = fix.errid[6:] if bitness == 32 else fix.errid[6:-6]
                fix.title = _("Broken %(bitness)s-bit %(library)s library") % {'bitness': bitness, 'library': lib}
            elif fix.errid.startswith("missinggstreamer1"):
                gst = fix.errid[17:] if bitness == 32 else fix.errid[17:-6]
                fix.title = _("Missing %(bitness)s-bit gst-plugins-%(plugin)s GStreamer plugins") % {'bitness': bitness, 'plugin': gst}
            else:
                missing += 1
        if fix.description is None:
            missing += 1

        if missing and bitness == 64:
            missing -= self._fill_in(prepared, fix, fix.errid[:-6])

        if missing and fix.fallback:
            self._fill_in(prepared, fix, fix.fallback)

        # Generate a matching 'BrokenLib' entry for the 'MissingLib' one
        if fix.errid.startswith("missinglib"):
            brokenid = 'broken' + fix.errid[7:]
            if brokenid not in self.fixes:
                lib = fix.lib
                if lib is None:
                    lib = fix.errid[7:] if bitness == 32 else fix.errid[7:-6]
                brokentitle = _("Broken %(bitness)s-bit %(library)s library") % {'bitness': bitness, 'library': lib}
                self.fixes[brokenid] = CXFix(brokenid, None, fix.lib, brokentitle, fix.description)

    def prepare(self):
        prepared = set()
        # self.fixes may get modified so iterate a copy
        errids = list(self.fixes.keys())
        for errid in errids:
            if errid not in prepared:
                self._prepare_fix(prepared, self.fixes[errid])

    def get_fix_level(self, fix, cxid):
        if fix is None:
            return None
        if cxid in self.cxproducts:
            while cxid is not None and cxid not in fix.cxlevels:
                cxid = self.cxproducts[cxid].fallback
        return None if cxid is None else fix.cxlevels[cxid]
