% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eval-rename.R, R/eval-select.R
\name{eval_rename}
\alias{eval_rename}
\alias{eval_select}
\title{Evaluate an expression with tidyselect semantics}
\usage{
eval_rename(
  expr,
  data,
  env = caller_env(),
  ...,
  strict = TRUE,
  name_spec = NULL
)

eval_select(
  expr,
  data,
  env = caller_env(),
  ...,
  include = NULL,
  exclude = NULL,
  strict = TRUE,
  name_spec = NULL,
  allow_rename = TRUE
)
}
\arguments{
\item{expr}{Defused R code describing a selection according to the
tidyselect syntax.}

\item{data}{A named list, data frame, or atomic vector.
Technically, \code{data} can be any vector with \code{names()} and \code{"[["}
implementations.}

\item{env}{The environment in which to evaluate \code{expr}. Discarded
if \code{expr} is a \link[rlang:enquo]{quosure}.}

\item{...}{These dots are for future extensions and must be empty.}

\item{strict}{If \code{TRUE}, out-of-bounds errors are thrown if \code{expr}
attempts to select or rename a variable that doesn't exist. If
\code{FALSE}, failed selections or renamings are ignored.}

\item{name_spec}{A name specification describing how to combine or
propagate names. This is used only in case nested \code{c()}
expressions like \code{c(foo = c(bar = starts_with("foo")))}. See the
\code{name_spec} argument of \code{\link[vctrs:vec_c]{vctrs::vec_c()}} for a description of
valid name specs.}

\item{include, exclude}{Character vector of column names to always
include or exclude from the selection.}

\item{allow_rename}{If \code{TRUE} (the default), the renaming syntax
\code{c(foo = bar)} is allowed. If \code{FALSE}, it causes an error. This
is useful to implement purely selective behaviour.}
}
\value{
A named vector of numeric locations, one for each of the
selected elements.

The names are normally the same as in the input data, except when
the user supplied named selections with \code{c()}. In the latter
case, the names reflect the new names chosen by the user.

A given element may be selected multiple times under different
names, in which case the vector might contain duplicate
locations.
}
\description{
\code{eval_select()} and \code{eval_rename()} evaluate defused R code
(i.e. quoted expressions) according to the special rules of the
\href{https://tidyselect.r-lib.org/articles/syntax.html}{tidyselect syntax}. They
power functions like \code{dplyr::select()}, \code{dplyr::rename()}, or
\code{tidyr::pivot_longer()}.

See the \href{https://tidyselect.r-lib.org/articles/tidyselect.html}{Get started}
vignette to learn how to use \code{eval_select()} and \code{eval_rename()} in
your packages.
}
\details{
The select and rename variants take the same types of inputs and
have the same type of return value. However \code{eval_rename()} has a
few extra constraints. It requires named inputs, and will fail if a
data frame column is renamed to another existing column name. See
the \href{https://tidyselect.r-lib.org/articles/syntax.html}{selecting versus renaming}
section in the syntax vignette for a description of the
differences.
}
\examples{
library(rlang)

# Interpret defused code as selection:
x <- expr(mpg:cyl)
eval_select(x, mtcars)

# Interpret defused code as a renaming selection. All inputs must
# be named within `c()`:
try(eval_rename(expr(mpg), mtcars))
eval_rename(expr(c(foo = mpg)), mtcars)


# Within a function, use `enquo()` to defuse one argument:
my_function <- function(x, expr) {
  eval_select(enquo(expr), x)
}

# If your function takes dots, evaluate a defused call to `c(...)`
# with `expr(c(...))`:
my_function <- function(.x, ...) {
  eval_select(expr(c(...)), .x)
}

# If your function takes dots and a named argument, use `{{ }}`
# inside the defused expression to tunnel it inside the tidyselect DSL:
my_function <- function(.x, .expr, ...) {
  eval_select(expr(c({{ .expr }}, ...)), .x)
}

# Note that the trick above works because `expr({{ arg }})` is the
# same as `enquo(arg)`.


# The evaluators return a named vector of locations. Here are
# examples of using these location vectors to implement `select()`
# and `rename()`:
select <- function(.x, ...) {
  pos <- eval_select(expr(c(...)), .x)
  set_names(.x[pos], names(pos))
}
rename <- function(.x, ...) {
  pos <- eval_rename(expr(c(...)), .x)
  names(.x)[pos] <- names(pos)
  .x
}

select(mtcars, mpg:cyl)
rename(mtcars, foo = mpg)
}
\seealso{
\url{https://tidyselect.r-lib.org/articles/syntax.html} or
\code{vignette("syntax", package = "tidyselect")} for a technical
description of the rules of evaluation.
}
