%feature("docstring") OT::GaussianNonLinearCalibration
"Gaussian non linear calibration algorithm (3DVAR).

Parameters
----------
model : :class:`~openturns.Function`
    The parametric function to be calibrated.
inputObservations : 2-d sequence of float
    The sample of input observations.
    Can have dimension 0 to specify no observations.
outputObservations : 2-d sequence of float
    The sample of output observations.
parameterMean : sequence of float
    The mean of the Gaussian prior distribution of the parameter.
parameterCovariance : 2-d sequence of float
    The covariance matrix of the Gaussian prior distribution of the parameter.
errorCovariance : 2-d sequence of float
    The covariance matrix of the Gaussian distribution of the observations error.

Notes
-----
This class implements the Maximum A Posteriori (MAP) estimator for a nonlinear model
under the Bayesian hypothesis that the prior and the observation errors have Gaussian distributions. 
This algorithm is also known as *3DVAR*. 

The given observation error covariance can be either *local*,
i.e. the same matrix applies for each observation and is of dimension the output
dimension of the model, or *global*, i.e. the matrix describes the full set of
observation errors, in which case its dimension is the product of the output
dimension of the model and the number of observations.

The prior distribution of the parameter is :class:`~openturns.Normal`.  

The posterior distribution of the parameter is also :class:`~openturns.Normal`.

Depending on the value of the `GaussianNonLinearCalibration-BootstrapSize` key 
in the :class:`~openturns.ResourceMap`, the parameters are estimated differently.

- If `GaussianNonLinearCalibration-BootstrapSize`
  is set to a nonzero positive integer, then a bootstrap resampling of the observations
  is performed and the posterior distribution is based on a :class:`~openturns.KernelSmoothing`
  of the sample of boostrap optimum parameters.
- If `GaussianNonLinearCalibration-BootstrapSize`
  is set to zero, then the posterior distribution is
  evaluated based on a linear approximation of the model at the optimum. 
  This corresponds to using :class:`~openturns.GaussianLinearCalibration` at the optimum,
  and is named *Laplace approximation* in the Bayesian context.

The default value of `GaussianNonLinearCalibration-BootstrapSize` is nonzero, 
meaning that a bootstrap is used by default.
This can be costly in some practical use cases, because it requires
several runs of the optimization algorithm.
In this case, please set the bootstrap size to zero.

The resulting error distribution is :class:`~openturns.Normal`
with a mean estimated from the sample mean of the residuals and with a
covariance matrix equal to the `errorCovariance` input argument.

Please read read :ref:`gaussian_calibration` for more details
on Gaussian calibration.

See also
--------
GaussianLinearCalibration, LinearLeastSquaresCalibration, NonLinearLeastSquaresCalibration

Examples
--------
Calibrate a nonlinear model using 3DVAR:

>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> m = 10
>>> x = [[0.5 + i] for i in range(m)]
>>> inVars = ['a', 'b', 'c', 'x']
>>> formulas = ['a + b * exp(c * x)']
>>> model = ot.SymbolicFunction(inVars, formulas)
>>> p_ref = [2.8, 1.2, 0.5]
>>> params = [0, 1, 2]
>>> modelX = ot.ParametricFunction(model, params, p_ref)
>>> y = modelX(x)
>>> y += ot.Normal(0.0, 0.05).getSample(m)
>>> parameterMean = [1.0]*3
>>> priorCovariance = ot.CovarianceMatrix(3)
>>> errorCovariance = ot.CovarianceMatrix(1, [0.1])
>>> algo = ot.GaussianNonLinearCalibration(modelX, x, y, parameterMean, priorCovariance, errorCovariance)
>>> algo.run()  # doctest: +SKIP
>>> pmap = algo.getResult().getParameterMAP()  # doctest: +SKIP"

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianNonLinearCalibration::getModelObservations
"Accessor to the model evaluation at the parameterMean.

Returns
-------
modelObservation : :class:`~openturns.Sample`
    Evaluation of the model at the parameterMean point."

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianNonLinearCalibration::getGradientObservations
"Accessor to the model gradient at the `parameterMean`.

Returns
-------
gradientObservation : :class:`~openturns.Matrix`
    Gradient of the model at the `parameterMean` point."

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianNonLinearCalibration::getParameterMean
"Accessor to the parameter `parameterMean`.

Returns
-------
parameterMean : :class:`~openturns.Point`
    Parameter `parameterMean`."

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianNonLinearCalibration::getParameterCovariance
"Accessor to the parameter prior covariance.

Returns
-------
prior : :class:`~openturns.CovarianceMatrix`
    Parameter prior covariance."

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianNonLinearCalibration::getErrorCovariance
"Accessor to the observation error covariance.

Returns
-------
error : :class:`~openturns.CovarianceMatrix`
    Observation error covariance."

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianNonLinearCalibration::getGlobalErrorCovariance
"Accessor to the flag for a global observation error covariance.

Returns
-------
flag : bool
    Flag telling if the given observation error covariance is global or not."

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianNonLinearCalibration::getOptimizationAlgorithm
"Accessor to the optimization algorithm used for the computation.

Returns
-------
algo : :class:`~openturns.OptimizationAlgorithm`
    Optimization algorithm used for the computation."

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianNonLinearCalibration::setOptimizationAlgorithm
"Accessor to the optimization algorithm used for the computation.

Parameters
----------
algo : :class:`~openturns.OptimizationAlgorithm`
    Optimization algorithm to use for the computation."

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianNonLinearCalibration::getBootstrapSize
"Accessor to the bootstrap size used to sample the posterior distribution.

Returns
-------
size : int
    Bootstrap size used to sample the posterior distribution. A value of 0
    means that no bootstrap has been done but a linear approximation has
    been used to get the posterior distribution, using the :class:`~openturns.GaussianLinearCalibration`
    algorithm at the maximum a posteriori estimate."

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianNonLinearCalibration::setBootstrapSize
"Accessor to the bootstrap size used to sample the posterior distribution.

Parameters
----------
size : int
    Bootstrap size used to sample the posterior distribution. A value of 0
    means that no bootstrap has to be done but a linear approximation has
    been used to get the posterior distribution, using the :class:`~openturns.GaussianLinearCalibration`
    algorithm at the maximum a posteriori estimate."

