/*
 * // Copyright (c) Radzivon Bartoshyk 8/2025. All rights reserved.
 * //
 * // Redistribution and use in source and binary forms, with or without modification,
 * // are permitted provided that the following conditions are met:
 * //
 * // 1.  Redistributions of source code must retain the above copyright notice, this
 * // list of conditions and the following disclaimer.
 * //
 * // 2.  Redistributions in binary form must reproduce the above copyright notice,
 * // this list of conditions and the following disclaimer in the documentation
 * // and/or other materials provided with the distribution.
 * //
 * // 3.  Neither the name of the copyright holder nor the names of its
 * // contributors may be used to endorse or promote products derived from
 * // this software without specific prior written permission.
 * //
 * // THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * // AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * // IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * // DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * // FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * // DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * // SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * // CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * // OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * // OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
use crate::double_double::DoubleDouble;
use crate::logs::log10td_coeffs::LOG10_NEG_TD;
use crate::pow_tables::POW_INVERSE;
use crate::triple_double::TripleDouble;

#[inline(always)]
fn log10_poly(z: f64) -> TripleDouble {
    /*
        Poly generated by Sollya:

        d = [-0.0040283203125,0.0040283203125];
        f = log10(1+x)/x;
        pf = fpminimax(f, 10, [|157...|], d, absolute, floating);

        See ./notes/td_log10.sollya
    */
    const P: [(u64, u64, u64); 11] = [
        (0xb8f29c4663277dc0, 0x3c695355baaafad3, 0x3fdbcb7b1526e50e),
        (0x38ee76408ee1b680, 0xbc595355bab3a517, 0xbfcbcb7b1526e50e),
        (0x38f58fe4d121c480, 0xbc59c8718361a853, 0x3fc287a7636f435f),
        (0xb8e80f425616e240, 0xbc49520fd0630f7e, 0xbfbbcb7b1526e50e),
        (0x38eb058acb9ab2c0, 0x3c443f5b242c637b, 0x3fb63c62775250d8),
        (0x38c32c088e8da180, 0xbc4a6666386cc60e, 0xbfb287a7636f436c),
        (0x38bb2e95bc6a3a80, 0x3c45ac4260107e46, 0x3fafc3fa615105f6),
        (0xb8e14f1e5b926690, 0x3c421ee12f4a23de, 0xbfabcb7b14ed42da),
        (0x38d4b4d370b610e0, 0x3c47e2c6f24d3d30, 0x3fa8b4df2ef1a288),
        (0x38bd632a8b3a6500, 0xbc1a922cdefb3bb1, 0xbfa63c98a8c44069),
        (0xb8c077b6551291c0, 0x3c4337142dd97816, 0x3fa4372045637cca),
    ];
    let mut t = TripleDouble::f64_mul_add(
        z,
        TripleDouble::from_bit_pair(P[9]),
        TripleDouble::from_bit_pair(P[8]),
    );
    t = TripleDouble::f64_mul_add(z, t, TripleDouble::from_bit_pair(P[7]));
    t = TripleDouble::f64_mul_add(z, t, TripleDouble::from_bit_pair(P[6]));
    t = TripleDouble::f64_mul_add(z, t, TripleDouble::from_bit_pair(P[5]));
    t = TripleDouble::f64_mul_add(z, t, TripleDouble::from_bit_pair(P[4]));
    t = TripleDouble::f64_mul_add(z, t, TripleDouble::from_bit_pair(P[3]));
    t = TripleDouble::f64_mul_add(z, t, TripleDouble::from_bit_pair(P[2]));
    t = TripleDouble::f64_mul_add(z, t, TripleDouble::from_bit_pair(P[1]));
    t = TripleDouble::f64_mul_add(z, t, TripleDouble::from_bit_pair(P[0]));
    TripleDouble::quick_mult_f64(t, z)
}

#[inline]
pub(crate) fn log10_td(x: f64) -> TripleDouble {
    let x_u = x.to_bits();
    let mut m = x_u & 0xfffffffffffff;
    let mut e: i64 = ((x_u >> 52) & 0x7ff) as i64;

    let t;
    if e != 0 {
        t = m | (0x3ffu64 << 52);
        m = m.wrapping_add(1u64 << 52);
        e -= 0x3ff;
    } else {
        /* x is a subnormal double  */
        let k = m.leading_zeros() - 11;

        e = -0x3fei64 - k as i64;
        m = m.wrapping_shl(k);
        t = m | (0x3ffu64 << 52);
    }

    /* now |x| = 2^_e*_t = 2^(_e-52)*m with 1 <= _t < 2,
    and 2^52 <= _m < 2^53 */

    //   log10(x) = log10(t) + E · log10(e)
    let mut t = f64::from_bits(t);

    // If m > sqrt(2) we divide it by 2 so ensure 1/sqrt(2) < t < sqrt(2)
    let c: usize = (m >= 0x16a09e667f3bcd) as usize;
    static CY: [f64; 2] = [1.0, 0.5];
    static CM: [u64; 2] = [44, 45];

    e = e.wrapping_add(c as i64);
    let be = e;
    let i = m >> CM[c];
    t *= CY[c];

    let r = f64::from_bits(POW_INVERSE[(i - 181) as usize]);
    let log10_r = TripleDouble::from_bit_pair(LOG10_NEG_TD[(i - 181) as usize]);

    let z = f64::mul_add(r, t, -1.0);

    const LOG10_2_DD: DoubleDouble =
        DoubleDouble::from_bit_pair((0xbc49dc1da994fd21, 0x3fd34413509f79ff));

    let v = TripleDouble::f64_mul_dd_add(be as f64, LOG10_2_DD, log10_r);

    let p = log10_poly(z);
    TripleDouble::add_f64(v.hi, TripleDouble::new(v.lo + p.lo, v.mid + p.mid, p.hi))
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_log10_dd() {
        assert_eq!(log10_td(10.).to_f64(), 1.);
    }
}
